import numpy as np
import scipy as sp

"""" additional functions for scientific computation """

def norm_ppf(q, loc=0, scale=1):
    """
    Computes the percent point function (PPF) for the normal distribution.
    Parameters:
    q (float or array-like): Quantile(s), must be between 0 and 1.
    loc (float, optional): Mean ("location") of the distribution. Default is 0.
    scale (float, optional): Standard deviation ("scale") of the distribution. Default is 1.
    Returns:
    float or array-like: The value(s) at the given quantile(s) of the normal distribution.
    """
    # Calculate the inverse of the CDF (percent point function) for the normal distribution
    # using the inverse error function (erfinv) from scipy.special.
    return loc + scale * np.sqrt(2) * sp.special.erfinv(2 * q - 1)
    
def clopper_pearson_confint(count, nobs, alpha=0.05):
    """
    Compute the Clopper-Pearson confidence interval for a binomial proportion.
    Parameters:
    - count: number of successes
    - nobs: number of trials or observations
    - alpha: significance level (default is 0.05 for a 95% confidence interval)
    Returns:
    - confint_low: lower bound of the confidence interval
    - confint_upp: upper bound of the confidence interval
    """
    # Calculate the lower bound of the confidence interval
    # using the the inverse of the beta cumulative distribution function from scipy.special.
    confint_low = sp.special.betaincinv(count, nobs - count + 1,alpha / 2)
    # Calculate the upper bound of the confidence interval
    # using the the inverse of the beta cumulative distribution function from scipy.special.
    confint_upp = sp.special.betaincinv(count + 1, nobs - count,1 - alpha / 2) 
    return confint_low, confint_upp