# this file is based on code publicly available at
#   https://github.com/bearpaw/pytorch-classification
# written by Wei Yang.

import argparse

import torch

import os
from torch.nn import CrossEntropyLoss
import torch.nn.utils as utils
from torch.utils.data import DataLoader
import torch.nn.utils.prune as prune
from datasets import get_dataset, DATASETS
from architectures import CIFAR_ARCHITECTURES, IMAGENET_ARCHITECTURES, get_architecture
from torch.optim import SGD, Optimizer
from torch.optim.lr_scheduler import StepLR
import time
import datetime
from train_utils import AverageMeter, accuracy, init_logfile, log
#import setGPU

from attacks import ATTACKS, AdversarialAttackApplier
#from device import DEVICE

import logging
logging.getLogger("torch").setLevel(logging.ERROR)


CLASSIFIER_CHOICES = {
    "cifar10": CIFAR_ARCHITECTURES,
    "imagenet": IMAGENET_ARCHITECTURES,
}

DEVICE = torch.device('cpu')

parser = argparse.ArgumentParser(description='Certify many examples')
parser.add_argument("dataset", choices=DATASETS, help="which dataset")

args, remaining_args = parser.parse_known_args()

parser.add_argument('arch', type=str, choices=CLASSIFIER_CHOICES[args.dataset], help="which classifier")
parser.add_argument('attack', type=str, choices=ATTACKS)
parser.add_argument('--proportion', default=0.5, type=float, metavar='N',
                    help='proportion of adversarial examples for training and testing (default: 0.5)')
parser.add_argument('--workers', default=0, type=int, metavar='N',
                    help='number of data loading workers (default: 0)')
parser.add_argument('--epochs', default=100, type=int, metavar='N',
                    help='number of total epochs to run')
parser.add_argument('--batch', default=128, type=int, metavar='N',
                    help='batchsize (default: 256)')
parser.add_argument('--lr', '--learning-rate', default=0.001, type=float,
                    help='initial learning rate', dest='lr')
parser.add_argument('--lr_step_size', type=int, default=30,
                    help='How often to decrease learning by gamma.')
parser.add_argument('--gamma', type=float, default=0.1,
                    help='LR is multiplied by gamma on schedule.')
parser.add_argument('--momentum', default=0.9, type=float, metavar='M',
                    help='momentum')
parser.add_argument('--weight-decay', '--wd', default=1e-4, type=float,
                    metavar='W', help='weight decay (default: 1e-4)')
parser.add_argument('--gpu', default=None, type=str,
                    help='id(s) for CUDA_VISIBLE_DEVICES')
parser.add_argument('--print-freq', default=1, type=int,
                    metavar='N', help='print frequency (default: 10)')
args = parser.parse_args()


def main():

    # make directory if it does not exist
    outdir = 'models/base_classifiers'
    if not os.path.exists(outdir): os.mkdir(outdir)

    train_dataset = get_dataset(args.dataset, 'train')
    test_dataset = get_dataset(args.dataset, 'test')

    pin_memory = (args.dataset == "imagenet")
    train_loader = DataLoader(train_dataset, shuffle=True, batch_size=args.batch, num_workers=args.workers, pin_memory=pin_memory)
    test_loader = DataLoader(test_dataset, shuffle=False, batch_size=args.batch, num_workers=args.workers, pin_memory=pin_memory)

    # load the base classifier
    base_classifier_file =  'models/base_classifiers/' + args.arch + '_' + args.dataset + '_' + args.attack + '_checkpoint.pth.tar'
    
    if os.path.exists(base_classifier_file):
        print('model already there')
        checkpoint = torch.load(base_classifier_file,map_location=DEVICE)
        model = get_architecture(checkpoint["arch"], args.dataset)
        model.load_state_dict(checkpoint['state_dict'])
    else :
        print('model not there')
        model = get_architecture(args.arch, args.dataset)

    # define attack
    attack = AdversarialAttackApplier(model, args.dataset, args.attack)

    logfilename = os.path.join(outdir, args.arch + '_' + args.dataset + '_' + args.attack + '_log.txt')
    init_logfile(logfilename, "epoch\ttime\tlr\ttrain loss\ttrain acc\ttestloss\ttest acc")

    criterion = CrossEntropyLoss().to(DEVICE)
    optimizer = SGD(model.parameters(), lr=args.lr, momentum=args.momentum, weight_decay=args.weight_decay)
    scheduler = StepLR(optimizer, step_size=args.lr_step_size, gamma=args.gamma)

    for epoch in range(args.epochs):
        scheduler.step(epoch)
        before = time.time()
        train_loss, train_acc = train(train_loader, model, attack, criterion, optimizer, epoch)
        test_loss, test_acc = test(test_loader, model, attack, criterion)
        after = time.time()

        log(logfilename, "{}\t{:.3}\t{:.3}\t{:.3}\t{:.3}\t{:.3}\t{:.3}".format(
            epoch, str(datetime.timedelta(seconds=(after - before))),
            scheduler.get_last_lr()[0], train_loss, train_acc, test_loss, test_acc))

        torch.save({
            'epoch': epoch + 1,
            'arch': args.arch,
            'state_dict': model.state_dict(),
            'optimizer': optimizer.state_dict(),
        }, os.path.join(outdir, args.arch + '_' + args.dataset + '_' + args.attack + '_checkpoint.pth.tar'))


def train(loader: DataLoader, model: torch.nn.Module, attack, criterion, optimizer: Optimizer, epoch: int):
    
    batch_time = AverageMeter()
    data_time = AverageMeter()
    losses = AverageMeter()
    top1 = AverageMeter()
    top5 = AverageMeter()
    end = time.time()

    # switch to train mode
    model.train()
    model.to(DEVICE)
    accumulation_steps = 5
    max_norm = 1.0

    for i, (inputs, targets) in enumerate(loader):

        print(i)

        # measure data loading time
        data_time.update(time.time() - end)

        inputs = inputs.to(DEVICE)
        targets = targets.to(DEVICE)

        # compute output
        perturbed_inputs = attack.apply_adversarial_with_probability(inputs, targets)
        perturbed_inputs = perturbed_inputs.to(DEVICE) 

        outputs = model(perturbed_inputs)
        loss = criterion(outputs, targets)

        # measure accuracy and record loss
        acc1, acc5 = accuracy(outputs, targets, topk=(1, 5))
        losses.update(loss.item(), inputs.size(0))
        top1.update(acc1.item(), inputs.size(0))
        top5.update(acc5.item(), inputs.size(0))

        # compute gradient and do SGD step
        loss.backward()

        optimizer.step() 
        optimizer.zero_grad()

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()

        if i % args.print_freq == 0:
            print('Epoch: [{0}][{1}/{2}]\t'
                  'Time {batch_time.val:.3f} ({batch_time.avg:.3f})\t'
                  'Data {data_time.val:.3f} ({data_time.avg:.3f})\t'
                  'Loss {loss.val:.4f} ({loss.avg:.4f})\t'
                  'Acc@1 {top1.val:.3f} ({top1.avg:.3f})\t'
                  'Acc@5 {top5.val:.3f} ({top5.avg:.3f})'.format(
                epoch, i, len(loader), batch_time=batch_time,
                data_time=data_time, loss=losses, top1=top1, top5=top5))

    return (losses.avg, top1.avg)


def test(loader: DataLoader, model: torch.nn.Module, attack, criterion):
    batch_time = AverageMeter()
    data_time = AverageMeter()
    losses = AverageMeter()
    top1 = AverageMeter()
    top5 = AverageMeter()
    end = time.time()

    # switch to eval mode
    model.eval()
    model.to(DEVICE)

    with torch.no_grad():
        for i, (inputs, targets) in enumerate(loader):

            # measure data loading time
            data_time.update(time.time() - end)

            inputs = inputs.to(DEVICE)
            targets = targets.to(DEVICE)

            # compute output
            perturbed_inputs = attack.apply_adversarial_with_probability(inputs, targets)
            perturbed_inputs = perturbed_inputs.to(DEVICE)

            outputs = model(perturbed_inputs)
            # outputs = model(inputs)
            loss = criterion(outputs, targets)

            # measure accuracy and record loss
            acc1, acc5 = accuracy(outputs, targets, topk=(1, 5))
            losses.update(loss.item(), inputs.size(0))
            top1.update(acc1.item(), inputs.size(0))
            top5.update(acc5.item(), inputs.size(0))

            # measure elapsed time
            batch_time.update(time.time() - end)
            end = time.time()

            if i % args.print_freq == 0:
                print('Test: [{0}/{1}]\t'
                      'Time {batch_time.val:.3f} ({batch_time.avg:.3f})\t'
                      'Data {data_time.val:.3f} ({data_time.avg:.3f})\t'
                      'Loss {loss.val:.4f} ({loss.avg:.4f})\t'
                      'Acc@1 {top1.val:.3f} ({top1.avg:.3f})\t'
                      'Acc@5 {top5.val:.3f} ({top5.avg:.3f})'.format(
                    i, len(loader), batch_time=batch_time,
                    data_time=data_time, loss=losses, top1=top1, top5=top5))

        return (losses.avg, top1.avg)
    

if __name__ == "__main__":
    main()
