"""
Based on PureJaxRL Implementation of IPPO, with changes to give a centralised critic.
"""

import jax
import jax.numpy as jnp
import flax.linen as nn
from flax import struct
import numpy as np
import optax
from flax.linen.initializers import constant, orthogonal
from typing import Sequence, NamedTuple, Any, Tuple, Union, Dict
import chex

from flax.training.train_state import TrainState
import distrax
import hydra
from omegaconf import DictConfig, OmegaConf
from functools import partial
import jaxmarl
from jaxmarl.wrappers.baselines import LogWrapper, JaxMARLWrapper
from jaxmarl.environments.multi_agent_env import MultiAgentEnv, State


import wandb
import functools
import matplotlib.pyplot as plt

    
class MABraxWorldStateWrapper(JaxMARLWrapper):
    
    @partial(jax.jit, static_argnums=0)
    def reset(self,
              key):
        obs, env_state = self._env.reset(key)
        obs["world_state"] = self.world_state(obs, env_state)
        return obs, env_state
    
    @partial(jax.jit, static_argnums=0)
    def step(self,
             key,
             state,
             action):
        obs, env_state, reward, done, info = self._env.step(
            key, state, action
        )
        obs["world_state"] = self.world_state(obs, state)
        return obs, env_state, reward, done, info

    @partial(jax.jit, static_argnums=0)
    def world_state(self, obs, state):
        """ 
        For each agent: [agent obs, own hand]
        """
        return jnp.array([state.obs for agent in self._env.agents])
        # hands = state.player_hands.reshape((self._env.num_agents, -1))
        # return jnp.concatenate((all_obs, hands), axis=1)
        
    
    def world_state_size(self):
        return self._env.state_spaces.shape[0]


class ActorFF(nn.Module):
    action_dim: Sequence[int]
    config: Dict

    @nn.compact
    def __call__(self, x):
        if self.config["ACTIVATION"] == "relu":
            activation = nn.relu
        else:
            activation = nn.tanh
        actor_mean = nn.Dense(
            self.config["FC_DIM_SIZE"], kernel_init=orthogonal(0.01), bias_init=constant(0.0)
        )(nn.LayerNorm()(x))
        actor_mean = activation(actor_mean)
        actor_mean = nn.Dense(
            self.config["FC_DIM_SIZE"], kernel_init=orthogonal(0.01), bias_init=constant(0.0)
        )(nn.LayerNorm()(actor_mean))
        actor_mean = activation(actor_mean)
        actor_mean = nn.Dense(
            self.config["FC_DIM_SIZE"], kernel_init=orthogonal(0.01), bias_init=constant(0.0)
        )(nn.LayerNorm()(actor_mean))
        actor_mean = activation(actor_mean)
        actor_mean = nn.Dense(
            self.action_dim, kernel_init=orthogonal(0.01), bias_init=constant(0.0)
        )(nn.LayerNorm()(actor_mean))
        #actor_logtstd = nn.Dense(
        #    self.action_dim, kernel_init=orthogonal(0.01), bias_init=constant(0.0)
        #)(nn.LayerNorm()(actor_mean))
        actor_logtstd = self.param('log_std', nn.initializers.ones, (self.action_dim,))
        #print(x.shape, actor_mean.shape, actor_logtstd.shape)
        pi = distrax.MultivariateNormalDiag(actor_mean, nn.sigmoid(actor_logtstd))

        return pi

class CriticFF(nn.Module):
    config: Dict
    
    @nn.compact
    def __call__(self, x):
        if self.config["ACTIVATION"] == "relu":
            activation = nn.relu
        else:
            activation = nn.tanh
            
        critic = nn.Dense(
            self.config["FC_DIM_SIZE"], kernel_init=orthogonal(), bias_init=constant(0.0)
        )(nn.LayerNorm()(x))
        critic = activation(critic)
        critic = nn.Dense(
            self.config["FC_DIM_SIZE"], kernel_init=orthogonal(), bias_init=constant(0.0)
        )(nn.LayerNorm()(critic))
        critic = activation(critic)
        critic = nn.Dense(
            self.config["FC_DIM_SIZE"], kernel_init=orthogonal(), bias_init=constant(0.0)
        )(nn.LayerNorm()(critic))
        critic = activation(critic)
        critic = nn.Dense(
            1, kernel_init=orthogonal(), bias_init=constant(0.0)
        )(nn.LayerNorm()(critic))
            
        return jnp.squeeze(critic, axis=-1)
        
class ValueNorm(nn.Module):

    def __init__(
        self,
        input_shape,
        norm_axes=0,
        beta=0.99999,
        epsilon=1e-5,
    ):
        super(ValueNorm, self).__init__()

        self.input_shape = input_shape
        self.norm_axes = norm_axes
        self.epsilon = epsilon
        self.beta = beta

        self.running_mean = jnp.zeros(input_shape)
        self.running_mean_sq = jnp.zeros(input_shape)
        self.debiasing_term = jnp.array(0.0)
        
    def running_mean_var(self):
        debiased_mean = self.running_mean / jnp.maximum(self.debiasing_term, self.epsilon)
        debiased_mean_sq = self.running_mean_sq / jnp.maximum(self.debiasing_term, self.epsilon)
        debiased_var = jnp.maximum(debiased_mean_sq - debiased_mean ** 2, 1e-2)
        return debiased_mean, debiased_var

    def update(self, input_vector):
        batch_mean = jnp.mean(input_vector, axis=tuple(range(self.norm_axes)))
        batch_sq_mean = jnp.mean(input_vector ** 2, axis=tuple(range(self.norm_axes)))
        weight = self.beta

        # Update the running statistics
        self.running_mean = self.running_mean * weight + batch_mean * (1.0 - weight)
        self.running_mean_sq = self.running_mean_sq * weight + batch_sq_mean * (1.0 - weight)
        self.debiasing_term = self.debiasing_term * weight + (1.0 - weight)

    def normalize(self, input_vector):
        mean, var = self.running_mean_var()
        normalized = (input_vector - mean[tuple([None] * self.norm_axes)]) / jnp.sqrt(var)[
            tuple([None] * self.norm_axes)
        ]
        return normalized

    def denormalize(self, input_vector):
        mean, var = self.running_mean_var()
        denormalized = input_vector * jnp.sqrt(var)[tuple([None] * self.norm_axes)] + mean[
            tuple([None] * self.norm_axes)
        ]
        return denormalized
    
class Transition(NamedTuple):
    done: jnp.ndarray
    action: jnp.ndarray
    value: jnp.ndarray
    reward: jnp.ndarray
    log_prob: jnp.ndarray
    obs: jnp.ndarray
    world_state: jnp.ndarray
    info: jnp.ndarray


def batchify(x: dict, agent_list, num_actors):
    max_dim = max([x[a].shape[-1] for a in agent_list])
    print('max_dim', max_dim)
    def pad(z):
        return jnp.concatenate([z, jnp.zeros(z.shape[:-1] + (max_dim - z.shape[-1],))], -1)

    x = jnp.stack([x[a] if x[a].shape[-1] == max_dim else pad(x[a]) for a in agent_list])
    return x.reshape((num_actors, -1))


def unbatchify(x: jnp.ndarray, agent_list, num_envs, num_actors):
    x = x.reshape((num_actors, num_envs, -1))
    return {a: x[i] for i, a in enumerate(agent_list)}


def make_train(config):
    env = jaxmarl.make(config["ENV_NAME"], **config["ENV_KWARGS"])
    config["NUM_ACTORS"] = env.num_agents * config["NUM_ENVS"]
    config["NUM_UPDATES"] = (
        config["TOTAL_TIMESTEPS"] // config["NUM_STEPS"] // config["NUM_ENVS"]
    )
    config["MINIBATCH_SIZE"] = (
        config["NUM_ACTORS"] * config["NUM_STEPS"] // config["NUM_MINIBATCHES"]
    )
    config["CLIP_EPS"] = config["CLIP_EPS"] / env.num_agents if config["SCALE_CLIP_EPS"] else config["CLIP_EPS"]

    env = MABraxWorldStateWrapper(env)
    env = LogWrapper(env, replace_info=True)

    def linear_schedule(count):
        frac = 1.0 - (count // (config["NUM_MINIBATCHES"] * config["UPDATE_EPOCHS"])) / config["NUM_UPDATES"]
        return config["LR"] * frac

    def train(rng):
        # INIT NETWORK
        actor_network = ActorFF(
            env.action_space(env.agents[0]).shape[0], config=config
        )
        max_dim = jnp.argmax(jnp.array([env.observation_space(a).shape[-1] for a in env.agents]))
        critic_network = CriticFF(config)
        rng, _rng_actor, _rng_critic = jax.random.split(rng, 3)
        
        ac_init_x = jnp.zeros((config["NUM_ENVS"], env.observation_space(env.agents[0]).shape[0]))
        actor_network_params = actor_network.init(_rng_actor, ac_init_x)
        
        cr_init_x = jnp.zeros(env.state_spaces[env.agents[0]].shape)
        
        critic_network_params = critic_network.init(_rng_critic, cr_init_x)
        
        if config["VALUE_NORM"]:
            value_normalizer = ValueNorm(1)
        
        if config["ANNEAL_LR"]:
            actor_tx = optax.chain(
                optax.clip_by_global_norm(config["MAX_GRAD_NORM"]),
                optax.adam(learning_rate=linear_schedule, eps=1e-5),
            )
            critic_tx = optax.chain(
                optax.clip_by_global_norm(config["MAX_GRAD_NORM"]),
                optax.adam(learning_rate=linear_schedule, eps=1e-5),
            )
        else:
            actor_tx = optax.chain(
                optax.clip_by_global_norm(config["MAX_GRAD_NORM"]),
                optax.adam(config["LR"], eps=1e-5),
            )
            critic_tx = optax.chain(
                optax.clip_by_global_norm(config["MAX_GRAD_NORM"]),
                optax.adam(config["LR"], eps=1e-5),
            )
        actor_train_state = TrainState.create(
            apply_fn=actor_network.apply,
            params=actor_network_params,
            tx=actor_tx,
        )
        critic_train_state = TrainState.create(
            apply_fn=actor_network.apply,
            params=critic_network_params,
            tx=critic_tx,
        )

        # INIT ENV
        rng, _rng = jax.random.split(rng)
        reset_rng = jax.random.split(_rng, config["NUM_ENVS"])
        obsv, env_state = jax.vmap(env.reset)(reset_rng)

        # TRAIN LOOP
        def _update_step(update_runner_state, unused):
            # COLLECT TRAJECTORIES
            runner_state, update_steps = update_runner_state
            
            def _env_step(runner_state, unused):
                train_states, env_state, last_obs, update_count, rng = runner_state

                obs_batch = batchify(last_obs, env.agents, config["NUM_ACTORS"])
                # SELECT ACTION
                rng, _rng = jax.random.split(rng)
                
                pi = actor_network.apply(train_states[0].params, obs_batch)
                action = pi.sample(seed=_rng)
                log_prob = pi.log_prob(action)
                env_act = unbatchify(action, env.agents, config["NUM_ENVS"], env.num_agents)
                
                # VALUE
                world_state = last_obs["world_state"].swapaxes(0,1)
                world_state = world_state.reshape((config["NUM_ACTORS"],-1))
                value = critic_network.apply(train_states[1].params, world_state)

                # STEP ENV
                rng, _rng = jax.random.split(rng)
                rng_step = jax.random.split(_rng, config["NUM_ENVS"])
                obsv, env_state, reward, done, info = jax.vmap(env.step)(
                    rng_step, env_state, env_act,
                )
                
                info = jax.tree.map(lambda x: x.reshape((config["NUM_ACTORS"])), info)
                transition = Transition(
                    batchify(done, env.agents, config["NUM_ACTORS"]).squeeze(),
                    action,
                    value,
                    batchify(reward, env.agents, config["NUM_ACTORS"]).squeeze(),
                    log_prob,
                    obs_batch,
                    world_state,
                    info,
                )
                runner_state = (train_states, env_state, obsv, update_count, rng)
                return runner_state, transition

            runner_state, traj_batch = jax.lax.scan(
                _env_step, runner_state, None, config["NUM_STEPS"]
            )
            print('traj_batch', traj_batch)
            # CALCULATE ADVANTAGE
            train_states, env_state, last_obs, update_count, rng = runner_state
      
            last_world_state = last_obs["world_state"].swapaxes(0,1)
            last_world_state = last_world_state.reshape((config["NUM_ACTORS"],-1))
            last_val = critic_network.apply(train_states[1].params, last_world_state)
            #last_val = last_val.squeeze()

            def _calculate_gae(traj_batch, last_val):
                def _get_advantages(gae_and_next_value, transition):
                    gae, next_value = gae_and_next_value
                    done, value, reward = (
                        transition.done,
                        transition.value,
                        transition.reward,
                    )
                    if config["VALUE_NORM"]:
                        value, next_value = value_normalizer.denormalize(value), value_normalizer.denormalize(next_value)
                    delta = reward + config["GAMMA"] * next_value * (1 - done) - value
                    gae = (
                        delta
                        + config["GAMMA"] * config["GAE_LAMBDA"] * (1 - done) * gae
                    )
                    return (gae, value), gae

                _, advantages = jax.lax.scan(
                    _get_advantages,
                    (jnp.zeros_like(last_val), last_val),
                    traj_batch,
                    reverse=True,
                    unroll=8,
                )
                if config["VALUE_NORM"]:
                    return advantages, advantages + value_normalizer.denormalize(traj_batch.value)
                return advantages, advantages + traj_batch.value

            advantages, targets = _calculate_gae(traj_batch, last_val)

            # UPDATE NETWORK
            def _update_epoch(update_state, unused):
                def _update_minbatch(train_states, batch_info):
                    actor_train_state, critic_train_state = train_states
                    traj_batch, advantages, targets = batch_info

                    def _actor_loss_fn(actor_params, traj_batch, gae):
                        # RERUN NETWORK
                        pi = actor_network.apply(
                            actor_params,
                            traj_batch.obs,
                        )
                        log_prob = pi.log_prob(traj_batch.action)

                        # CALCULATE ACTOR LOSS
                        ratio = jnp.exp(log_prob - traj_batch.log_prob)
                        gae = (gae - gae.mean()) / (gae.std() + 1e-8)
                        loss_actor1 = ratio * gae
                        loss_actor2 = (
                            jnp.clip(
                                ratio,
                                1.0 - config["CLIP_EPS"],
                                1.0 + config["CLIP_EPS"],
                            )
                            * gae
                        )
                        loss_actor = -jnp.minimum(loss_actor1, loss_actor2)
                        loss_actor = loss_actor.mean()
                        entropy = pi.entropy().mean()
                        
                        # debug
                        approx_kl = ((ratio - 1) - ratio).mean()
                        clip_frac = jnp.mean(jnp.abs(ratio - 1) > config["CLIP_EPS"])
                        
                        actor_loss = (
                            loss_actor
                            - config["ENT_COEF"] * entropy
                        )
                        return actor_loss, (loss_actor, entropy, ratio, approx_kl, clip_frac)
                    
                    def _critic_loss_fn(critic_params, traj_batch, targets):
                        
                        if config["VALUE_NORM"]:
                            value_normalizer.update(targets)
                            targets = value_normalizer.normalize(targets)
                        # RERUN NETWORK
                        value = critic_network.apply(critic_params, traj_batch.world_state)
                        
                        # CALCULATE VALUE LOSS
                        value_pred_clipped = traj_batch.value + (
                            value - traj_batch.value
                        ).clip(-config["CLIP_EPS"], config["CLIP_EPS"])
                        value_losses = optax.huber_loss(value, targets, delta=10)
                        value_losses_clipped = optax.huber_loss(value_pred_clipped, targets, delta=10)
                        value_loss = (
                            0.5 * jnp.maximum(value_losses, value_losses_clipped).mean()
                        )
                        critic_loss = config["VF_COEF"] * value_loss
                        return critic_loss, (value_loss)

                    actor_grad_fn = jax.value_and_grad(_actor_loss_fn, has_aux=True)
                    actor_loss, actor_grads = actor_grad_fn(
                        actor_train_state.params, traj_batch, advantages
                    )
                    critic_grad_fn = jax.value_and_grad(_critic_loss_fn, has_aux=True)
                    critic_loss, critic_grads = critic_grad_fn(
                        critic_train_state.params, traj_batch, targets
                    )
                    
                    actor_train_state = actor_train_state.apply_gradients(grads=actor_grads)
                    critic_train_state = critic_train_state.apply_gradients(grads=critic_grads)
                    
                    total_loss = actor_loss[0] + critic_loss[0]
                    loss_info = {
                        "total_loss": total_loss,
                        "actor_loss": actor_loss[0],
                        "critic_loss": critic_loss[0],
                        "entropy": actor_loss[1][1],
                        "ratio": actor_loss[1][2],
                        "approx_kl": actor_loss[1][3],
                        "clip_frac": actor_loss[1][4],
                    }
                    
                    return (actor_train_state, critic_train_state), loss_info

                train_states, traj_batch, advantages, targets, rng = update_state
                rng, _rng = jax.random.split(rng)
                batch_size = config["MINIBATCH_SIZE"] * config["NUM_MINIBATCHES"]
                assert (
                    batch_size == config["NUM_STEPS"] * config["NUM_ACTORS"]
                ), "batch size must be equal to number of steps * number of actors"
                permutation = jax.random.permutation(_rng, batch_size)
                batch = (traj_batch, advantages, targets)
                batch = jax.tree.map(
                    lambda x: x.reshape((batch_size,) + x.shape[2:]), batch
                )
                shuffled_batch = jax.tree.map(
                    lambda x: jnp.take(x, permutation, axis=0), batch
                )
                minibatches = jax.tree.map(
                    lambda x: jnp.reshape(
                        x, [config["NUM_MINIBATCHES"], -1] + list(x.shape[1:])
                    ),
                    shuffled_batch,
                )
                train_states, loss_info = jax.lax.scan(
                    _update_minbatch, train_states, minibatches
                )
                update_state = (train_states, traj_batch, advantages, targets, rng)
                return update_state, loss_info

            def callback(metric):
                wandb.log(
                    metric,
                    step=metric["update_step"],
                )
                print(metric)
                
            update_state = (train_states, traj_batch, advantages, targets, rng)
            update_state, loss_info = jax.lax.scan(
                _update_epoch, update_state, None, config["UPDATE_EPOCHS"]
            )            
            train_states = update_state[0]
            metric = traj_batch.info
            rng = update_state[-1]

            update_count = update_count + 1
            r0 = {"ratio0": loss_info["ratio"][0,0].mean()}
            loss_info = jax.tree.map(lambda x: x.mean(), loss_info)
            metric = jax.tree.map(lambda x: x.mean(), metric)
            metric["update_step"] = update_count
            metric["env_step"] = update_count * config["NUM_STEPS"] * config["NUM_ENVS"]
            metric = {**metric, **loss_info, **r0}
            jax.experimental.io_callback(callback, None, metric)
            runner_state = (train_states, env_state, last_obs, update_count, rng)
            return (runner_state, update_steps), metric

        rng, _rng = jax.random.split(rng)
        runner_state = ((actor_train_state, critic_train_state), env_state, obsv, 0, _rng)
        #runner_state = ((actor_train_state, critic_train_state), env_state, obsv, jnp.zeros((config["NUM_ACTORS"]), dtype=bool), _rng,)
        runner_state, metric = jax.lax.scan(
            _update_step, (runner_state, 0), None, config["NUM_UPDATES"]
        )
        return {"runner_state": runner_state, "metrics": metric}

    return train

@hydra.main(version_base=None, config_path="config", config_name="mappo_homogenous_ff_mabrax")
def main(config):

    config = OmegaConf.to_container(config)
    wandb.init(
        entity=config["ENTITY"],
        project=config["PROJECT"],
        tags=["MAPPO", "FF", config["ENV_NAME"]],
        config=config,
        mode=config["WANDB_MODE"],
    )
    rng = jax.random.PRNGKey(config["SEED"])
    with jax.disable_jit(False):
        train_jit = jax.jit(make_train(config)) 
        out = train_jit(rng)

    
if __name__=="__main__":
    main()
