import torch
import torch.nn as nn
import math
import torch.utils.model_zoo as model_zoo
from .resnet18_encoder import mode_context


def conv3x3(in_planes, out_planes, stride=1):
    """3x3 convolution with padding"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=1, bias=False)


def conv1x1(in_planes, out_planes, stride=1):
    """1x1 convolution"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)


class AdapterLayer(nn.Module):
    def __init__(self, inplanes, planes, stride=1):
        super(AdapterLayer, self).__init__()
        self.adapter = conv1x1(inplanes, planes, stride)

    def forward(self, x):
        x = self.adapter(x)
        return x


class LowRankAdapterLayer(nn.Module):
    def __init__(self, inplanes, planes, stride=1):
        super(LowRankAdapterLayer, self).__init__()
        self.adapter1 = nn.Conv2d(inplanes, inplanes // 2, kernel_size=1, stride=1, padding=0, bias=False)
        self.adapter2 = nn.Conv2d(inplanes // 2, inplanes // 2, kernel_size=3, stride=stride, padding=1, bias=False)
        self.adapter3 = nn.Conv2d(inplanes // 2, planes, kernel_size=1, stride=1, padding=0, bias=False)

        # 初始化权重
        # nn.init.kaiming_normal_(self.conv1.weight, mode='fan_out', nonlinearity='relu')
        # nn.init.kaiming_normal_(self.conv2.weight, mode='fan_out', nonlinearity='relu')
        # nn.init.kaiming_normal_(self.conv3.weight, mode='fan_out', nonlinearity='relu')

    def forward(self, x):
        out = self.adapter1(x)
        out = self.adapter2(out)
        out = self.adapter3(out)
        return out


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1, downsample=None, last=False, include_adapter=False):
        super(BasicBlock, self).__init__()
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = nn.BatchNorm2d(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = nn.BatchNorm2d(planes)
        self.downsample = downsample
        self.stride = stride
        self.last = last
        self.include_adapter = include_adapter
        if self.include_adapter:
            # self.adapter1 = LowRankAdapterLayer(inplanes, planes, stride)
            # self.adapter2 = LowRankAdapterLayer(planes, planes)
            self.adapter1 = AdapterLayer(inplanes, planes, stride)
            self.adapter2 = AdapterLayer(planes, planes)
        else:
            self.adapter1 = None
            self.adapter2 = None
        self.alpha = nn.Parameter(torch.tensor([2.5e-1]), requires_grad=False)
        self.beta = nn.Parameter(torch.tensor([1.0]), requires_grad=False)

    def forward(self, x):
        identity = x

        # First convolutional layer
        out = self.conv1(x)
        if self.include_adapter and self.adapter1 is not None and mode_context.get() == 'parallel_adapters':
            adapter_out = self.adapter1(x)  # Compute adapter1 output
            out = out + self.alpha * adapter_out  # Adjust out with adapter1 output
        out = self.bn1(out)
        out = self.relu(out)

        # Second convolutional layer and adapter2 in parallel
        conv_out = self.conv2(out)
        adapter2_out = self.adapter2(
            out) if self.include_adapter and self.adapter2 is not None and mode_context.get() == 'parallel_adapters' else torch.zeros_like(
            conv_out)

        # Combine outputs from conv2 and adapter2
        out = conv_out + self.alpha * adapter2_out
        out = self.bn2(out)

        # Downsample if necessary
        if self.downsample is not None:
            identity = self.downsample(x)

        # Add identity to the output
        out += identity
        out = self.relu(out)

        return out


# class BasicBlock(nn.Module):
#     expansion = 1
#
#     def __init__(self, inplanes, planes, stride=1, downsample=None, groups=1,
#                  base_width=64, dilation=1, norm_layer=None, mode='parallel_adapters'):
#         super(BasicBlock, self).__init__()
#         if norm_layer is None:
#             norm_layer = nn.BatchNorm2d
#         self.mode = mode
#
#         self.conv1 = conv3x3(inplanes, planes, stride)
#         self.bn1 = norm_layer(planes)
#         self.relu = nn.ReLU(inplace=True)
#
#         self.conv2 = conv3x3(planes, planes)
#         self.bn2 = norm_layer(planes)
#
#         self.downsample = downsample
#         self.stride = stride
#
#         if mode == 'parallel_adapters':
#             # 为每个卷积层添加并行的Adapter层
#             self.adapter1 = AdapterLayer(inplanes, planes, stride)
#             self.adapter2 = AdapterLayer(planes, planes)
#
#     def forward(self, x):
#         identity = x
#
#         if self.mode == 'parallel_adapters':
#             out1 = self.conv1(x)
#             out2 = self.adapter1(x)  # Adapter处理相同的输入
#             out = out1 + out2  # 合并卷积层和Adapter层的输出
#         else:
#             out = self.conv1(x)
#
#         out = self.bn1(out)
#         out = self.relu(out)
#
#         if self.mode == 'parallel_adapters':
#             out1 = self.conv2(out)
#             out2 = self.adapter2(out)  # Adapter处理相同的输入
#             out = out1 + out2  # 合并第二个卷积层和第二个Adapter层的输出
#         else:
#             out = self.conv2(out)
#
#         out = self.bn2(out)
#
#         if self.downsample is not None:
#             identity = self.downsample(x)
#
#         out += identity
#         out = self.relu(out)
#
#         return out


class ResNet(nn.Module):

    def __init__(self, block, layers, num_classes=10, mode='parallel_adapters'):
        self.mode = mode
        self.inplanes = 16
        super(ResNet, self).__init__()
        self.conv1 = nn.Conv2d(3, 16, kernel_size=3, stride=1, padding=1,
                               bias=False)
        self.bn1 = nn.BatchNorm2d(16)
        self.relu = nn.ReLU(inplace=True)
        self.layer1 = self._make_layer(block, 16, layers[0])
        self.layer2 = self._make_layer(block, 32, layers[1], stride=2)
        self.layer3 = self._make_layer(block, 64, layers[2], stride=2, last_phase=True)
        # self.avgpool = nn.AvgPool2d(8, stride=1)

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)

    def _make_layer(self, block, planes, blocks, stride=1, last_phase=False):
        downsample = None
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                nn.Conv2d(self.inplanes, planes * block.expansion,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample))
        self.inplanes = planes * block.expansion
        if last_phase:
            for i in range(1, blocks - 1):
                layers.append(block(self.inplanes, planes, include_adapter=True))
            layers.append(block(self.inplanes, planes, last=True))
        else:
            for i in range(1, blocks):
                layers.append(block(self.inplanes, planes, include_adapter=True))

        return nn.Sequential(*layers)

    def forward(self, x):
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)

        # x = self.avgpool(x)
        # x = x.view(x.size(0), -1)
        # x = self.fc(x)

        return x


def resnet20(**kwargs):
    n = 3
    model = ResNet(BasicBlock, [n, n, n], **kwargs)
    return model
