using LinearAlgebra
using SparseArrays
include("struct.jl")


function edge_weights(G::AdjacencyList, X::Matrix{Float64}, gamma::Float64)
# Compute all edge weights for G
# The edge weights are aligned with G.neighbors

    weights = Vector{Vector{Float64}}()

    for i in 1:G.nv
        w = Vector{Float64}()
        for j in G.neighbors[i]
            push!(w, exp(-gamma*norm(X[i,:]-X[j,:])^2))
        end
        push!(weights, w)
    end

    return weights
end

function compute_conductance(G::AdjacencyList, C::Vector{Int64};
    weighted::Bool=true, edge_weight=nothing, attributes=nothing,
    gamma::Float64=0.0)

    if weighted
        if edge_weight === nothing
            edge_weight = edge_weights(G, attributes, gamma)
        end
        weighted_degree = [sum(w) for w in edge_weight]
        vol_G = sum(weighted_degree)
        vol_C = sum(weighted_degree[C])
    else
        vol_G = sum(G.degree)
        vol_C = sum(G.degree[C])
    end

    cut_C = 0
    set_C = Set(C)

    for i in C
        for (idx,j) in enumerate(G.neighbors[i])
            if !(j in set_C)
                if weighted
                    w_ij = edge_weight[i][idx]
                else
                    w_ij = 1.0
                end
                cut_C += w_ij
            end
        end
    end

    return cut_C, vol_C, cut_C/min(vol_C, vol_G - vol_C)
end

function sweepcut(G::AdjacencyList, x::Vector{Float64},
        edge_weight::Vector{Vector{Float64}}; min_size::Int64=1)

    weighted_degree = [sum(w) for w in edge_weight]
    vol_G = sum(weighted_degree)
    vol_C = 0
    cut_C = 0
    C = Set{Int64}()
    best_cluster = Set{Int64}()
    best_conductance = 1.0
    x_nzind = findall(!iszero, x)
    x_nzval = x[x_nzind]
    sorted_ind = x_nzind[sortperm(x_nzval, rev=true)]
    min_size = min(min_size, G.nv-1)
    for i in sorted_ind
        push!(C, i)
        if length(C) == G.nv
            break
        end
        vol_C += weighted_degree[i]
        for (idx,j) in enumerate(G.neighbors[i])
            if j in C
                cut_C -= edge_weight[i][idx]
            else
                cut_C += edge_weight[i][idx]
            end
        end
        if length(C) >= min_size
            cond_C = cut_C / min(vol_C, vol_G - vol_C)
            if cond_C <= best_conductance
                best_cluster = copy(C)
                best_conductance = cond_C
            end
        end
    end
    return collect(best_cluster), best_conductance
end

function adjacency_matrix(G::AdjacencyList)
    n = G.nv
    colptr = Vector{Int}(undef, n+1)
    colptr[1] = 1
    for i = 1:n
        colptr[i+1] = colptr[i] + G.degree[i]
    end
    rowval = Int[]
    for i in 1:n
        for j in G.neighbors[i]
            push!(rowval, j)
        end
    end
    nzval = ones(Int, sum(G.degree))
    return SparseMatrixCSC(n, n, colptr, rowval, nzval)
end

function averaging_matrix(A::SparseMatrixCSC{Int,Int}, d::Vector{Int})
    ind = [i == 0 ? 0.0 : 1.0 for i in d]
    dsqn = [i == 0 ? 0.0 : 1.0/i for i in sqrt.(d)]
    Dsqn = spdiagm(0 => dsqn)
    return spdiagm(0 => ind) - Dsqn*(A*Dsqn)
end


function compute_symmetric_laplacian(A::SparseMatrixCSC{Int,Int}, d::Vector{Int})
    ind = [i == 0 ? 0.0 : 1.0 for i in d]
    dsqn = [i == 0 ? 0.0 : 1.0/i for i in sqrt.(d)]
    Dsqn = spdiagm(0 => dsqn)
    return spdiagm(0 => ind) - Dsqn*(A*Dsqn)
end

function compute_normalized_laplacian(A::SparseMatrixCSC{Int,Int}, d::Vector{Int})
    ind = [i == 0 ? 0.0 : 1.0 for i in d]
    dinv = [i == 0 ? 0.0 : 1.0/i for i in d]
    Dinv = spdiagm(0 => dinv)
    return spdiagm(0 => ind) - A*Dinv
end
