import numpy as np
from torchvision import datasets, transforms
from utils.toolkit import split_images_labels


class iData(object):
    train_trsf = []
    test_trsf = []
    common_trsf = []
    class_order = None


class iCIFAR10(iData):
    use_path = False
    train_trsf = [
        transforms.RandomCrop(32, padding=4),
        transforms.RandomHorizontalFlip(p=0.5),
        transforms.ColorJitter(brightness=63 / 255),
    ]
    test_trsf = []
    common_trsf = [
        transforms.ToTensor(),
        transforms.Normalize(
            mean=(0.4914, 0.4822, 0.4465), std=(0.2023, 0.1994, 0.2010)
        ),
    ]

    class_order = np.arange(10).tolist()

    def download_data(self):
        train_dataset = datasets.cifar.CIFAR10("./data", train=True, download=True)
        test_dataset = datasets.cifar.CIFAR10("./data", train=False, download=True)
        self.train_data, self.train_targets = train_dataset.data, np.array(
            train_dataset.targets
        )
        self.test_data, self.test_targets = test_dataset.data, np.array(
            test_dataset.targets
        )


class iCIFAR100(iData):
    use_path = False
    train_trsf = [
        transforms.RandomCrop(32, padding=4),
        transforms.RandomHorizontalFlip(),
        transforms.ColorJitter(brightness=63 / 255),
        transforms.ToTensor()
    ]
    test_trsf = [transforms.ToTensor()]
    common_trsf = [
        transforms.Normalize(
            mean=(0.5071, 0.4867, 0.4408), std=(0.2675, 0.2565, 0.2761)
        ),
    ]

    class_order = np.arange(100).tolist()

    def download_data(self):
        train_dataset = datasets.cifar.CIFAR100("./data", train=True, download=True)
        test_dataset = datasets.cifar.CIFAR100("./data", train=False, download=True)
        self.train_data, self.train_targets = train_dataset.data, np.array(
            train_dataset.targets
        )
        self.test_data, self.test_targets = test_dataset.data, np.array(
            test_dataset.targets
        )

def build_transform_coda_prompt(is_train, args):
    if is_train:        
        transform = [
            transforms.RandomResizedCrop(224),
            transforms.RandomHorizontalFlip(),
            transforms.ToTensor(),
            transforms.Normalize((0.0,0.0,0.0), (1.0,1.0,1.0)),
        ]
        return transform

    t = []
    if args["dataset"].startswith("imagenet"):
        t = [
            transforms.Resize(256),
            transforms.CenterCrop(224),
            transforms.ToTensor(),
            transforms.Normalize((0.0,0.0,0.0), (1.0,1.0,1.0)),
        ]
    else:
        t = [
            transforms.Resize(224),
            transforms.ToTensor(),
            transforms.Normalize((0.0,0.0,0.0), (1.0,1.0,1.0)),
        ]

    return t

def build_transform(is_train, args):
    input_size = 224
    resize_im = input_size > 32
    if is_train:
        scale = (0.05, 1.0)
        ratio = (3. / 4., 4. / 3.)
        
        transform = [
            transforms.RandomResizedCrop(input_size, scale=scale, ratio=ratio),
            transforms.RandomHorizontalFlip(p=0.5),
            transforms.ToTensor(),
        ]
        return transform

    t = []
    if resize_im:
        size = int((256 / 224) * input_size)
        t.append(
            transforms.Resize(size, interpolation=3),  # to maintain same ratio w.r.t. 224 images
        )
        t.append(transforms.CenterCrop(input_size))
    t.append(transforms.ToTensor())
    
    # return transforms.Compose(t)
    return t

class iCIFAR224(iData):
    def __init__(self, args):
        super().__init__()
        self.args = args
        self.use_path = False

        if args["model_name"] == "coda_prompt":
            self.train_trsf = build_transform_coda_prompt(True, args)
            self.test_trsf = build_transform_coda_prompt(False, args)
        else:
            self.train_trsf = build_transform(True, args)
            self.test_trsf = build_transform(False, args)
        self.common_trsf = [
            # transforms.ToTensor(),
        ]

        self.class_order = np.arange(100).tolist()

    def download_data(self):
        train_dataset = datasets.cifar.CIFAR100("./data", train=True, download=True)
        test_dataset = datasets.cifar.CIFAR100("./data", train=False, download=True)
        self.train_data, self.train_targets = train_dataset.data, np.array(
            train_dataset.targets
        )
        self.test_data, self.test_targets = test_dataset.data, np.array(
            test_dataset.targets
        )

class iImageNet1000(iData):
    use_path = True
    train_trsf = [
        transforms.RandomResizedCrop(224),
        transforms.RandomHorizontalFlip(),
        transforms.ColorJitter(brightness=63 / 255),
    ]
    test_trsf = [
        transforms.Resize(256),
        transforms.CenterCrop(224),
    ]
    common_trsf = [
        transforms.ToTensor(),
        transforms.Normalize(mean=[0.485, 0.456, 0.406], std=[0.229, 0.224, 0.225]),
    ]

    class_order = np.arange(1000).tolist()

    def download_data(self):
        assert 0, "You should specify the folder of your dataset"
        train_dir = "[DATA-PATH]/train/"
        test_dir = "[DATA-PATH]/val/"

        train_dset = datasets.ImageFolder(train_dir)
        test_dset = datasets.ImageFolder(test_dir)

        self.train_data, self.train_targets = split_images_labels(train_dset.imgs)
        self.test_data, self.test_targets = split_images_labels(test_dset.imgs)


class iImageNet100(iData):
    use_path = True
    train_trsf = [
        transforms.RandomResizedCrop(224),
        transforms.RandomHorizontalFlip(),
    ]
    test_trsf = [
        transforms.Resize(256),
        transforms.CenterCrop(224),
    ]
    common_trsf = [
        transforms.ToTensor(),
        transforms.Normalize(mean=[0.485, 0.456, 0.406], std=[0.229, 0.224, 0.225]),
    ]

    class_order = np.arange(1000).tolist()

    def download_data(self):
        assert 0, "You should specify the folder of your dataset"
        train_dir = "[DATA-PATH]/train/"
        test_dir = "[DATA-PATH]/val/"

        train_dset = datasets.ImageFolder(train_dir)
        test_dset = datasets.ImageFolder(test_dir)

        self.train_data, self.train_targets = split_images_labels(train_dset.imgs)
        self.test_data, self.test_targets = split_images_labels(test_dset.imgs)


class iImageNetR(iData):
    def __init__(self, args):
        super().__init__()
        self.args = args
        self.use_path = True

        if args["model_name"] == "coda_prompt":
            self.train_trsf = build_transform_coda_prompt(True, args)
            self.test_trsf = build_transform_coda_prompt(False, args)
        else:
            self.train_trsf = build_transform(True, args)
            self.test_trsf = build_transform(False, args)
        self.common_trsf = [
            # transforms.ToTensor(),
        ]

        self.class_order = np.arange(200).tolist()

    def download_data(self):
        # assert 0, "You should specify the folder of your dataset"
        train_dir = "/dataset/imagenet-r/train/"
        test_dir = "/dataset/imagenet-r/test/"

        train_dset = datasets.ImageFolder(train_dir)
        test_dset = datasets.ImageFolder(test_dir)

        self.train_data, self.train_targets = split_images_labels(train_dset.imgs)
        self.test_data, self.test_targets = split_images_labels(test_dset.imgs)

        
        
class iImageNetR100_1(iData):
    def __init__(self, args):
        super().__init__()
        self.args = args
        self.use_path = True

        if args["model_name"] == "coda_prompt":
            self.train_trsf = build_transform_coda_prompt(True, args)
            self.test_trsf = build_transform_coda_prompt(False, args)
        else:
            self.train_trsf = build_transform(True, args)
            self.test_trsf = build_transform(False, args)
        self.common_trsf = [
            # transforms.ToTensor(),
        ]

        self.class_order = np.arange(200).tolist()

    def download_data(self):
        # assert 0, "You should specify the folder of your dataset"
        train_dir = "/dataset/imagenet-r/train/"
        test_dir = "/dataset/imagenet-r/test/"

        train_dset = datasets.ImageFolder(train_dir)
        test_dset = datasets.ImageFolder(test_dir)

        self.train_data_temp, self.train_targets_temp = split_images_labels(train_dset.imgs)
        self.test_data_temp, self.test_targets_temp = split_images_labels(test_dset.imgs)
        
        tr_data_len = self.train_targets_temp.shape[0]
        te_data_len = self.test_targets_temp.shape[0]
        
        order = [i for i in range(len(np.unique(self.train_targets_temp)))][:100]
        np.random.shuffle(order)
        
        self.train_data = []
        self.train_targets = []
        
        for i in range(tr_data_len):
            
            if self.train_targets_temp[i] in order:
                self.train_data.append(self.train_data_temp[i])
                self.train_targets.append(self.train_targets_temp[i])
                
        self.train_data = np.array(self.train_data)
        self.train_targets = np.array(self.train_targets) % 100
                
        
        self.test_data = []
        self.test_targets = []
        
        for i in range(te_data_len):
            
            if self.test_targets_temp[i] in order:
                self.test_data.append(self.test_data_temp[i])
                self.test_targets.append(self.test_targets_temp[i])
        
        self.test_data = np.array(self.test_data)
        self.test_targets = np.array(self.test_targets) % 100
            
        print ('len(train_data) : ', self.train_data.shape[0],' len(train_targets) : ',  self.train_targets.shape[0])
        print ('len(test_data) : ', self.test_data.shape[0],' len(test_targets) : ',  self.test_targets.shape[0])
        

class iImageNetR100_2(iData):
    def __init__(self, args):
        super().__init__()
        self.args = args
        self.use_path = True

        if args["model_name"] == "coda_prompt":
            self.train_trsf = build_transform_coda_prompt(True, args)
            self.test_trsf = build_transform_coda_prompt(False, args)
        else:
            self.train_trsf = build_transform(True, args)
            self.test_trsf = build_transform(False, args)
        self.common_trsf = [
            # transforms.ToTensor(),
        ]

        self.class_order = np.arange(200).tolist()

    def download_data(self):
        # assert 0, "You should specify the folder of your dataset"
        train_dir = "/dataset/imagenet-r/train/"
        test_dir = "/dataset/imagenet-r/test/"

        train_dset = datasets.ImageFolder(train_dir)
        test_dset = datasets.ImageFolder(test_dir)

        self.train_data_temp, self.train_targets_temp = split_images_labels(train_dset.imgs)
        self.test_data_temp, self.test_targets_temp = split_images_labels(test_dset.imgs)
        
        tr_data_len = self.train_targets_temp.shape[0]
        te_data_len = self.test_targets_temp.shape[0]
        
        order = [i for i in range(len(np.unique(self.train_targets_temp)))][100:]
        np.random.shuffle(order)
        
        self.train_data = []
        self.train_targets = []
        
        for i in range(tr_data_len):
            
            if self.train_targets_temp[i] in order:
                self.train_data.append(self.train_data_temp[i])
                self.train_targets.append(self.train_targets_temp[i])
                
        self.train_data = np.array(self.train_data)
        self.train_targets = np.array(self.train_targets) % 100
                
        
        self.test_data = []
        self.test_targets = []
        
        for i in range(te_data_len):
            
            if self.test_targets_temp[i] in order:
                self.test_data.append(self.test_data_temp[i])
                self.test_targets.append(self.test_targets_temp[i])
        
        self.test_data = np.array(self.test_data)
        self.test_targets = np.array(self.test_targets) % 100
            
        print ('len(train_data) : ', self.train_data.shape[0],' len(train_targets) : ',  self.train_targets.shape[0])
        print ('len(test_data) : ', self.test_data.shape[0],' len(test_targets) : ',  self.test_targets.shape[0])
        
        
        

class iImageNetA(iData):
    use_path = True
    
    train_trsf = build_transform(True, None)
    test_trsf = build_transform(False, None)
    common_trsf = [    ]

    class_order = np.arange(200).tolist()

    def download_data(self):
        # assert 0, "You should specify the folder of your dataset"
        train_dir = "/dataset/imagenet-a/train/"
        test_dir = "/dataset/imagenet-a/test/"

        train_dset = datasets.ImageFolder(train_dir)
        test_dset = datasets.ImageFolder(test_dir)

        self.train_data, self.train_targets = split_images_labels(train_dset.imgs)
        self.test_data, self.test_targets = split_images_labels(test_dset.imgs)
        

class iImageNetA100_1(iData):
    use_path = True
    
    train_trsf = build_transform(True, None)
    test_trsf = build_transform(False, None)
    common_trsf = [    ]

    class_order = np.arange(200).tolist()

    def download_data(self):
        # assert 0, "You should specify the folder of your dataset"
        train_dir = "/dataset/imagenet-a/train/"
        test_dir = "/dataset/imagenet-a/test/"

        train_dset = datasets.ImageFolder(train_dir)
        test_dset = datasets.ImageFolder(test_dir)

        self.train_data_temp, self.train_targets_temp = split_images_labels(train_dset.imgs)
        self.test_data_temp, self.test_targets_temp = split_images_labels(test_dset.imgs)
        
        tr_data_len = self.train_targets_temp.shape[0]
        te_data_len = self.test_targets_temp.shape[0]
        
        order = [i for i in range(len(np.unique(self.train_targets_temp)))][:100]
        np.random.shuffle(order)
        
        self.train_data = []
        self.train_targets = []
        
        for i in range(tr_data_len):
            
            if self.train_targets_temp[i] in order:
                self.train_data.append(self.train_data_temp[i])
                self.train_targets.append(self.train_targets_temp[i])
                
        self.train_data = np.array(self.train_data)
        self.train_targets = np.array(self.train_targets) % 100
                
        
        self.test_data = []
        self.test_targets = []
        
        for i in range(te_data_len):
            
            if self.test_targets_temp[i] in order:
                self.test_data.append(self.test_data_temp[i])
                self.test_targets.append(self.test_targets_temp[i])
        
        self.test_data = np.array(self.test_data)
        self.test_targets = np.array(self.test_targets) % 100
            
        print ('len(train_data) : ', self.train_data.shape[0],' len(train_targets) : ',  self.train_targets.shape[0])
        print ('len(test_data) : ', self.test_data.shape[0],' len(test_targets) : ',  self.test_targets.shape[0])
        

        

class iImageNetA100_2(iData):
    use_path = True
    
    train_trsf = build_transform(True, None)
    test_trsf = build_transform(False, None)
    common_trsf = [    ]

    class_order = np.arange(200).tolist()

    def download_data(self):
        # assert 0, "You should specify the folder of your dataset"
        train_dir = "/dataset/imagenet-a/train/"
        test_dir = "/dataset/imagenet-a/test/"

        train_dset = datasets.ImageFolder(train_dir)
        test_dset = datasets.ImageFolder(test_dir)

        self.train_data_temp, self.train_targets_temp = split_images_labels(train_dset.imgs)
        self.test_data_temp, self.test_targets_temp = split_images_labels(test_dset.imgs)
        
        tr_data_len = self.train_targets_temp.shape[0]
        te_data_len = self.test_targets_temp.shape[0]
        
        order = [i for i in range(len(np.unique(self.train_targets_temp)))][100:]
        np.random.shuffle(order)
        
        self.train_data = []
        self.train_targets = []
        
        for i in range(tr_data_len):
            
            if self.train_targets_temp[i] in order:
                self.train_data.append(self.train_data_temp[i])
                self.train_targets.append(self.train_targets_temp[i])
                
        self.train_data = np.array(self.train_data)
        self.train_targets = np.array(self.train_targets) % 100
                
        
        self.test_data = []
        self.test_targets = []
        
        for i in range(te_data_len):
            
            if self.test_targets_temp[i] in order:
                self.test_data.append(self.test_data_temp[i])
                self.test_targets.append(self.test_targets_temp[i])
        
        self.test_data = np.array(self.test_data)
        self.test_targets = np.array(self.test_targets) % 100
            
        print ('len(train_data) : ', self.train_data.shape[0],' len(train_targets) : ',  self.train_targets.shape[0])
        print ('len(test_data) : ', self.test_data.shape[0],' len(test_targets) : ',  self.test_targets.shape[0])
        



class CUB(iData):
    use_path = True
    
    train_trsf = build_transform(True, None)
    test_trsf = build_transform(False, None)
    common_trsf = [    ]

    class_order = np.arange(200).tolist()

    def download_data(self):
        # assert 0, "You should specify the folder of your dataset"
        train_dir = "./data/cub/train/"
        test_dir = "./data/cub/test/"

        train_dset = datasets.ImageFolder(train_dir)
        test_dset = datasets.ImageFolder(test_dir)

        self.train_data, self.train_targets = split_images_labels(train_dset.imgs)
        self.test_data, self.test_targets = split_images_labels(test_dset.imgs)
        



class CUB100_1(iData):
    use_path = True
    
    train_trsf = build_transform(True, None)
    test_trsf = build_transform(False, None)
    common_trsf = [    ]

    class_order = np.arange(100).tolist()

    def download_data(self):
        # assert 0, "You should specify the folder of your dataset"
        train_dir = "/dataset/cub/train/"
        test_dir = "/dataset/cub/test/"

        train_dset = datasets.ImageFolder(train_dir)
        test_dset = datasets.ImageFolder(test_dir)

        self.train_data_temp, self.train_targets_temp = split_images_labels(train_dset.imgs)
        self.test_data_temp, self.test_targets_temp = split_images_labels(test_dset.imgs)
        
        tr_data_len = self.train_targets_temp.shape[0]
        te_data_len = self.test_targets_temp.shape[0]
        
        order = [i for i in range(len(np.unique(self.train_targets_temp)))][:100]
        np.random.shuffle(order)
        
        self.train_data = []
        self.train_targets = []
        
        for i in range(tr_data_len):
            
            if self.train_targets_temp[i] in order:
                self.train_data.append(self.train_data_temp[i])
                self.train_targets.append(self.train_targets_temp[i])
                
        self.train_data = np.array(self.train_data)
        self.train_targets = np.array(self.train_targets) % 100
                
        
        self.test_data = []
        self.test_targets = []
        
        for i in range(te_data_len):
            
            if self.test_targets_temp[i] in order:
                self.test_data.append(self.test_data_temp[i])
                self.test_targets.append(self.test_targets_temp[i])
        
        self.test_data = np.array(self.test_data)
        self.test_targets = np.array(self.test_targets) % 100
            
        print ('len(train_data) : ', self.train_data.shape[0],' len(train_targets) : ',  self.train_targets.shape[0])
        print ('len(test_data) : ', self.test_data.shape[0],' len(test_targets) : ',  self.test_targets.shape[0])
        



class CUB100_2(iData):
    use_path = True
    
    train_trsf = build_transform(True, None)
    test_trsf = build_transform(False, None)
    common_trsf = [    ]

    class_order = np.arange(100).tolist()

    def download_data(self):
        # assert 0, "You should specify the folder of your dataset"
        train_dir = "/dataset/cub/train/"
        test_dir = "/dataset/cub/test/"

        train_dset = datasets.ImageFolder(train_dir)
        test_dset = datasets.ImageFolder(test_dir)

        self.train_data_temp, self.train_targets_temp = split_images_labels(train_dset.imgs)
        self.test_data_temp, self.test_targets_temp = split_images_labels(test_dset.imgs)
        
        tr_data_len = self.train_targets_temp.shape[0]
        te_data_len = self.test_targets_temp.shape[0]
        
        order = [i for i in range(len(np.unique(self.train_targets_temp)))][100:]
        np.random.shuffle(order)
        
        self.train_data = []
        self.train_targets = []
        
        for i in range(tr_data_len):
            
            if self.train_targets_temp[i] in order:
                self.train_data.append(self.train_data_temp[i])
                self.train_targets.append(self.train_targets_temp[i])
                
        self.train_data = np.array(self.train_data)
        self.train_targets = np.array(self.train_targets) % 100
                
        
        self.test_data = []
        self.test_targets = []
        
        for i in range(te_data_len):
            
            if self.test_targets_temp[i] in order:
                self.test_data.append(self.test_data_temp[i])
                self.test_targets.append(self.test_targets_temp[i])
        
        self.test_data = np.array(self.test_data)
        self.test_targets = np.array(self.test_targets) % 100
            
        print ('len(train_data) : ', self.train_data.shape[0],' len(train_targets) : ',  self.train_targets.shape[0])
        print ('len(test_data) : ', self.test_data.shape[0],' len(test_targets) : ',  self.test_targets.shape[0])
        


class objectnet(iData):
    use_path = True
    
    train_trsf = build_transform(True, None)
    test_trsf = build_transform(False, None)
    common_trsf = [    ]

    class_order = np.arange(200).tolist()

    def download_data(self):
        # assert 0, "You should specify the folder of your dataset"
        train_dir = "./data/objectnet/train/"
        test_dir = "./data/objectnet/test/"

        train_dset = datasets.ImageFolder(train_dir)
        test_dset = datasets.ImageFolder(test_dir)

        self.train_data, self.train_targets = split_images_labels(train_dset.imgs)
        self.test_data, self.test_targets = split_images_labels(test_dset.imgs)


class omnibenchmark(iData):
    use_path = True
    
    train_trsf = build_transform(True, None)
    test_trsf = build_transform(False, None)
    common_trsf = [    ]

    class_order = np.arange(300).tolist()

    def download_data(self):
        # assert 0, "You should specify the folder of your dataset"
        train_dir = "./data/omnibenchmark/train/"
        test_dir = "./data/omnibenchmark/test/"

        train_dset = datasets.ImageFolder(train_dir)
        test_dset = datasets.ImageFolder(test_dir)

        self.train_data, self.train_targets = split_images_labels(train_dset.imgs)
        self.test_data, self.test_targets = split_images_labels(test_dset.imgs)



class vtab(iData):
    use_path = True
    
    train_trsf = build_transform(True, None)
    test_trsf = build_transform(False, None)
    common_trsf = [    ]

    class_order = np.arange(50).tolist()

    def download_data(self):
        # assert 0, "You should specify the folder of your dataset"
        train_dir = "./data/vtab-cil/vtab/train/"
        test_dir = "./data/vtab-cil/vtab/test/"

        train_dset = datasets.ImageFolder(train_dir)
        test_dset = datasets.ImageFolder(test_dir)

        print(train_dset.class_to_idx)
        print(test_dset.class_to_idx)

        self.train_data, self.train_targets = split_images_labels(train_dset.imgs)
        self.test_data, self.test_targets = split_images_labels(test_dset.imgs)