import csv
import os
import argparse
from pymatgen.ext.matproj import MPRester
from emmet.core.summary import HasProps

def download_and_organize_cif_files(api_key, csv_file_path, root_dir):
    # Initialize the MPRester with your API key
    with MPRester(api_key) as m:
        with open(csv_file_path, newline='') as csvfile:
            reader = csv.reader(csvfile)
            next(reader)
            
            # Ensure the root directory exists
            os.makedirs(root_dir, exist_ok=True)
            
            id_prop_path = os.path.join(root_dir, 'id_prop.csv')
            write_header = not os.path.exists(id_prop_path)
            with open(id_prop_path, 'a', newline='') as id_prop_file:
                prop_writer = csv.writer(id_prop_file)
                if write_header:
                    prop_writer.writerow(['ID', 'Formation Energy'])  # Update header to reflect property type
                
                for row in reader:
                    material_id = row[0]  # Assuming the material ID is in the first column
                    try:
                        material_data = m.summary.search(material_ids=[material_id], fields=["formation_energy_per_atom"])
                        formation_energy = material_data[0].formation_energy_per_atom
                        
                        # Save the CIF file
                        cif_data = m.get_structure_by_material_id(material_id, final=True)
                        cif_path = os.path.join(root_dir, f'{material_id}.cif')
                        cif_data.to(filename=cif_path)
                        
                        # Write to id_prop.csv, using the formation energy as the property value
                        prop_writer.writerow([material_id, formation_energy])
                        
                    except Exception as e:
                        print(f"Failed to fetch data or save CIF for {material_id}: {e}")

def search_materials_and_save_data(api_key, root_dir):
    # Initialize the MPRester with your API key
    with MPRester(api_key) as mpr:
        # Search for materials with all required properties
        mats = mpr.materials.summary.search(has_props=[HasProps.thermo, HasProps.electronic_structure], fields=["material_id", "formation_energy_per_atom", "band_gap"])
        # mats = mpr.materials.summary.search(has_props=[HasProps.elasticity], fields=["material_id", "bulk_modulus", "shear_modulus"])
        os.makedirs(root_dir, exist_ok=True)
        
        id_prop_formation_path = os.path.join(root_dir, 'id_prop_formation.csv')
        with open(id_prop_formation_path, 'w', newline='') as id_prop_formation_file:
            formation_writer = csv.writer(id_prop_formation_file)
            formation_writer.writerow(['ID', 'Formation Energy'])

        id_prop_bandgap_path = os.path.join(root_dir, 'id_prop_bandgap.csv')
        with open(id_prop_bandgap_path, 'w', newline='') as id_prop_bandgap_file:
            bandgap_writer = csv.writer(id_prop_bandgap_file)
            bandgap_writer.writerow(['ID', 'Band Gap'])
        
        for material in mats:
            try:           
                material_id = material.material_id
                formation_energy = material.formation_energy_per_atom
                band_gap = material.band_gap
                # Save the CIF file
                cif_data = mpr.get_structure_by_material_id(material_id, final=True)
                cif_path = os.path.join(root_dir, f'{material_id}.cif')
                cif_data.to(filename=cif_path)
                
                # Write to formation energy prop file
                with open(id_prop_formation_path, 'a', newline='') as id_prop_formation_file:
                    formation_writer = csv.writer(id_prop_formation_file)
                    formation_writer.writerow([material_id, formation_energy])
                
                # Write to band gap prop file
                with open(id_prop_bandgap_path, 'a', newline='') as id_prop_bandgap_file:
                    bandgap_writer = csv.writer(id_prop_bandgap_file)
                    bandgap_writer.writerow([material_id, band_gap])
                
            except Exception as e:
                print(f"Failed to fetch data or save CIF for {material_id}: {e}")


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description="Download material data and CIF files.")
    parser.add_argument("--api-key", required=True, help="Your Materials Project API key.")
    parser.add_argument("--root-dir", default="data/mp-elastic", help="Root directory to save data.")
    args = parser.parse_args()

    search_materials_and_save_data(args.api_key, args.root_dir)
