import torch
import torch.optim as optim
from tqdm.auto import tqdm
from src.utils.utility import get_lr_scheduler


def calculate_test_loss(model, loader, loss_fn, device, epoch):
    model.eval()
    losses = []
    with torch.no_grad():
        for batch_idx, (data, target) in enumerate(tqdm(loader, total=len(loader))):
            data, target = data.to(device), target.to(device)
            output = model(data)
            loss = -1 * loss_fn(output.logits, target)
            losses.append(loss)
        total_loss = torch.mean(torch.tensor(losses)).item()
        print("Val Epoch: {} \tLoss: {}".format(epoch, total_loss))
    return total_loss


def train_unlearn_step(
    model,
    train_loader,
    optimizer,
    loss_fn,
    epoch,
    device,
    lr_scheduler=None,
):
    model.train()
    losses = []
    for batch_idx, (data, target) in enumerate(
        tqdm(train_loader, total=len(train_loader))
    ):
        data, target = data.to(device), target.to(device)
        optimizer.zero_grad()
        output = model(data)
        loss = -1 * loss_fn(output.logits, target)
        loss.backward()
        optimizer.step()
        if lr_scheduler is not None:
            lr_scheduler.step(epoch + batch_idx / len(train_loader))
        losses.append(loss)
    total_loss = torch.mean(torch.tensor(losses)).item()
    print("Train Epoch: {} \tLoss: {}".format(epoch, total_loss))
    return model, total_loss


def train_unlearn(
    model,
    train_loader,
    device,
    lr,
    num_epochs: int = 10000,
    val_loader=None,
    loss_fn=None,
    lr_scheduler: None | str = None,
    patience: int = 5,
    weight_decay: float = 1e-6,
    save_path: None | str = None,
):
    model.to(device)
    optimizer = optim.AdamW(model.parameters(), lr=lr, weight_decay=weight_decay)
    lrs = get_lr_scheduler(lr_scheduler=lr_scheduler, optimizer=optimizer)
    best_loss = torch.inf
    counter = 0
    for e in range(1, num_epochs + 1):
        model, epoch_loss = train_unlearn_step(
            model=model,
            train_loader=train_loader,
            optimizer=optimizer,
            loss_fn=loss_fn,
            epoch=e,
            device=device,
            lr_scheduler=(
                lrs.scheduler
                if lrs.scheduler is not None and lrs.is_batch_scheduler
                else None
            ),
        )

        if val_loader is not None:
            val_loss = calculate_test_loss(model, val_loader, loss_fn, device, epoch=e)

            if lrs.scheduler is not None and (not lrs.is_batch_scheduler):
                lrs.scheduler.step(val_loss)

            if best_loss > val_loss:
                best_loss = val_loss
                counter = 0
                if save_path is not None:
                    torch.save(
                        {
                            "best_epoch": e + 1,
                            "model_state_dict": model.state_dict(),
                            "optimizer_state_dict": optimizer.state_dict(),
                            "loss": best_loss,
                        },
                        save_path,
                    )

            counter += 1

            if counter > patience:
                print(
                    "Early Stopping: Epoch {} \t Best Val Loss {}".format(e, best_loss)
                )
                break
        else:
            if save_path is not None:
                torch.save(
                    {
                        "best_epoch": e + 1,
                        "model_state_dict": model.state_dict(),
                        "optimizer_state_dict": optimizer.state_dict(),
                        "loss": epoch_loss,
                    },
                    save_path,
                )

    return model
