import torch
import torch.optim as optim
from tqdm.auto import tqdm
from src.utils.utility import get_lr_scheduler
from itertools import cycle


def calculate_test_loss(
    model, val_retain_loader, val_forget_loader, loss_fn, alpha, device, epoch
):
    model.eval()
    losses = []
    with torch.no_grad():
        for batch_idx, ((data_r, target_r), (data_f, target_f)) in tqdm(
            enumerate(zip(val_retain_loader, cycle(val_forget_loader))),
            total=len(val_retain_loader),
        ):
            data_r, target_r = data_r.to(device), target_r.to(device)
            data_f, target_f = data_f.to(device), target_f.to(device)
            output_r = model(data_r)
            output_f = model(data_f)
            loss_r = loss_fn(output_r.logits, target_r)
            loss_f = loss_fn(output_f.logits, target_f)
            loss = (alpha * loss_r) - ((1 - alpha) * loss_f)
            losses.append(loss)
        total_loss = torch.mean(torch.tensor(losses)).item()
        print("Val Epoch: {} \tLoss: {}".format(epoch, total_loss))
    return total_loss


def train_unlearn_step(
    model,
    retain_loader,
    forget_loader,
    optimizer,
    loss_fn,
    epoch,
    device,
    lr_scheduler=None,
    alpha=0.5,
):
    model.train()
    losses = []
    for batch_idx, ((data_r, target_r), (data_f, target_f)) in tqdm(
        enumerate(zip(retain_loader, cycle(forget_loader))), total=len(retain_loader)
    ):
        data_r, target_r = data_r.to(device), target_r.to(device)
        data_f, target_f = data_f.to(device), target_f.to(device)
        optimizer.zero_grad()
        output_r = model(data_r)
        output_f = model(data_f)
        loss_r = loss_fn(output_r.logits, target_r)
        loss_f = loss_fn(output_f.logits, target_f)
        loss = (alpha * loss_r) - ((1 - alpha) * loss_f)
        loss.backward()
        optimizer.step()
        if lr_scheduler is not None:
            lr_scheduler.step(epoch + batch_idx / len(retain_loader))
        losses.append(loss)
    total_loss = torch.mean(torch.tensor(losses)).item()
    print("Train Epoch: {} \tLoss: {}".format(epoch, total_loss))
    return model, total_loss


def train_unlearn(
    model,
    retain_loader,
    forget_loader,
    device,
    lr,
    alpha: float = 0.5,
    num_epochs: int = 10000,
    val_retain_loader=None,
    val_forget_loader=None,
    loss_fn=None,
    lr_scheduler: None | str = None,
    patience: int = 5,
    weight_decay: float = 1e-6,
    save_path: None | str = None,
):
    model.to(device)
    optimizer = optim.AdamW(model.parameters(), lr=lr, weight_decay=weight_decay)
    lrs = get_lr_scheduler(lr_scheduler=lr_scheduler, optimizer=optimizer)
    best_loss = torch.inf
    counter = 0
    for e in range(1, num_epochs + 1):
        model, epoch_loss = train_unlearn_step(
            model=model,
            retain_loader=retain_loader,
            forget_loader=forget_loader,
            optimizer=optimizer,
            loss_fn=loss_fn,
            epoch=e,
            device=device,
            lr_scheduler=(
                lrs.scheduler
                if lrs.scheduler is not None and lrs.is_batch_scheduler
                else None
            ),
            alpha=alpha,
        )

        if val_retain_loader is not None and val_forget_loader is not None:
            val_loss = calculate_test_loss(
                model, retain_loader, forget_loader, loss_fn, alpha, device, epoch=e
            )

            if lrs.scheduler is not None and (not lrs.is_batch_scheduler):
                lrs.scheduler.step(val_loss)

            if best_loss > val_loss:
                best_loss = val_loss
                counter = 0
                if save_path is not None:
                    torch.save(
                        {
                            "best_epoch": e + 1,
                            "model_state_dict": model.state_dict(),
                            "optimizer_state_dict": optimizer.state_dict(),
                            "loss": best_loss,
                        },
                        save_path,
                    )

            counter += 1

            if counter > patience:
                print(
                    "Early Stopping: Epoch {} \t Best Val Loss {}".format(e, best_loss)
                )
                break
        else:
            if save_path is not None:
                torch.save(
                    {
                        "best_epoch": e + 1,
                        "model_state_dict": model.state_dict(),
                        "optimizer_state_dict": optimizer.state_dict(),
                        "loss": epoch_loss,
                    },
                    save_path,
                )

    return model
