from typing import Dict, Optional, Union
from pathlib import Path
from threading import Lock
from .schema import DatabaseSchema
from .extractor import SchemaExtractor
from ..config import Config

class SchemaManager:
    _instance = None
    _lock = Lock()
    _schema_cache: Dict[str, DatabaseSchema] = {}
    
    def __new__(cls):
        if cls._instance is None:
            with cls._lock:
                if cls._instance is None:
                    cls._instance = super().__new__(cls)
        return cls._instance
    
    def get_schema(self, db_id: str, db_path: str) -> DatabaseSchema:

        if db_id not in self._schema_cache:
            with self._lock:
                if db_id not in self._schema_cache:
                    self._schema_cache[db_id] = SchemaExtractor.extract_from_db(db_path, db_id)
        return self._schema_cache[db_id]
    
    def get_formatted_enriched_schema(self, db_id: str, source: str) -> str:

        db_folder = f"{source}_{db_id}"
        db_file = f"{db_id}.sqlite"
        db_path = Config().database_dir / db_folder / db_file
        
        try:
            schema = self.get_schema(db_id, str(db_path))
            schema_dict = schema.to_dict()

            result = []

            result.append(f"Database: {schema_dict['database']}\n")

            for table in schema_dict['tables']:
                result.append(f"Table name: {table['table']}")

                result.append("Columns:")
                for col_name, col_info in table['columns'].items():
                    col_desc = []
                    if col_info['expanded_name']:
                        col_desc.append(f"Meaning: {col_info['expanded_name']}")
                    if col_info['description']:
                        col_desc.append(f"Description: {col_info['description']}")
                    if col_info['data_format']:
                        col_desc.append(f"Format: {col_info['data_format']}")
                    if col_info['value_description']:
                        col_desc.append(f"Value description: {col_info['value_description']}")
                    if col_info['value_examples']:
                        col_desc.append(f"Value examples: {', '.join(col_info['value_examples'])}")
                        
                    if col_desc:
                        result.append(f"  - {col_name} ({col_info['type']}): {' | '.join(col_desc)}")
                    else:
                        result.append(f"  - {col_name} ({col_info['type']})")

                if table['primary_keys']:
                    result.append(f"Primary key: {', '.join(table['primary_keys'])}")
                    
                result.append("")  

            if schema_dict.get('foreign_keys'):
                result.append("Foreign keys:")
                for fk in schema_dict['foreign_keys']:
                    result.append(
                        f"  {fk['table'][0]}.{fk['column'][0]} = "
                        f"{fk['table'][1]}.{fk['column'][1]}"
                    )
                    
            return "\n".join(result)
            
        except Exception as e:
            raise ValueError(f"Access schema failure: {str(e)}")
    
    def get_db_schema_by_id_source(self, db_id: str, source: str) -> Dict:

        db_folder = f"{source}_{db_id}"  
        db_file = f"{db_id}.sqlite"     
        db_path = str(Config().database_dir / db_folder / db_file)
        
        try:
            schema = self.get_schema(db_id, db_path)
            return schema.to_dict()
        except Exception as e:
            raise ValueError(f"Access schema failure: {str(e)}")
    
    def clear_cache(self):
        with self._lock:
            self._schema_cache.clear() 
    
    def format_enriched_db_schema(self, schema: Dict) -> str:

        result = []

        result.append(f"Database: {schema['database']}\n")

        for table in schema['tables']:
            result.append(f"Table name: {table['table']}")

            result.append("Columns:")
            for col_name, col_info in table['columns'].items():
                col_desc = []
                if col_info['expanded_name']:
                    col_desc.append(f"Meaning: {col_info['expanded_name']}")
                if col_info['description']:
                    col_desc.append(f"Description: {col_info['description']}")
                if col_info['data_format']:
                    col_desc.append(f"Format: {col_info['data_format']}")
                if col_info['value_description']:
                    col_desc.append(f"Value description: {col_info['value_description']}")
                if col_info['value_examples']:
                    col_desc.append(f"Value examples: {', '.join(col_info['value_examples'])}")
                    
                if col_desc:
                    result.append(f"  - {col_name} ({col_info['type']}): {' | '.join(col_desc)}")
                else:
                    result.append(f"  - {col_name} ({col_info['type']})")

            if table['primary_keys']:
                result.append(f"Primary key: {', '.join(table['primary_keys'])}")
                
            result.append("")  

        if schema.get('foreign_keys'):
            result.append("Foreign keys:")
            for fk in schema['foreign_keys']:
                result.append(
                    f"  {fk['table'][0]}.{fk['column'][0]} = "
                    f"{fk['table'][1]}.{fk['column'][1]}"
                )
                
        return "\n".join(result)
    
    def format_linked_schema(self, linked_schema: Dict) -> str:

        result = []

        for table in linked_schema["tables"]:
            table_name = table["table"]
            columns = table.get("columns", [])
            columns_info = table.get("columns_info", {})

            result.append(f"Table: {table_name}")

            if columns:
                result.append("Columns:")
                for col in columns:
                    col_info = columns_info.get(col, {})
                    col_desc = []
                    
                    col_type = col_info.get("type", "")
                    if col_type:
                        col_desc.append(f"Type: {col_type}")

                    if col_info.get("expanded_name"):
                        col_desc.append(f"Meaning: {col_info['expanded_name']}")
                    if col_info.get("description"):
                        col_desc.append(f"Description: {col_info['description']}")
                    if col_info.get("data_format"):
                        col_desc.append(f"Format: {col_info['data_format']}")
                    if col_info.get("value_description"):
                        col_desc.append(f"Value description: {col_info['value_description']}")
                    if col_info.get("value_examples"):
                        col_desc.append(f"Value examples: {', '.join(col_info['value_examples'])}")
                    
                    if col_desc:
                        result.append(f"  - {col}: {' | '.join(col_desc)}")
                    else:
                        result.append(f"  - {col}")

            if "primary_keys" in table:
                result.append(f"Primary key: {', '.join(table['primary_keys'])}")

            if "foreign_keys" in table:
                result.append("Foreign keys:")
                for fk in table["foreign_keys"]:
                    result.append(
                        f"  - {fk['column']} -> "
                        f"{fk['referenced_table']}.{fk['referenced_column']}"
                    )
            
            result.append("") 
            
        return "\n".join(result) 