from typing import Dict
from .base import SchemaLinkerBase
from ...core.schema.manager import SchemaManager

class SkipSchemaLinker(SchemaLinkerBase):

    def __init__(self, max_retries: int = 1):
        super().__init__("SkipSchemaLinker", max_retries)
        self.schema_manager = SchemaManager()
        
    async def link_schema(self, query: str, database_schema: Dict, query_id: str = None) -> str:
        try:
            
            linked_schema = {
                "tables": [
                    {
                        "table": table["table"],
                        "columns": list(table["columns"].keys()),
                        "columns_info": table["columns"]
                    }
                    for table in database_schema["tables"]
                ]
            }
            
           
            source = database_schema.get("source", "unknown")
            self.save_linked_schema_result(
                query_id=query_id,
                source=source,
                linked_schema={
                    "database": database_schema.get("database", ""),
                    "tables": linked_schema.get("tables", [])
                }
            )

            formatted_linked_schema = self.schema_manager.format_linked_schema(linked_schema)

            self.save_intermediate(
                input_data={
                    "query": query,
                },
                output_data={
                    "formatted_linked_schema": formatted_linked_schema
                },
                model_info={
                    "model": "none",
                    "input_tokens": 0,
                    "output_tokens": 0,
                    "total_tokens": 0
                },
                query_id=query_id
            )

            raw_output = """```json {"tables": [{"table": "<skip schema linking>", "columns": ["<skip schema linking>"]}]}```"""

            self.log_io(
                {
                    "query": query, 
                    "database_schema": database_schema, 
                    "formatted_schema": formatted_linked_schema
                }, 
                raw_output
            )
            return raw_output
            
        except Exception as e:
            self.logger.error(f"Skip schema linking failed: {str(e)}")
            raise 