#pragma once

#include "random.h"

#include <memory>
#include <ostream>
#include <vector>

long long multiply (long long a, long long b, long long prime);

long long fastExp (long long a, long long b, long long p);

long long modInv (long long a, long long p);

class Variable {
public:
	friend std::ostream& operator<< (std::ostream &os, const Variable &var);

private:
	virtual std::ostream& print (std::ostream& os) const = 0;
};

class Monomial {
public:
	Monomial ();

	void setCoefficient (int coefficient);

	void addFactor (const std::shared_ptr<Variable> &variable, unsigned exponent);

	void popFactor ();

	friend std::ostream& operator<< (std::ostream &os, const Monomial &monomial);

private:
	std::vector<std::pair<std::shared_ptr<Variable>, unsigned>> variables;
	int coefficient;
};

class Polynomial {
public:
	Polynomial ();

	void addMonomial (const Monomial &monomial);

	friend std::ostream& operator<< (std::ostream &os, const Polynomial &polynomial);

private:
	std::vector<Monomial> monomials;
};

class FASTP {
public:
	FASTP ();
	FASTP (long long p, long long q, long long r, long long s, long long t);

	long long getP () const;
	long long getQ () const;
	long long getR () const;
	long long getS () const;
	long long getT () const;

	void setP (long long p);
	void setQ (long long q);
	void setR (long long r);
	void setS (long long s);
	void setT (long long t);

	friend std::ostream& operator<< (std::ostream &os, const FASTP &fastp);

	friend long long getA (const FASTP &, const FASTP&, long long, long long, long long, long long, long long);
	friend long long getB (const FASTP &, const FASTP&, long long, long long, long long, long long, long long);

private:
	long long p, q, r, s, t;
};

class Fraction {
public:
	Fraction ();

	Polynomial& getP ();
	Polynomial& getQ ();

	void setP (const Polynomial &p);
	void setQ (const Polynomial &q);

	friend std::ostream& operator<< (std::ostream &os, const Fraction &fraction);

private:
	Polynomial p, q;
};

class Cycle {
public:
	Cycle (const std::vector<size_t> &nodes);

	void setTwoIdentifiable ();
	void setOneIdentifiableAZero (size_t reasonI, size_t reasonJ);
	void setOneIdentifiableDiscriminantZero ();
	void setOneIdentifiableOneOption (size_t reasonI, size_t reasonJ);

	bool isTwoIdentifiable () const;

	friend std::ostream& operator<< (std::ostream &os, const Cycle &cycle);

private:
	enum Identifiability {
		twoIdentifiable,
		oneIdentifiableAZero,
		oneIdentifiableDiscriminantZero,
		oneIdentifiableOneOption
	};

	std::vector<size_t> nodes;
	Identifiability identifiability;
	size_t reasonI, reasonJ;
};

class Path {
public:
	Path (const std::vector<size_t> &nodes);

	size_t getBack () const;

	friend std::ostream& operator<< (std::ostream &os, const Path &path);

private:
	std::vector<size_t> nodes;
};

class Lambda : public Variable {
public:
	Lambda (const std::shared_ptr<RandomTool> &randomTool, size_t x, size_t y);

	long long getValue () const;

private:
	size_t x, y;
	const long long value;

	std::ostream& print (std::ostream &os) const override;
};

class Omega : public Variable {
public:
	Omega (const std::shared_ptr<RandomTool> &randomTool, size_t x, size_t y);

	long long getValue () const;

private:
	size_t x, y;
	const long long value;

	std::ostream& print (std::ostream &os) const override;
};

class Sigma : public Variable {
public:
	Sigma (const std::shared_ptr<RandomTool> &randomTool, size_t x, size_t y);

	void setInTermsOfLambdaAndOmega (long long);

	long long getInTermsOfLambdaAndOmega () const;

private:
	const std::shared_ptr<RandomTool> randomTool;
	size_t x, y;
	long long inTermsOfLambdaAndOmega;

	std::ostream& print (std::ostream &os) const override;
};
