#include "identification.h"

#include <cassert>
#include <cstring>
#include <iostream>
#include <openssl/bn.h>
#include <random>

int main (int argc, char **argv) {
	long long seed = 42, prime = -1;
	Verbosity verbosity = normal;
	for (int i = 1; i < argc; i++) {
		if (std::strcmp(argv[i], "--seed") == 0) {
			assert(i + 1 < argc);
			seed = std::stoull(argv[i + 1]);
		}
		if (std::strcmp(argv[i], "--prime") == 0) {
			assert(i + 1 < argc);
			prime = std::stoll(argv[i + 1]);
		}
		if (std::strcmp(argv[i], "--verbose") == 0) {
			verbosity = verbose;
		}
		if (std::strcmp(argv[i], "--testing") == 0) {
			verbosity = testing;
		}
		if (std::strcmp(argv[i], "--help") == 0) {
			std::cout << "Usage: " << argv[0] << " [OPTION]..." << std::endl
					  << "Reads the input from stdin and writes to stdout" << std::endl << std::endl
					  << "Input format:" << std::endl
					  << "n\np_1 … p_{n-1}\nm\nu_0 v_0\n…\nu_{m-1} v_{m-1}\n" << std::endl
					  << "where n is the number of nodes, m is the number of edges," << std::endl
					  << "the nodes are numbered 0, …, n-1," << std::endl
					  << "p_1, …, p_{n-1} are the directed parents of 1, …, n-1" << std::endl
					  << "and {u_0, v_0}, …, {u_{m-1},v_{m-1}} are the bidirected edges" << std::endl << std::endl
					  << "Options:" << std::endl
					  << "   --seed SEED        Seeds the random with SEED. Default is 42." << std::endl
					  << "   --prime PRIME      Use PRIME as the prime. PRIME should be large enough, but" << std::endl
					  << "                      smaller than 2^60 to avoid overflows. Selects a random prime" << std::endl
					  << "                      of 59 bits by default." << std::endl
					  << "   --verbose          Output extra information such as the rank of every missing edge" << std::endl
					  << "   --testing          Only output n-1 integers i_1 … i_{n-1}, i_j∈{0,1,2}" << std::endl
					  << "                      0 means unidentifiable, 1 means 1-identifiable, 2 means 2-identifiable" << std::endl
					  << "   --help             Display this help and exit" << std::endl;
			return 0;
		}
	}

	/* Generate a random prime of 59 bits using OpenSSL
	 * The prime should be large enough, but smaller than 2^60 to avoid overflows
	 */
	if (prime == -1) {
		BIGNUM *prime_bignum;
		prime_bignum = BN_new();
		BN_generate_prime_ex(prime_bignum, 59, 0, NULL, NULL, NULL);
		char *prime_string = BN_bn2dec(prime_bignum);
		prime = std::atoll(prime_string);
		free(prime_string);
		BN_free(prime_bignum);
	}

	size_t n, m;
	assert(std::cin >> n);
	assert(n > 0);
	std::vector<size_t> directed(n - 1);
	for (size_t &par : directed) {
		assert(std::cin >> par);
	}
	assert(std::cin >> m);
	std::vector<std::pair<size_t, size_t>> bidirected(m);
	for (size_t i = 0; i < m; i++) {
		size_t u, v;
		assert(std::cin >> u >> v);
		assert(u != v && u < n && v < n);
		if (u > v) std::swap(u, v);
		bidirected[i] = {u, v};
	}
	auto result = identify(bidirected, directed, seed, prime, verbosity);
	if (verbosity == testing) {
		for (size_t i = 0; i < n-1; i++) {
			std::cout << static_cast<int>(result[i]) << " ";
		}
		std::cout << "\n";
	}
}
