from sqlalchemy import JSON, Integer, Sequence, String
from sqlalchemy.orm import Mapped, mapped_column, relationship

from synthetic_agents.database.entity.agent import Agent

# ApplicationAgent needs to be imported to correctly load the "agents" relationship. That's why we
# suppress the warning with F401.
from synthetic_agents.database.entity.application_agent import (  # noqa: F401
    ApplicationAgent,
)
from synthetic_agents.database.entity.base import Base


class Application(Base):
    """
    This class represents an application.
    """

    __tablename__ = "application"
    __table_args__ = {"comment": "This table stores application executions."}

    application_id: Mapped[int] = mapped_column(
        "application_id",
        Integer,
        Sequence("application_id_seq", start=1, increment=1),
        primary_key=True,
        comment="Sequential ID",
    )
    application_type: Mapped[str] = mapped_column(
        "application_type", String, comment="Type of the application."
    )

    attributes: Mapped[dict[str, dict]] = mapped_column(
        "attributes",
        JSON,
        comment="Attributes of the application in a JSON format",
    )

    __mapper_args__ = {"polymorphic_identity": "application", "polymorphic_on": "application_type"}

    agents: Mapped[list["Agent"]] = relationship(secondary="application_agent", lazy="selectin")
