from datetime import datetime
from typing import Optional


class TextMessage:
    """
    This class is a container for a chat message and the agent that produced it.
    """

    def __init__(
        self, creation_timestamp: datetime, content: str, agent_name: str, agent_type: str
    ):
        """
        Creates a text message object.

        :param creation_timestamp: timestamp defining when the message was created.
        :param content: message produced by the agent.
        :param agent_name: agent that produced the message.
        """
        self.creation_timestamp = creation_timestamp
        self.content = content
        self.agent_name = agent_name
        self.agent_type = agent_type

    @property
    def labeled_content(self) -> Optional[str]:
        """
        Gets content prepended with agent's name.

        :return: content prepended with agent's name.
        """
        if self.content is None:
            return None

        return f"{self.agent_name}: {self.content}"


class WorldState:
    """
    This class is a container for different information that represent the state of the world in
    a given moment.
    """

    def __init__(
        self,
        application_id: int,
        session_id: str,
        last_text_message: Optional[TextMessage] = None,
        current_date: datetime = datetime.now(),
    ):
        """
        Creates a world state instance.

        :param application_id: the ID of the application the agent is playing.
        :param session_id: the session ID of the application the agent is playing.
        :param last_text_message: last textual message delivered in an application.
        :param current_date: timestamp when the world state was obtained. Set this if you want to
            control when the conversation was happening.
        """

        self.application_id = application_id
        self.session_id = session_id
        self.current_date = current_date

        # For now, we only have texts, but this class makes the system prepared to future data like
        # audio and video, which can be added as part of the state of the world to be perceived by
        # the agents.
        self.last_text_message = last_text_message
