import os
import pandas as pd
import pydicom
import zipfile

from ... import const


DICOM_ZIP = const.DICOM_ZIP
XRAY_JSW_PATH = const.XRAY_JSW_PATH
KOA_RAW_DATA_DIR = const.KOA_RAW_DATA_DIR



def _convert_value(v):
    '''
    Converts DICOM values
    Based off of: https://github.com/cxr-eye-gaze/eye-gaze-dataset
    '''
    t = type(v)
    if t in (list, int, float):
        cv = v
    elif t == str:
        cv = v.replace(u"\u0000", "").strip()
    elif t == bytes:
        s = v.decode('utf-8', 'ignore')
        cv = s.replace(u"\u0000", "").strip()
    elif t == pydicom.valuerep.DSfloat:
        cv = float(v)
    elif t == pydicom.valuerep.IS:
        cv = int(v)
    elif t == pydicom.valuerep.PersonName:
        cv = str(v)
    else:
        cv = repr(v)
    return cv



def dicom_dataset_to_dict(dicom_header):
    '''
    Auxilary method to convert dicom header to python dictionary
    Based off of: https://github.com/cxr-eye-gaze/eye-gaze-dataset
    :param dicom_header: input dicome header
    :return: python dictionary of dicom header
    '''
    dicom_dict = {}
    repr(dicom_header)
    for dicom_value in dicom_header.values():
        if dicom_value.tag == (0x7fe0, 0x0010):
            continue
        if type(dicom_value.value) == pydicom.dataset.Dataset:
            dicom_dict[dicom_value.tag] = dicom_dataset_to_dict(dicom_value.value)
        else:
            if type(dicom_value.value) == pydicom.sequence.Sequence:
                for value in dicom_value.value:
                    for m in value:
                        if m.name != 'LUT Data':
                            dicom_dict[m.name] = _convert_value(m.value)
            else:
                v = _convert_value(dicom_value.value)
                dicom_dict[dicom_value.name] = v
    return dicom_dict



if __name__ == "__main__":

    xray_jsw_df = pd.read_csv(XRAY_JSW_PATH, delimiter='|')
    barcodes = xray_jsw_df['V01BARCDJD'].to_numpy()
    archive = zipfile.ZipFile(DICOM_ZIP)

    for entry in archive.infolist():
        print(entry.filename, ' : ', entry.file_size, ' : ')
        if not entry.is_dir() and '.jpg' not in entry.filename:
            entry.filename = os.path.basename(entry.filename)
            archive.extract(entry, path=KOA_RAW_DATA_DIR)

            dicom_img = pydicom.dcmread(os.path.join(KOA_RAW_DATA_DIR, entry.filename))
            dictionary = dicom_dataset_to_dict(dicom_img)

            if any(int(dictionary['Accession Number']) == int(barcode) for barcode in barcodes):
                new_name = os.path.join(KOA_RAW_DATA_DIR, dictionary['Accession Number'])
                os.rename(os.path.join(KOA_RAW_DATA_DIR, entry.filename), new_name)
            else:
                os.remove(os.path.join(KOA_RAW_DATA_DIR, entry.filename))