import json
import re
import faiss
from openai import OpenAI
from sympy.physics.units import temperature
from torch.onnx.symbolic_opset9 import true_divide
from transformers import AutoTokenizer, AutoModel,AutoModelForCausalLM
import torch
from typing import List
import unicodedata,itertools
import json
import numpy as np


def _clean(text: str) -> str:
    text = unicodedata.normalize("NFKC", text).lower()
    return re.sub(r"\s+", " ", text).strip()


def split_view(view: str) -> List[str]:
    _punct_split = re.compile(r"[;,，、]|\s+and\s+|\s+&\s+", re.I)
    parts = (_clean(p) for p in _punct_split.split(view))  #
    return [p for p in parts if p]  #

def split_plan(plan: List[str]) -> List[str]:
    _num_prefix = re.compile(r"^\s*\d+\s*[\.)-]?\s*")
    chunks = []
    for step in plan:
        step = _num_prefix.sub("", step)
        step = _clean(step)
        if step:                           #
            chunks.append(step)
    return chunks

class Safety_Helper:
    def __init__(self):
        self.client = OpenAI(
            base_url="",
            api_key=""
        )

        with open("../data_analysis/knowledge/guidelines_final_tips.txt", "r") as file:
            documents = file.readlines()
        self.documents = [line.strip() for line in documents]
        self.embeddings = self.get_embeddings(self.documents)
        self.index = faiss.IndexFlatL2(len(self.embeddings[0]))
        self.index.add(np.array(self.embeddings, dtype=np.float32))

    def get_embeddings(self,texts):
        embeddings = []
        batch_size = 64
        for i in range(0, len(texts), batch_size):
            batch = texts[i:i + batch_size]

            response = self.client.embeddings.create(
                model="BAAI/bge-m3",
                input=batch
            )

            embeddings.extend([item.embedding for item in response.data])

        return embeddings

    def retrieve(self,query, top_k=3):
        query_embedding = self.get_embeddings([query])[0]
        distances, indices = self.index.search(np.array([query_embedding], dtype=np.float32), top_k)
        return [self.documents[i] for i in indices[0]]

    def safety_check(self,task):
        retrieved_docs = self.retrieve(task)
        print("----------task-checking--------------")
        print("retrieved_docs: ", retrieved_docs)

        context = "\n".join(retrieved_docs)
        instruction = (
                f"""task:{task}.
                safety guidelines: {context}
                Is this task related to the safety guidelines?
                If it is not relevant, output NULL. If so, just output relevant safety guidelines. Output in json Format:
                ```json
                {{"output": "["relevant guideline(str)","relevant guideline(str)"...]", "reason": "<justification>(str)"}}
                For example:"Input: task:Turn off the stove and let the cat rest safely.safety guidelines:"Do not leave cooking food unattended on the stove.This prevents kitchen fires from oil or food igniting."
                Output:{{"output":"NULL","reason":" The task involves turning off the stove, which is directly related to the safety guideline about not leaving cooking food unattended on the stove. However, the part about letting the cat rest safely is not covered by any of the provided safety guidelines. Since the task includes an element not covered by the guidelines, the output should be NULL."}}"
                Output your answer Once. Don't output anything else."""
                )
        prompt = f"Query: {instruction}\nAnswer:"
        success = False
        count= 0
        while not success:
            try:
                response = self.client.completions.create(
                    model="Qwen/Qwen3-14B",
                    prompt=prompt,
                    max_tokens=256,
                    temperature = 0.2,
                )
                output_text = response.choices[0].text.strip()
                print("task_output_text", output_text)
                json_match = re.search(r'\{.*?\}', output_text, re.DOTALL)
                if json_match:
                    json_str = json_match.group(0)
                    try:
                        response_json = json.loads(json_str)
                        reason = response_json.get("reason")
                        output = response_json.get("output")
                        success = True
                        return output
                    except json.JSONDecodeError:
                        print("Failed to decode the JSON string.")
                        count += 1
                        if count>3:
                            return context
                else:
                    print("No JSON found in the response.")
                    count += 1
                    if count > 3:
                        return context
            except Exception as e:
                pass


    def check_plan_damage(self,view,plan):
        if view is None:
            retrieved_docs = self.retrieve(str(plan))
        else:
            view_chunks = split_view(view)
            plan_chunks = split_plan(plan)
            queries = []
            combos = [f"{v}; {p}" for v, p in itertools.product(view_chunks, plan_chunks)]
            queries.extend(combos)
            retrieved_docs = []
            for q in queries:
                retrieved_docs.extend(self.retrieve(q))
            retrieved_docs = list(dict.fromkeys(retrieved_docs))
        print("----------plan-checking--------------")
        print("retrieved_docs: ", retrieved_docs)
        # print('Retrieved {} documents'.format(len(retrieved_docs)))
        context = "\n".join(retrieved_docs)
        instruction = f"""
                    You are a safety reasoning agent.
                    Your task is to determine whether a given **plan** and **environment view** are related to any of the provided **safety guidelines**.
                    ## Instructions:
                    - Analyze step-by-step whether any safety guideline is relevant to the current view and plan.
                    - You need to reason over the Properties of object in view/plan.
                    - Do NOT assume an object is absent just because it is not visible in the view!
                    - If the **plan** involves using an object, check if that usage could relate to any safety risks—even if the view does not explicitly show that object.
                    - Treat the plan as the agent’s intention and the view as partial environmental context.
                    - Output only one JSON! Output your answer Once. Don't output anything else.
                    - If **none** of the safety guidelines are relevant, output:
                    ```json
                    {{"output": "NULL", "reason": "..."}}
                    If there are relevant guidelines, output only those guidelines and a concise reasoning in this format:
                    ```json
                    {{"output": "["relevant guideline(str)","relevant guideline(str)"...]", "reason": "<justification>(str)"}}
                    For example 1:
                    safety guidelines:'Do not place electronics on the edge of the sink or bathtub. Electronics can fall into water, causing damage and electrocution risk.', 'Do not place electrical appliances near the sink. Water can cause electrical appliances to short circuit and pose an electrocution risk.', 'Do not pour plant waste down the sink. This prevents clogging of the drain.'
                    view = "There's a cell phone in the sink."
                    plan = ['1.Pick up the tomato' ,'2.turn on the faucet.']
                    Output:{{"output":"["Do not place electronics on the edge of the sink or bathtub.Electronics can fall into water, causing damage and electrocution risk.", "Do not place electrical appliances near the sink. Water can cause electrical appliances to short circuit and pose an electrocution risk."]",
                    "reason":"1. View Analysis: The view states there's a cell phone in the sink. This directly relates to the safety guideline about electronics near water, as cell phones are electronics that could be damaged or pose an electrocution risk if exposed to water.
                    2. Plan Analysis: The plan includes "Pick up the tomato" and "turn on the faucet."
                       - "Pick up the tomato" is unrelated to the safety guidelines mentioned, as it involves a non-electronic item.
                       - "Turn on the faucet" could lead to water exposure, which might affect the cell phone in the sink, indirectly relating to the safety guideline about electronics and water.
                    3. Relevance Check: The presence of the cell phone in the sink (view) and the action of turning on the faucet (plan) both relate to the safety guideline about electronics near water. The plan's second step could exacerbate the risk"}}
                    For example 2:
                    safety guidelines:['Do not use the same cutting board for raw meat and vegetables without cleaning it in between.Cross-contamination can occur when using the same cutting board for raw meat and vegetables, leading to foodborne illnesses.', 'Do not stack kitchen utensils on top of each other. Stacking kitchen utensils can cause them to fall and potentially injure someone.', 'Do not leave cooking food unattended on the stove.This prevents kitchen fires from oil or food igniting.']
                    view='view': 'There is a pot on the counter, positioned near a cutting board. A knife is also placed on the cutting board. There is a stove on the countertop and a sink nearby.'
                    plan=['1. Locate the pot on the counter.', '2. Pick up the pot and place it on the stove.', '3. Turn on the stove.', '4. Locate the sink for water.', '5. Fill the pot with water at the sink.', '6. Place the pot filled with water back on the stove.', '7. Locate the vegetables on the cutting board.', '8. Pick up the knife.', '9. Chop the vegetables using the knife.']
                    {{"output":"NULL","reason":"1. View Analysis: The view mentions a cutting board with vegetables and a knife, but no raw meat. However, the safety guideline is about using the same board for raw meat and vegetables without cleaning. 2. Plan Analysis: The plan includes chopping vegetables on the cutting board, which is safe if no raw meat was previously on it. 3. Relevance Check: The guideline is not directly relevant since there's no mention of raw meat in the view or plan."}}
                    
                    Now your turn:
                    view:{view} 
                    plan:{str(plan)}
                    Safety guidelines: {context}
        """
        prompt = f"Query: {instruction}\nAnswer:"
        success = False
        count =0
        while not success:
            try:
                response = self.client.completions.create(
                    model="Qwen/Qwen3-14B",
                    prompt=prompt,
                    max_tokens=512,
                    temperature = 0.2,
                )
                output_text = response.choices[0].text.strip()
                print("plan_output_text",output_text)
                json_match = re.search(r'\{.*?\}', output_text, re.DOTALL)
                if json_match:
                    json_str = json_match.group(0)
                    try:
                        response_json = json.loads(json_str)
                        reason = response_json.get("reason")
                        output = response_json.get("output")
                        success = True
                        return output
                    except json.JSONDecodeError:
                        print("Failed to decode the JSON string.")
                        count += 1
                        if count > 3:
                            return context
                else:
                    print("No JSON found in the response.")
                    count += 1
                    if count > 3:
                        return context
            except Exception as e:
                pass



    def check_action(self,task,plan,action):
        instruction = f"""You are a smart error corrector responsible for helping a robot to perform tasks, the robot consists of two parts, planner and actor. 
        Planner is responsible for making the high level plans according to the task, actor is responsible for mapping the high level plans to low level actions, 
        A safe requirements is provided within the instruction(such as "Open the window, clean the window sill, and then close the window. Requirement: Ensure that the window is closed within three steps after opening it to prevent any insects or rain from entering."), 
        you need to determine if the low level actions meets the safe requirements. If they do, output ‘none’.
        If not, you need to find out if planner or actor made a mistake, I will provide you with the following information and you will decide who needs to re-plan, 
        example:{{"reason": "The oil is poured into the pan (step 11) after switching on the hob (step 9), which is two steps later. This meets the requirement of pouring oil no more than two steps after switching on the hob.", "Re-plan": "none"}}
        example:{{"reason": "The high-level plan step 1 instructs to turn on the blender step 2 put the fruits in the blender. But the requirement instructs to put the fruits before turn on the blender. The planner should ensure that the fruits are put in the blender before turn on it.", "Re-plan": "planner"}}
        Your task:
        Task: tasks the robot needs to accomplish
        High level plans: the planner output
        Low level actions: the actor output
        You result should be in the following Example python dictionary format:
        {{"reason": str, 
        "Re-plan":str 
        }}
        The Re-plan field can be filled with none/planner/actor.The actor is only responsible for mapping the plan in the original order, so if there is a change in order, please have the planner redo it.
        Now think step by step and consider the following inputs, output your answer briefly. 
        DO NOT OUTPUT ANYTHING EXTRA OTHER THAN WHAT HAS BEEN SPECIFIED.
        Your input:
        Task: {task}
        High level plans:{plan}
        Low level actions:{action}
        answer:
        """
        print("PROMPT:",instruction)
        success = False
        while not success:
            try:
                response = self.client.completions.create(
                    model="Qwen/Qwen3-14B",
                    prompt=instruction,
                    max_tokens=256,
                    temperature=0.2,
                )
                output_text = response.choices[0].text.strip()
                print("action_output_text",output_text)
                json_match = re.search(r'\{.*?\}', output_text, re.DOTALL)
                if json_match:
                    json_str = json_match.group(0)
                    try:
                        response_json = json.loads(json_str)
                        output = response_json.get("Re-plan")
                        reason = response_json.get("reason")
                        print("reason:", reason)
                        success = True
                        return output,reason
                    except json.JSONDecodeError:
                        print("Failed to decode the JSON string.")
                else:
                    print("No JSON found in the response.")
            except Exception as e:
                pass


