import json
import os
import requests

import random
import time
import tiktoken
from xopen import xopen
from copy import deepcopy
from tqdm import tqdm
from collections import defaultdict
import numpy as np

from llmlingua import PromptCompressor
from datasets import load_dataset

encoding = tiktoken.encoding_for_model("gpt-3.5-turbo")

TASKS = ["narrativeqa", "qasper", "multifieldqa_en", "hotpotqa", "2wikimqa", "musique", \
            "gov_report", "qmsum", "multi_news", "trec", "triviaqa", "samsum", \
            "passage_count", "passage_retrieval_en", "lcc", "repobench-p"]
O_QUOTA = {
    "narrativeqa": 512,
    "qasper": 128,
    "multifieldqa_en": 64,
    "hotpotqa": 32,
    "2wikimqa": 32,
    "musique": 32,
    "gov_report": 512,
    "qmsum": 512,
    "multi_news": 512,
    "trec": 64,
    "triviaqa": 32,
    "samsum": 128,
    "passage_count": 32,
    "passage_retrieval_en": 32,
    "lcc": 64,
    "repobench-p": 64
}
INSTRUCTIONS = {
    "narrativeqa": "You are given a story, which can be either a novel or a movie script, and a question. Answer the question asconcisely as you can, using a single phrase if possible. Do not provide any explanation.\n\nStory: {context}\n\n",
    "qasper": "You are given a scientific article and a question. Answer the question as concisely as you can, using a single phrase or sentence if possible. If the question cannot be answered based on the information in the article, write \"unanswerable\". If the question is a yes/no question, answer \"yes\", \"no\", or \"unanswerable\". Do not provide any explanation.\n\nArticle: {context}\n\n",
    "multifieldqa_en": "Read the following text and answer briefly.\n\n{context}\n\n",
    "hotpotqa": "Answer the question based on the given passages. Only give me the answer and do not output any other words.\n\nThe following are given passages.\n{context}\n\n",
    "2wikimqa": "Answer the question based on the given passages. Only give me the answer and do not output any other words.\n\nThe following are given passages.\n{context}\n\n",
    "musique": "Answer the question based on the given passages. Only give me the answer and do not output any other words.\n\nThe following are given passages.\n{context}\n\n",
    "gov_report": "You are given a report by a government agency. Write a one-page summary of the report.\n\nReport:\n{context}\n\n",
    "qmsum": "You are given a meeting transcript and a query containing a question or instruction. Answer the query in one or more sentences.\n\nTranscript:\n{context}\n\n",
    "multi_news": "You are given several news passages. Write a one-page summary of all news. \n\nNews:\n{context}\n\n",
    "trec": "Please determine the type of the question below. Here are some examples of questions.\n\n{context}\n",
    "triviaqa": "Answer the question based on the given passage. Only give me the answer and do not output any other words. The following are some examples.\n\n{context}\n\n",
    "samsum": "Summarize the dialogue into a few short sentences. The following are some examples.\n\n{context}\n\n",
    "passage_count": "There are some paragraphs below sourced from Wikipedia. Some of them may be duplicates. Please carefully read these paragraphs and determine how many unique paragraphs there are after removing duplicates. In other words, how many non-repeating paragraphs are there in total?\n\n{context}\n\n",
    "passage_retrieval_en": "Here are 30 paragraphs from Wikipedia, along with an abstract. Please determine which paragraph the abstract is from.\n\n{context}\n\n",
    "lcc": "Please complete the code given below. \n{context}",
    "repobench-p": "Please complete the code given below. \n{context}"
}
QUESTIONS = {
    "narrativeqa": "Now, answer the question based on the story asconcisely as you can, using a single phrase if possible. Do not provide any explanation.\n\nQuestion: {input}\n\nAnswer:",
    "qasper": "Answer the question based on the above article as concisely as you can, using a single phrase or sentence if possible. If the question cannot be answered based on the information in the article, write \"unanswerable\". If the question is a yes/no question, answer \"yes\", \"no\", or \"unanswerable\". Do not provide any explanation.\n\nQuestion: {input}\n\nAnswer:",
    "multifieldqa_en": "Now, answer the following question based on the above text, only give me the answer and do not output any other words.\n\nQuestion: {input}\nAnswer:",
    "hotpotqa": "Answer the question based on the given passages. Only give me the answer and do not output any other words.\n\nQuestion: {input}\nAnswer:",
    "2wikimqa": "Answer the question based on the given passages. Only give me the answer and do not output any other words.\n\nQuestion: {input}\nAnswer:",
    "musique": "Answer the question based on the given passages. Only give me the answer and do not output any other words.\n\nQuestion: {input}\nAnswer:",
    "gov_report": "Now, write a one-page summary of the report.\n\nSummary:",
    "qmsum": "Now, answer the query based on the above meeting transcript in one or more sentences.\n\nQuery: {input}\nAnswer:",
    "multi_news": "Now, write a one-page summary of all the news.\n\nSummary:",
    "trec": "{input}",
    "triviaqa": "{input}",
    "samsum": "{input}",
    "passage_count": "Please enter the final count of unique paragraphs after removing duplicates. The output format should only contain the number, such as 1, 2, 3, and so on.\n\nThe final answer is: ",
    "passage_retrieval_en": "The following is an abstract.\n\n{input}\n\nPlease enter the number of the paragraph that the abstract is from. The answer format must be like \"Paragraph 1\", \"Paragraph 2\", etc.\n\nThe answer is: ",
    "lcc": "Next line of code:\n",
    "repobench-p": "{input}Next line of code:\n"
}

import openai

openai.api_key = "sk-"

def get_results(predict_path: str):
    import re
    import string

    from fuzzywuzzy import fuzz
    import difflib

    from collections import Counter
    from rouge import Rouge

    def normalize_answer(s):
        """Lower text and remove punctuation, articles and extra whitespace."""
        def remove_articles(text):
            return re.sub(r"\b(a|an|the)\b", " ", text)
        def white_space_fix(text):
            return " ".join(text.split())
        def remove_punc(text):
            exclude = set(string.punctuation)
            return "".join(ch for ch in text if ch not in exclude)
        def lower(text):
            return text.lower()
        return white_space_fix(remove_articles(remove_punc(lower(s))))

    def count_score(prediction, ground_truth, **kwargs):
        numbers = re.findall(r"\d+", prediction)
        right_num = 0
        for number in numbers:
            if str(number) == str(ground_truth):
                right_num += 1
        final_score = 0.0 if len(numbers) == 0 else right_num / len(numbers)
        return float(final_score)

    def retrieval_score(prediction, ground_truth, **kwargs):
        pattern = r'Paragraph (\d+)'
        matches = re.findall(pattern, ground_truth)
        ground_truth_id = matches[0]
        numbers = re.findall(r"\d+", prediction)
        right_num = 0
        for number in numbers:
            if str(number) == str(ground_truth_id):
                right_num += 1
        final_score = 0.0 if len(numbers) == 0 else right_num / len(numbers)
        return float(final_score)

    def code_sim_score(prediction, ground_truth, **kwargs):
        all_lines = prediction.lstrip('\n').split('\n')
        prediction = ""
        for line in all_lines:
            if ('`' not in line) and ('#' not in line) and ('//' not in line):
                prediction = line
                break
        return (fuzz.ratio(prediction, ground_truth) / 100)

    def classification_score(prediction, ground_truth, **kwargs):
        em_match_list = []
        all_classes = kwargs["all_classes"]
        for class_name in all_classes:
            if class_name in prediction:
                em_match_list.append(class_name)
        for match_term in em_match_list:
            if match_term in ground_truth and match_term != ground_truth:
                em_match_list.remove(match_term)
        if em_match_list != 0:
            if ground_truth in em_match_list:
                score = (1.0 / len(em_match_list))
            else:
                score = 0.0
        else:
            best_match = None
            highest_similarity = 0
            for string in all_classes:
                similarity = difflib.SequenceMatcher(None, string, prediction).ratio()
                if similarity > highest_similarity:
                    highest_similarity = similarity
                    best_match = string
            score = float(best_match == ground_truth)
        return score
        
    def rouge_score(prediction, ground_truth, **kwargs):
        rouge = Rouge()
        # print(prediction, ground_truth)
        scores = rouge.get_scores([prediction], [ground_truth], avg=True)
        return scores["rouge-l"]["f"]

    def f1_score(prediction, ground_truth, **kwargs):
        common = Counter(prediction) & Counter(ground_truth)
        num_same = sum(common.values())
        if num_same == 0:
            return 0
        precision = 1.0 * num_same / len(prediction)
        recall = 1.0 * num_same / len(ground_truth)
        f1 = (2 * precision * recall) / (precision + recall)
        return f1

    def qa_f1_score(prediction, ground_truth, **kwargs):
        normalized_prediction = normalize_answer(prediction)
        normalized_ground_truth = normalize_answer(ground_truth)

        prediction_tokens = normalized_prediction.split()
        ground_truth_tokens = normalized_ground_truth.split()
        return f1_score(prediction_tokens, ground_truth_tokens)

    def scorer(dataset, predictions, answers, all_classes):
        total_score = 0.
        for (prediction, ground_truths) in zip(predictions, answers):
            score = 0.
            if dataset in ["trec", "triviaqa", "samsum", "lsht"]:
                prediction = prediction.lstrip('\n').split('\n')[0]
            for ground_truth in ground_truths[0]:
                score = max(score, dataset2metric[dataset](prediction, ground_truth, all_classes=all_classes))
            # print(score)
            total_score += score
        return round(100 * total_score / len(predictions), 2)

    dataset2metric = {
        "narrativeqa": qa_f1_score,
        "qasper": qa_f1_score,
        "multifieldqa_en": qa_f1_score,
        "hotpotqa": qa_f1_score,
        "2wikimqa": qa_f1_score,
        "musique": qa_f1_score,
        "gov_report": rouge_score,
        "qmsum": rouge_score,
        "multi_news": rouge_score,
        "trec": classification_score,
        "triviaqa": qa_f1_score,
        "samsum": rouge_score,
        "lsht": classification_score,
        "passage_retrieval_en": retrieval_score,
        "passage_count": count_score,
        "lcc": code_sim_score,
        "repobench-p": code_sim_score,
    }
    preds = json.load(open(predict_path))
    preds_g, refers_g, classes_g = defaultdict(list), defaultdict(list), defaultdict(list)
    for v in preds.values():
        task, refer, pred, all_classes = [v[k] for k in ["task", "reference", "pred", "all_classes"]]
        pred = pred.replace("<|im_end|>", "").replace("\end{document}", "").strip()
        if task in [
            "narrativeqa", "qasper", "multifieldqa_en", "hotpotqa", "2wikimqa", "musique",
            "triviaqa", "trec", "samsum", "passage_count", "passage_retrieval_en"
        ]:
            pred = pred.split("\n")[0]
        if not pred:
            pred = "A."
        preds_g[task].append(pred)
        refers_g[task].append([refer])
        classes_g[task].append(all_classes)
    long_bench = []
    for task in TASKS:
        if task not in preds_g:
            long_bench.append(0)
            continue
        p, r, all_c = preds_g[task], refers_g[task], classes_g[task]
        results = scorer(task, p, r, all_c[0])
        print(task, len(p), results)
        long_bench.append(results)
    print(",".join([f"{ii:.2f}" for ii in long_bench]))

def get_token_length(text: str):
    return len(encoding.encode(text))

def get_dataset_retrieval(path: str, doc_num: int = 20, d_idx: int = 0):
    def get_distance_bm25(corpus, query):
        from rank_bm25 import BM25Okapi
        tokenized_corpus = [doc.split(" ") for doc in corpus]
        bm25 = BM25Okapi(tokenized_corpus)
        tokenized_query = query.split(" ")
        doc_scores = bm25.get_scores(tokenized_query)
        idx = [ii for ii in (-doc_scores).argsort()].index(d_idx)
        return idx
    def get_distance_gzip(corpus, query):
        def get_score(x, y):
            cx, cy = len(gzip.compress(x.encode())), len(gzip.compress(y.encode()))
            cxy = len(gzip.compress(f"{x} {y}".encode()))
            return (cxy - min(cx, cy)) / max(cx, cy)
        import gzip
        doc_scores = [get_score(doc, query) for doc in corpus]
        return np.argsort(doc_scores)
        idx = [ii for ii in np.argsort(doc_scores)].index(d_idx)
        return idx

    def get_distance_sentbert(corpus, query):
        doc_embeds = model.encode(corpus)
        query = model.encode(query)
        doc_scores = -util.dot_score(doc_embeds, query).cpu().numpy().reshape(-1)
        idx = [ii for ii in np.argsort(doc_scores)].index(d_idx)
        return idx

    def get_distance_sentbert_bge(corpus, query):
        instruction = "Represent this sentence for searching relevant passages:"
        doc_embeds = model.encode([i for i in corpus], normalize_embeddings=True)
        query = model.encode(query, normalize_embeddings=True)
        doc_scores = -util.dot_score(doc_embeds, query).cpu().numpy().reshape(-1)
        idx = [ii for ii in np.argsort(doc_scores)].index(d_idx)
        return idx

    def get_distance_openai(corpus, query):
        import openai
        openai.api_key = ""
        openai.api_base = ""
        openai.api_version = '2023-05-15'
        deployment_name="embed"
        def get_embed(text):
            try:
                return openai.Embedding.create(input=[text.replace("\n", " ")], engine=deployment_name)['data'][0]['embedding']
            except openai.error.RateLimitError:
                time.sleep(5)
                return get_embed(text)
        doc_embeds = [get_embed(i) for i in corpus]
        query = get_embed(query)
        doc_scores = -util.dot_score(doc_embeds, query).cpu().numpy().reshape(-1)
        idx = [ii for ii in np.argsort(doc_scores)].index(d_idx)
        return idx
    def ensemble(rank1, rank2):
        c = defaultdict(int)
        for idx, ii in enumerate(rank1):
            c[ii] += idx
        for idx, ii in enumerate(rank2):
            c[ii] += idx
        return [ii for ii, _ in sorted(c.items(), key=lambda x: x[1])].index(d_idx)

    from sentence_transformers import SentenceTransformer, util
    model = SentenceTransformer('multi-qa-mpnet-base-dot-v1')
    # model = SentenceTransformer('all-mpnet-base-v2')
    # model = SentenceTransformer('BAAI/bge-large-en')

    res = []
    with xopen(path) as f:
        for ii, jj in tqdm(enumerate(f), total=2655):
            if ii < len(res):
                continue
            input_example = json.loads(jj)
            question = input_example["question"]
            documents = []
            for ctx in deepcopy(input_example["ctxs"]):
                documents.append(Document.from_dict(ctx))

            prompt = get_qa_prompt(
                question,
                documents,
                mention_random_ordering=False,
                query_aware_contextualization=False,
            )

            c = prompt.split("\n\n")
            instruction, question = c[0], c[-1]
            demonstration = "\n".join(c[1:-1])
            corpus = demonstration.split("\n")

            # idx = get_distance_openai(corpus, question)
            idx = get_distance_sentbert(corpus, question)
            res.append(idx)
    for idx in [1, 2, 5, 10]:
        print("R@{},{:.2f}".format(idx, len([ii for ii in res if ii < idx]) / len(res) * 100))

def get_windows_prompt(task, context, inputs, return_all: bool = False):
    max_len = O_QUOTA[task]
    question = QUESTIONS[task].format(input=inputs)
    prompt = INSTRUCTIONS[task].format(context=context)
    if len(encoding.encode(prompt)) + len(encoding.encode(question)) + max_len > 8000 and not return_all:
        delta = len(encoding.encode(prompt)) + len(encoding.encode(question)) + max_len - 8000
        context = encoding.decode(encoding.encode(context)[delta:])
        prompt = INSTRUCTIONS[task].format(context=context)
    return prompt + question

def get_dataset():
    res = []
    for task in TASKS:
        dataset = load_dataset('THUDM/LongBench', task, split='test')
        for idx, data in enumerate(dataset):
            context, inputs, all_classes, answers = [data[i] for i in ["context", "input", "all_classes", "answers"]]
            prompt = get_windows_prompt(task, context, inputs, True)
            res.append({"id": idx, "task": task, "prompt": prompt, "answers": answers, "all_classes": all_classes})
    json.dump(res, open("prompt/long_bench/full_all.json", "w"))


def get_dataset_baselies_summarization(num: int = 1):
    res = []
    dataset = load_dataset('THUDM/LongBench', task, split='test')
    for idx, data in tqdm(enumerate(dataset), total=dataset.num_rows):
        context, inputs, all_classes, answers = [data[i] for i in ["context", "input", "all_classes", "answers"]]
        # docs = split_context(context)
        # need = [i for i, j in ranker[idx][:num]]
        # context = "".join([docs[i] for i in need])
        # context = "".join([ii for ii in docs[-num:]])
        context = encoding.decode(encoding.encode(context)[:num])
        # r = llm_lingua.get_rank_results(docs, QUESTIONS[task].format(input=inputs), "longllmlingua", "after", [0] * len(docs))
        # ranker.append(r)

        prompt = get_windows_prompt(task, context, inputs)
        res.append({"id": idx, "task": task, "prompt": prompt, "answers": answers, "all_classes": all_classes})
    json.dump(res, open(f"prompt/long_bench/keep_first_{num}_gov_report.json", "w"))

def get_dataset_baselines1(num: int = 1):
    def split_context(context):
        res = []
        names = re.findall("([\w0-9_\-]*/)?([\w0-9_\-]*\.[\w0-9_\-]+)\n", context)
        pre = 0
        for name in names[1:]:
            name = name[0] + name[1]
            idx = context.find(name, pre + len(name) + 10)
            while idx > pre and context[idx] != "\n":
                idx -= 1
            idx += 1
            if context[pre] in "\t ":
                res[-1] += context[pre: idx]
            else:
                res.append(context[pre: idx])
            pre = idx
        res.append(context[pre:])
        return res
    res = []
    for task in TASKS:
        dataset = load_dataset('THUDM/LongBench', task, split='test')
        for idx, data in tqdm(enumerate(dataset), total=dataset.num_rows):
            context, inputs, all_classes, answers = [data[i] for i in ["context", "input", "all_classes", "answers"]]
            docs = split_context(context)
            # need = [i for i, j in ranker[idx][:num]]
            # context = "".join([docs[i] for i in need])
            context = "".join([ii for ii in docs[-num:]])
            # r = llm_lingua.get_rank_results(docs, QUESTIONS[task].format(input=inputs), "longllmlingua", "after", [0] * len(docs))
            # ranker.append(r)

            prompt = get_windows_prompt(task, context, inputs)
            res.append({"id": idx, "task": task, "prompt": prompt, "answers": answers, "all_classes": all_classes})
    json.dump(res, open(f"prompt/long_bench/keep_doc_last_{num}.json", "w"))

def get_dataset_baseline():
    res = []
    n = 0
    for task in TASKS:
        dataset = load_dataset('THUDM/LongBench', task, split='test')
        for idx, data in tqdm(enumerate(dataset), total=dataset.num_rows):
            if len(res) > n:
                n += 1
                continue
            context, inputs, all_classes, answers = [data[i] for i in ["context", "input", "all_classes", "answers"]]
            # context = encoding.decode(encoding.encode(context)[:8096])
            # context = INSTRUCTIONS[task].format(context=context)
            question = QUESTIONS[task].format(input=inputs)
            docs = get_doc(context, task)

            context = llm_lingua(context, "", question, target_token=2048, use_sentence_level_filter=True, condition_in_question="none", reorder_demonstrations=False, dynamic_demonstration_compression_ratio=0, condition_compare=False, concate_question=False, demonstration_budget="+0", use_demonstrate_level_filter=True, use_token_level_filter=False, rank_method="bm25", token_budget_ratio=1.0)

            prompt = get_windows_prompt(task, context["compressed_prompt"], inputs)
            res.append({"id": idx, "task": task, "prompt": prompt, "answers": answers, "all_classes": all_classes})
            n += 1

    json.dump(res, open(f"prompt/long_bench/bm25_2k.json", "w"))

    res = []
    n = 0
    for task in TASKS:
        dataset = load_dataset('THUDM/LongBench', task, split='test')
        for idx, data in tqdm(enumerate(dataset), total=dataset.num_rows):
            if len(res) > n:
                n += 1
                continue
            context, inputs, all_classes, answers = [data[i] for i in ["context", "input", "all_classes", "answers"]]
            # context = encoding.decode(encoding.encode(context)[:8096])
            # context = INSTRUCTIONS[task].format(context=context)
            question = QUESTIONS[task].format(input=inputs)
            # docs = get_doc(context, task)

            context = llm_lingua(context, "", question, target_token=2048, use_sentence_level_filter=True, condition_in_question="after", reorder_demonstrations=False, dynamic_demonstration_compression_ratio=0, condition_compare=False, concate_question=False, demonstration_budget="+0", use_demonstrate_level_filter=True, use_token_level_filter=False, rank_method="longllmlingua", token_budget_ratio=1.0)

            prompt = get_windows_prompt(task, context["compressed_prompt"], inputs)
            res.append({"id": idx, "task": task, "prompt": prompt, "answers": answers, "all_classes": all_classes})
            n += 1

    json.dump(res, open(f"prompt/long_bench/longllmlingua_2k_4.json", "w"))

    res = []
    n = 0
    for task in TASKS:
        dataset = load_dataset('THUDM/LongBench', task, split='test')
        for idx, data in tqdm(enumerate(dataset), total=dataset.num_rows):
            if len(res) > n:
                n += 1
                continue
            context, inputs, all_classes, answers = [data[i] for i in ["context", "input", "all_classes", "answers"]]
            # context = encoding.decode(encoding.encode(context)[:8096])
            # context = INSTRUCTIONS[task].format(context=context)
            question = QUESTIONS[task].format(input=inputs)
            # docs = get_doc(context, task)

            context = llm_lingua(context, "", question, target_token=2048, use_sentence_level_filter=True, condition_in_question="none", reorder_demonstrations=False, dynamic_demonstration_compression_ratio=0, condition_compare=False, concate_question=False, demonstration_budget="+0", use_demonstrate_level_filter=True, use_token_level_filter=True, rank_method="longllmlingua", token_budget_ratio=1.2)

            prompt = get_windows_prompt(task, context["compressed_prompt"], inputs)
            res.append({"id": idx, "task": task, "prompt": prompt, "answers": answers, "all_classes": all_classes})
            n += 1

    json.dump(res, open(f"prompt/long_bench/llmlingua_2k_4.json", "w"))

def get_dataset_sc():
    n = 0
    for task in TASKS:
        dataset = load_dataset('THUDM/LongBench', task, split='test')
        for idx, data in tqdm(enumerate(dataset), total=dataset.num_rows):
            if f"{task},{idx}" in res:
                n += 1
                continue
            context, inputs, all_classes, answers = [data[i] for i in ["context", "input", "all_classes", "answers"]]
            context = INSTRUCTIONS[task].format(context=context) + QUESTIONS[task].format(input=inputs)
            # docs = get_doc(context, task)
            o1 = len(tokenizer.encode(context))

            try:
                y = sc(context, reduce_ratio=max(min(1, 1 - 4100/o1), 0))
                compressed_prompt = y[0].replace("<s>", "").strip()
            except:
                compressed_prompt = encoding.decode(encoding.encode(context)[-4100:])

            # prompt = get_windows_prompt(task, context["compressed_prompt"], inputs)
            res[f"{task},{idx}"] = {"id": idx, "task": task, "prompt": compressed_prompt, "answers": answers, "all_classes": all_classes}
            n += 1
            torch.cuda.empty_cache()

    res = {f"{k['task']},{k['id']}": k for k in res}

    json.dump(list(res.values()), open(f"prompt/long_bench/sc_2k_2.json", "w"))


def get_dataset_ours():
    def get_doc(context, task):
        if task == "trec":
            docs = get_doc_trec(context)
        elif task == "qasper":
            docs = get_doc_qasper(context)
        elif task == "repobench-p":
            docs = get_doc_code(context)
        else:
            docs = [ii.strip() for ii in context.split("    _") if ii.strip()]
            if len(docs) == 1:
                docs = [ii.strip() for ii in context.split("\n\n\n\n\n") if ii.strip()]
            if len(docs) < 5:
                docs = [ii.strip() for ii in context.split("\n\n") if ii.strip()]
            if len(docs) > 300:
                c = encoding.encode(context)
                docs = [encoding.decode(c[ii - 1000:ii]) for ii in range(1000, len(c) + 1000, 1000)]
            if len(docs) == 1:
                docs = get_doc_qasper(context)
        return docs
    def get_doc_trec(context):
        cs = context.split("\n")
        res = [cs[ii] + '\n' + cs[ii + 1] for ii in range(0, len(cs), 2)]
        return res
    def get_doc_qasper(context):
        cs = context.split("\n")
        res = [[]]
        for c in cs:
            if not c or c.endswith("pt") or c == "]":
                continue
            if c[-1] in ".:?!":
                res[-1].append(c)
            else:
                res.append([c])
        res = ["\n".join(ii) for ii in res if ii]
        return res
    def get_doc_code(context):
        res = []
        names = re.findall("([\w0-9_\-]*/)?([\w0-9_\-]*\.[\w0-9_\-]+)\n", context)

        pre = 0
        for name in names[1:]:
            name = name[0] + name[1]
            idx = context.find(name, pre + len(name) + 10)
            while idx > pre and context[idx] != "\n":
                idx -= 1
            idx += 1
            y = context[pre:idx].find(".")
            if (context[pre] in "\t " or (y != -1 and " " in context[pre:y])) and res:
                res[-1] += context[pre: idx]
            else:
                res.append(context[pre: idx])
            pre = idx
        res.append(context[pre:])
        return res


    res = []
    n = 0
    for task in TASKS[9:10]:
        dataset = load_dataset('THUDM/LongBench', task, split='test')
        for idx, data in tqdm(enumerate(dataset), total=dataset.num_rows):
            if len(res) > n:
                n += 1
                continue
            context, inputs, all_classes, answers = [data[i] for i in ["context", "input", "all_classes", "answers"]]
            question = QUESTIONS[task].format(input=inputs)
            # context = encoding.decode(encoding.encode(context)[-8096:])
            docs = get_doc(context, task)
            docs = [ii for ii in docs if len(ii) > 20]
            context = llm_lingua(docs, "", question, target_token=2048, use_sentence_level_filter=False, condition_in_question="after_condition", reorder_demonstrations=False, dynamic_demonstration_compression_ratio=0.3, condition_compare=True, concate_question=False, demonstration_budget="+500")
            # context = llm_lingua(context, "", question, target_token=2048, use_sentence_level_filter=True, condition_in_question="after_condition", reorder_demonstrations=False, dynamic_demonstration_compression_ratio=0.4, condition_compare=True, concate_question=False, demonstration_budget="+100")
            prompt = get_windows_prompt(task, context["compressed_prompt"], inputs)
            res.append({"id": idx, "task": task, "prompt": prompt, "answers": answers, "all_classes": all_classes})
            n += 1

    res = []
    n = 0
    for task in TASKS:
        dataset = load_dataset('THUDM/LongBench', task, split='test')
        for idx, data in tqdm(enumerate(dataset), total=dataset.num_rows):
            if len(res) > n:
                n += 1
                continue
            context, inputs, all_classes, answers = [data[i] for i in ["context", "input", "all_classes", "answers"]]
            question = QUESTIONS[task].format(input=inputs)
            if task not in ["trec", "passage_count", "repobench-p"]:
                context = encoding.decode(encoding.encode(context)[-8096:])
            if task in ["lcc"]:
                cs = context.split("\n")
                docs, q = "\n".join(cs[:-50]), "\n".join(cs[-50:])
                question = q + question
            else:
                docs = get_doc(context, task)
                docs = [ii for ii in docs if len(ii) > 20]
                q = ""
            if task in ["passage_count"]:
                prefix = [ii.split(":")[0] for ii in docs]
                docs = [ii.split(":", 1)[1][1:] for ii in docs]

            context = llm_lingua(docs, "", question, target_token=2048, use_sentence_level_filter=True, condition_in_question="after_condition", reorder_demonstrations=False, dynamic_demonstration_compression_ratio=0.3, condition_compare=True, concate_question=False, demonstration_budget="+500", use_demonstrate_level_filter=task not in ["passage_count"], keep_split=task in ["passage_count"], keep_first_sentence=10 if task in ["gov_report", "qmsum", "multi_news"] else 0)
            if task in ["passage_count"]:
                cs = [j + ": " + i for i, j in zip(context["compressed_prompt"].split("\n\n"), prefix)]
                context["compressed_prompt"] = "\n\n".join(cs)
                
            # context = llm_lingua(context, "", question, target_token=2048, use_sentence_level_filter=True, condition_in_question="after_condition", reorder_demonstrations=False, dynamic_demonstration_compression_ratio=0.4, condition_compare=True, concate_question=False, demonstration_budget="+100")
            prompt = get_windows_prompt(task, context["compressed_prompt"] + q, inputs)
            res.append({"id": idx, "task": task, "prompt": prompt, "answers": answers, "all_classes": all_classes})
            n += 1


    json.dump(res, open(f"prompt/long_bench/ours_4k_dem_500_after_add_prompt1_dy04_compare_condition_ori_order_8k.json", "w"))

def get_token_length(path):
    data = json.load(open(path))
    res = [len(encoding.encode(k["prompt"])) for k in data]
    return sum(res) / len(res)

def recover(
    # self,
    original_prompt: str,
    compressed_prompt: str,
    response: str,
):
    def match_from_compressed(response_words):
        # response_input_ids = self.tokenizer(response)["input_ids"]
        response_input_ids = tokenizer(response_words, add_special_tokens=False)["input_ids"]
        starts_ids = [idx for idx in range(M) if response_input_ids[0] == original_input_ids[idx]]
        res, res_min = None, float("inf")
        n = len(response_input_ids)
        for l in starts_ids:
            x, y = 0, l
            while x < n and y < M:
                if response_input_ids[x] == original_input_ids[y]:
                    x += 1
                    y += 1
                else:
                    y += 1
                if y - l >= res_min:
                    break
            if x == n:
                if y - l < res_min:
                    res_min = y - l
                    res = (l, y)
        if res is None:
            return response_words
        # while l > 0 and not tokenizer.convert_ids_to_tokens(original_input_ids[l]).startswith("_"):
        #     l -= 1
        # while r < M - 1 and not tokenizer.convert_ids_to_tokens(original_input_ids[l]).startswith("_"):
        #     l -= 1
        return tokenizer.decode(original_input_ids[res[0]:res[1]])


    response_words = response.split()
    compressed_prompt_words = compressed_prompt.split()
    original_prompt_words = original_prompt.split(" ")

    # original_tokens = self.tokenizer(original_prompt, return_offsets_mapping=True, add_special_tokens=False)

    original_input_ids = tokenizer(original_prompt, add_special_tokens=False)['input_ids']
    N, M = len(response_words), len(original_input_ids)
    recovered_response_words = []
    l = 0
    while l < N:
        if response_words[l] not in compressed_prompt:
            recovered_response_words.append(response_words[l])
            l += 1
            continue
        r = l
        while r + 1 < N and " ".join(response_words[l:r+2]) in compressed_prompt:
            r += 1
        # print(l, r, " ".join(response_words[l:r+1]))
        match_words = match_from_compressed(" ".join(response_words[l:r+1]))
        recovered_response_words.append(match_words)
        l = r + 1
    return " ".join(recovered_response_words)

def recover_v2(
    # self,
    original_prompt: str,
    compressed_prompt: str,
    response: str,
):
    import bisect
    from collections import defaultdict
    # def get_LCS():

    def match_from_compressed(response_word):
        response_input_ids = tokenizer(response_word, add_special_tokens=False)["input_ids"]
        if response_input_ids and response_input_ids[0] == 29871 and response_word[0] != " ":
            response_input_ids = response_input_ids[1:]
        response_set, response_c = set(response_input_ids), defaultdict(list)
        for idx in range(M):
            if original_input_ids[idx] in response_set:
                response_c[original_input_ids[idx]].append(idx)
        res, res_min, res_c = None, float("inf"), 1
        n = len(response_input_ids)
        if n == 0:
            return response_word
        for l in response_c[response_input_ids[0]]:
            x, y, c = 0, l, 1
            flag = True
            for x in range(1, n):
                idx = bisect.bisect_right(response_c[response_input_ids[x]], y)
                if idx >= len(response_c[response_input_ids[x]]) or response_c[response_input_ids[x]][idx] - y > 10:
                    continue
                c += 1
                    # flag = False
                    # break
                y = response_c[response_input_ids[x]][idx]
            # if not flag:
            #     break
            if c > res_c:
                res_c = c
                res_min = y - l + 1
                res = (l, y + 1)
            elif c == res_c and y - l + 1 < res_min:
                res_min = y - l + 1
                res = (l, y + 1)

        if res is None:
            return response_word
        # while l > 0 and not tokenizer.convert_ids_to_tokens(original_input_ids[l]).startswith("_"):
        #     l -= 1
        # while r < M - 1 and not tokenizer.convert_ids_to_tokens(original_input_ids[l]).startswith("_"):
        #     l -= 1
        return tokenizer.decode(original_input_ids[res[0]:res[1]])

    def replace_dot(r, compressed_prompt):
        if response_words[r + 1] == "-" and " ".join(response_words[l:r+1]) + "–" in compressed_prompt:
            response_words[r + 1] = "–"
            return True
        return False


    response_words = response.split(" ")
    compressed_prompt_words = compressed_prompt.split()
    original_prompt_words = original_prompt.split(" ")

    # original_tokens = self.tokenizer(original_prompt, return_offsets_mapping=True, add_special_tokens=False)

    original_input_ids = tokenizer(original_prompt, add_special_tokens=False)['input_ids']
    N, M = len(response_words), len(original_input_ids)
    recovered_response_words = []
    l = 0
    while l < N:
        if response_words[l] not in compressed_prompt:
            recovered_response_words.append(response_words[l])
            l += 1
            continue
        r = l
        while r + 1 < N and (" ".join(response_words[l:r+2]) in compressed_prompt or replace_dot(r, compressed_prompt)):
            r += 1
        match_words = match_from_compressed(" ".join(response_words[l:r+1]))
        recovered_response_words.append(match_words)
        l = r + 1
    return " ".join(recovered_response_words)

def get_pred(preds: dict, prompt_path: str):
    compressed_prompts = json.load(open(prompt_path))
    prompts = json.load(open("prompt/long_bench/full_20_0.json"))
    for idx, example in tqdm(preds.items()):
        pred, ref = example["pred"].strip(), example["reference"]
        pred = pred.split("\n\n")[0].split("\n")[0].strip()
        original_prompt = prompts[int(idx)]["prompt"]
        compressed_prompt = compressed_prompts[int(idx)]["prompt"]["compressed_prompt"]
        # pred1 = recover(original_prompt, compressed_prompt, pred)
        pred2 = recover_v2(original_prompt, compressed_prompt, pred)
        example["pred_covered"] = pred2
    return preds


def analysis_results(predict_path: str, prompt_path: str):
    from tqdm import tqdm
    import json
    import statistics
    from lost_in_the_middle.metrics import best_subspan_em
    METRICS = [(best_subspan_em, "best_subspan_em")]

    def get_metrics_for_example(example):
        gold_answers = example["answers"]
        model_answer = example["model_answer"]

        example_metrics = {}
        for (metric, metric_name) in METRICS:
            example_metrics[metric_name] = metric(prediction=model_answer, ground_truths=gold_answers)
        return (example_metrics, example)
    preds = json.load(open(predict_path))
    compressed_prompts = json.load(open(prompt_path))
    prompts = json.load(open("prompt/long_bench/full_20_0.json"))
    pairs = []
    all_example_metrics = []
    for idx, example in tqdm(preds.items()):
        pred, ref = example["pred"].strip(), example["reference"]
        pred = pred.split("\n\n")[0].split("\n")[0].strip()
        original_prompt = prompts[int(idx)]["prompt"]
        compressed_prompt = compressed_prompts[int(idx)]["prompt"]["compressed_prompt"]
        # pred1 = recover(original_prompt, compressed_prompt, pred)
        pred = pred.replace("<|im_end|>", "")
        pred2 = recover_v2(original_prompt, compressed_prompt, pred)
        pred3 = recover_v2(original_prompt, compressed_prompt, pred2.replace(", ", ",").replace(". ", ".").replace("! ", "!").replace("? ", "?"))
        if pred3 != pred2.replace(", ", ",").replace(". ", ".").replace("! ", "!").replace("? ", "?"):
            pred2 = pred3
        # all_example_metrics.append({"answers": ref, "model_answer": pred2, "origin": pred, "idx": idx})
        pairs.append([ref, idx, pred, pred2])
        all_example_metrics.append(get_metrics_for_example({"answers": ref, "model_answer": pred2}))
    # return all_example_metrics
    # Average metrics across examples
    for (_, metric_name) in METRICS:
        average_metric_value = statistics.mean(
            example_metrics[metric_name] for (example_metrics, _) in all_example_metrics
        )
        print(f"{metric_name}: {average_metric_value}")
    return pairs

# for num in [500, 1000, 1500, 2000, 3000, 5000, 10000]:
for num in [500]:
# for threshold in ["keep_doc_last_1", "keep_doc_last_5", "keep_doc_last_15", "keep_doc_last_30", "keep_doc_1", "keep_doc_5", "keep_doc_15", "keep_doc_30", "keep_1000", "keep_500"]:
# for threshold in ["keep_doc_last_3", "keep_doc_last_10", "keep_doc_last_20", "keep_doc_3", "keep_doc_10", "keep_doc_20", "keep_100", "keep_50", "keep_1500", "keep_200"]:
    model = "gpt-3.5-turbo"
    done, res = set(), {}
    threshold = "full_all"
    # threshold = f"keep_first_{num}_gov_report"
    # threshold = f"ours_20_{position}_4x_dem_after_add_prompt1_dy04dem_compare_condition_ori_order"
    data = json.load(open(f"prompt/long_bench/{threshold}.json"))

    # output_path = f'outputs/gpt_3.5_turbo_{threshold}_text_llama2.txt'
    output_path = f'outputs_long_bench/{threshold}_text512.txt'
    if os.path.exists(output_path):
        res = json.load(open(output_path))

    for demonstration in tqdm(
        data, total=len(data)
    ):
        idx, task, prompt, answer, all_classes = [demonstration[k] for k in ['id', 'task', 'prompt', 'answers', 'all_classes']]
        if isinstance(prompt, dict):
            prompt = prompt["compressed_prompt"]
        if f"{idx},{task}" in res:
            continue

        # message = [
        #     {"role": "user", "content":  prompt},
        # ]

        request_data = {
            # "messages": message,
            "prompt": prompt,
            "max_tokens": 200,
            "temperature": 0,
            "top_p": 1,
            "n": 1,
            "stream": False,
            "stop": "\r\n",
        }

        response = openai.Completion.create(
            model="gpt-3.5-turbo",
            **request_data,
        )

        # torch.save(response, "debug.pt")
        if "choices" not in response:
            print("TOO FAST")
            if "error" in response:
                print(response["error"]["message"])
            time.sleep(62)
            continue
        # ans_model = response["choices"][0]["message"]["content"]
        ans_model = response["choices"][0]["text"]
        print(task, idx, ans_model.replace("\n", " "))
        res[f"{idx},{task}"] = {
            "idx": str(idx),
            "task": task,
            "pred": ans_model,
            "reference": answer,
            "all_classes": all_classes,
        }
        json.dump(res, open(output_path, "w"))
        time.sleep(8)
