from typing import Tuple

import streamlit as st

from ...preprocessing.preprocessor import ProcessedData
from ...hypothesis.hypothesizer import Hypothesis
from ...experiment.experimenter import ChaosExperiment, ChaosExperimentResult
from ...utils.wrappers import LLM, LLMBaseModel, LLMField
from ...utils.llms import build_json_agent, LLMLog, LoggingCallback


SYS_ANALYZE_RESULT = """\
You are a helpful AI assistant for Chaos Engineering.
Given K8s manifests that defines a network system, its hypothesis, the overview of a Chaos-Engineeering experiment, and the experiment's results, you will analyze the experiment's results.
Always keep the following rules:
- Analyze step by step why why the test(s) failed, based on the system configuraions (manifests) and the flow of the experiment.
- Specify the cause while mentioning the corresponding system configurations and the corresponding phenamena Chaos-Engineering experiment.
- The anaysis report here will be used for reconfiguring the system later to avoid the failes (improve resiliency). Therefore, Therefore, make carefully the report rich in insights so that it will be helpful at that time.
- {format_instructions}"""

USER_ANALYZE_RESULT = """\
# Here is the overview of my system:
{system_overview}

# Here is the hypothesis for my system:
{hypothesis_overview}

# Here is the overview of my Chaos-Engineering experiment to verify the hypothesis:
{experiment_plan_summary}

# The experiment's results are as follows:
{experiment_result}

Now, please analyze the results and provide an analysis report rich in insights."""


class AnalysisReport(LLMBaseModel):
    report: str = LLMField(description="Analysis of the experiment result.")


class AnalysisAgent:
    def __init__(self, llm: LLM) -> None:
        self.llm = llm
        self.agent = build_json_agent(
            llm=llm,
            chat_messages=[("system", SYS_ANALYZE_RESULT), ("human", USER_ANALYZE_RESULT)],
            pydantic_object=AnalysisReport,
            is_async=False
        )

    def analyze(
        self,
        input_data: ProcessedData,
        hypothesis: Hypothesis,
        experiment: ChaosExperiment,
        experiment_result: ChaosExperimentResult
    ) -> Tuple[LLMLog, str]: # NOTE: not Analysis, but str
        logger = LoggingCallback(name="analysis_experiment", llm=self.llm)
        empty = st.empty()
        for token in self.agent.stream({
            "system_overview": input_data.to_k8s_overview_str(),
            "hypothesis_overview": hypothesis.to_str(),
            "experiment_plan_summary": experiment.plan["summary"],
            "experiment_result": experiment_result.to_str()},
            {"callbacks": [logger]}
        ):
            if (analysis := token.get("report")) is not None:
                empty.write(analysis)
        return logger.log, analysis