from typing import List, Dict, Tuple

import streamlit as st

from ....preprocessing.preprocessor import ProcessedData
from ....utils.wrappers import LLM, LLMBaseModel, LLMField
from ....utils.llms import build_json_agent, LoggingCallback, LLMLog
from ....utils.app_utils import st_spinner


SYS_DRAFT_STEADY_STATES = """\
You are a helpful AI assistant for Chaos Engineering. 
Given K8s manifests that define a network system and user's instructions, you will list steady states (i.e., normal behaviors) of the system that must be retained to maintain the service.
Always keep the following rules:
- Steady states must be measurable states (e.g., the number of pods, throughput, error rates, latency percentiles, etc.).
- If you identify the resiliency issues of the given K8s manifests,  to confirm whether it's really a problem through Chaos Engineering, prioritize listing the related steady states.
- list 1-2 of the most important steady states.
- NEVER list muliple similar steady states. 
- NEVER select states that does not affect the system health directly (e.g., pod restart count).
- {format_instructions}"""

USER_DRAFT_STEADY_STATES = """\
# Here is the overview of my system.
{user_input}

# Please follow the instructions below regarding Chaos Engineering:
{ce_instructions}

Please list 1-2 of the most important steady states for my system."""


class SteadyStateName(LLMBaseModel):
    name: str = LLMField(description="Steady state name. Please write using a-z, A-Z, and 0-9.")
    description: str = LLMField(description="Describe the reason for selecting the steady state individually. Additionally, explain in detail how this steady states differs from other listed steady states.")

class SteadyStateNames(LLMBaseModel):
    thought: str = LLMField(description="Describe the key points of the steady states you are about to select (why you should select them). ")
    steady_states: List[SteadyStateName] = LLMField(descripton="List of diifferent types of steady states for each resource. NEVER list muliple similar steady states. When names are similar, add identifying keywords to the names to clarify the differences.")


class SteadyStateDraftAgent:
    def __init__(self, llm: LLM) -> None:
        self.llm = llm
        self.agent = build_json_agent(
            llm=llm,
            chat_messages=[("system", SYS_DRAFT_STEADY_STATES), ("human", USER_DRAFT_STEADY_STATES)],
            pydantic_object=SteadyStateNames,
            is_async=False,
            streaming_func=self._extract_json_items_streaming
        )
        self.ICON = {
            "reason": "💬",
            "cmd": "🔧",
            "value": "🔍",
            "threshold": "🚩",
            "unittest": "📄",
        }

    def draft_steady_states(self, input_data: ProcessedData) -> Tuple[LLMLog, List[Dict[str, str]]]:
        logger = LoggingCallback(name="steady_state_draft", llm=self.llm)
        steady_state_names = []
        for steady_state in self.agent.stream({
            "user_input": input_data.to_k8s_overview_str(), 
            "ce_instructions": input_data.ce_instructions},
            {"callbacks": [logger]}
        ):
            if steady_state["number"] + 1 > len(st.session_state.steady_states):
                st.session_state.steady_states.append(self.get_steady_state_items())
                steady_state_names.append({"name": None, "reason": None})
            if steady_state["name"] is not None:
                st.session_state.steady_states[steady_state["number"]]["name"].expander("##### " + "⬜  " + steady_state["name"], expanded=True)
                steady_state_names[steady_state["number"]]["name"] = steady_state["name"]
            if steady_state["description"] is not None:
                st.session_state.steady_states[steady_state["number"]]["reason"]["spinner"].end()
                st.session_state.steady_states[steady_state["number"]]["reason"]["empty"].write(steady_state["description"])
                steady_state_names[steady_state["number"]]["reason"] = steady_state["description"]
        return logger.log, steady_state_names

    def _extract_json_items_streaming(self, input_stream):
        for input in input_stream:
            if not isinstance(input, dict):
                continue
            if "steady_states" not in input:
                continue
            steady_states = input["steady_states"]
            if not isinstance(steady_states, list):
                continue
            for i, steady_state in enumerate(steady_states):
                yield {"number": i} | {key: steady_state.get(key) for key in SteadyStateName.__fields__.keys()}
    
    def get_steady_state_items(self):
        steady_state_items = {}
        steady_state_items["name"] = st.empty()
        expander = steady_state_items["name"].expander("##### " + "⬜  ", expanded=True)
        with expander:
            for key, value in self.ICON.items():
                frame_empty = st.empty()
                col1, col2 = frame_empty.columns([1, 20])
                with col1:
                    st.markdown(f"<span title='{key}'>{value}</span>", unsafe_allow_html=True)
                with col2:
                    if key == "cmd":
                        steady_state_items[key] = {
                            "tool": {"spinner": st_spinner("Pending..."), "empty": st.empty()},
                            "cmd": {"spinner": st_spinner("Pending..."), "empty": st.empty()},
                        }
                    else:
                        steady_state_items[key] = {"spinner": st_spinner("Pending..."), "empty": st.empty()}
        return steady_state_items