# default packages
from typing import List, Dict, Tuple
# chaos eater classes
from .write_cmd import InspectionCMDAgent
from .define_threshold import ThresholdAgent
from .write_unittests import UnittestAgent
from .utils import Inspection
from ....preprocessing.preprocessor import ProcessedData
from ....utils.wrappers import LLM, BaseModel
from ....utils.schemas import File
from ....utils.llms import LLMLog
from ....utils.functions import int_to_ordinal

K6_POD_TEMPLATE_PATH = "chaos_eater/ce_tools/k6/templates/k6_pod_template.j2"

STEADY_STATE_OVERVIEW_TEMPLATE = """\
{number} steady states are defined.
{steady_state_list}"""

STEADY_STATE_LIST_TEMPLATE = """\
{ordinal_number} steady states:
- Name: {name}
- Description: {description}
- Threshold for the steady state: {threshold}; {threshold_description}
- Whether the steady state meets the threshold is determined by the following {script_type}:
```
{script}
```"""


class SteadyState(BaseModel):
    id: int
    name: str
    description: str
    inspection: Inspection
    threshold: Dict[str, str]
    unittest: File

class SteadyStates(BaseModel):
    elems: List[SteadyState]

    def to_overview_str(self) -> str:
        steady_state_list_str = ""
        for steady_state in self.elems:
            steady_state_list_str += STEADY_STATE_LIST_TEMPLATE.format(
                ordinal_number=int_to_ordinal(steady_state.id+1),
                name=steady_state.name,
                description=steady_state.description,
                threshold=steady_state.threshold['threshold'],
                threshold_description=steady_state.threshold['reason'],
                script_type="K6 Javascript" if steady_state.inspection.tool_type == "k6" else "Python script with K8s API",
                script=steady_state.unittest.content
            )
        return STEADY_STATE_OVERVIEW_TEMPLATE.format(
            number=len(self.elems),
            steady_state_list=steady_state_list_str
        )

    def to_str(self) -> str:
        return self.to_overview_str()


class SteadyStateRefiner:
    def __init__(
        self,
        llm: LLM,
        namespace: str = "chaos-eater"
    ) -> None:
        self.llm = llm
        self.namespace = namespace
        self.inspection_agent = InspectionCMDAgent(llm, namespace)
        self.threshold_agent = ThresholdAgent(llm)
        self.unittest_agent = UnittestAgent(llm)

    def refine(
        self,
        data: ProcessedData,
        steady_state_names: List[Dict[str, str]],
        work_dir: str
    ) -> Tuple[List[LLMLog], SteadyStates]:
        logs = []
        #------------------------------------------
        # define commands to inspect steady states
        #------------------------------------------
        cmd_log, inspections = self.inspection_agent.write_cmd(
            input_data=data,
            steady_state_names=steady_state_names,
            work_dir=work_dir
        )
        logs.append(cmd_log)

        #------------------------------------------
        # define thresholds for each steady steate
        #------------------------------------------
        threshold_log, thresholds = self.threshold_agent.define_thresholds(
            input_data=data,
            steady_state_names=steady_state_names,
            inspections=inspections
        )
        logs.append(threshold_log)

        #----------------------------------------
        # write unit tests for each steady state
        #----------------------------------------
        unittest_log, unittests = self.unittest_agent.write_unittests(
            input_data=data,
            steady_state_names=steady_state_names,
            inspections=inspections,
            thresholds=thresholds,
            work_dir=work_dir
        )
        logs.append(unittest_log)

        #----------
        # epilogue
        #----------
        steady_states = []
        for id, (steady_state, inspection, threshold, unittest) in enumerate(zip(steady_state_names, inspections, thresholds, unittests)):
            steady_states.append(SteadyState(
                id=id,
                name=steady_state["name"],
                description=steady_state["reason"],
                inspection=inspection,
                threshold=threshold,
                unittest=unittest
            ))
        return logs, SteadyStates(elems=steady_states)