import os
import time
import subprocess
import yaml
from typing import List, Tuple, Optional

import streamlit as st

from .llm_agents.k8s_summary_agent import K8sSummaryAgent
from .llm_agents.k8s_analysis_agent import K8sAnalysisAgent, K8sDependencies
from .llm_agents.k8s_app_assuption_agent import K8sAppAssumptionAgent, K8sAppAssumption
from .llm_agents.ce_instruct_agent import CEInstructAgent
from ..utils.functions import write_file, save_json, pseudo_streaming_text, type_cmd, recursive_to_dict
from ..utils.wrappers import LLM, BaseModel
from ..utils.schemas import File
from ..utils.k8s import wait_for_resources_ready
from ..utils.llms import LLMLog


INPUT_TEMPLATE = """\
K8s manifest: {k8s_yaml_name}
```yaml
{k8s_yaml}
```
Summary of {k8s_yaml_name}:
{k8s_summary}"""


class ChaosEaterInput(BaseModel):
    skaffold_yaml: File
    files: List[File]
    ce_instructions: Optional[str]

class ProcessedData(BaseModel):
    work_dir: str
    input: ChaosEaterInput
    k8s_yamls: List[File]
    k8s_summaries: List[str]
    k8s_dependencies: K8sDependencies
    k8s_app: K8sAppAssumption
    ce_instructions: str

    def to_k8s_overview_str(self) -> str:
        user_input = "The system consists of the following K8s manifest(s):"
        # add k8s yamls and their summaries
        for k8s_yaml, k8s_summary in zip(self.k8s_yamls, self.k8s_summaries):
            user_input += INPUT_TEMPLATE.format(
                k8s_yaml=k8s_yaml.content,
                k8s_yaml_name=k8s_yaml.fname,
                k8s_summary=k8s_summary
            )
            user_input += "\n\n"
        # add dependencies
        user_input += "The intra/inter dependencies of the above K8s manifests are as follows:" 
        for intra_dependency in self.k8s_dependencies.intra:
            user_input += f"- Dependencies within {intra_dependency.file}:\n{intra_dependency.dependency}\n\n"
        for inter_dependency in self.k8s_dependencies.inter:
            user_input += f"- Dependencies from {inter_dependency.src_file} to {inter_dependency.dst_file}:\n{inter_dependency.dependency}\n\n"
        # add application
        user_input += f"The expected type of application on the system (i.e., K8s manfests):\n{self.k8s_app.k8s_application}; {self.k8s_app.thought}"
        return user_input

    def to_str(self) -> str:
        overview_str = self.to_k8s_overview_str() + "\n\n"
        if self.ce_instructions != "":
            overview_str += f"Chaos-Engineering instructions for the system are as follows: {self.ce_instructions}"
        else:
            overview_str += f"No Chaos-Engineering instructions for the system are provided."
        return overview_str


class PreProcessor:
    def __init__(self, llm: LLM) -> None:
        self.llm = llm
        self.k8s_summary_agent = K8sSummaryAgent(llm)
        self.k8s_analysis_agent = K8sAnalysisAgent(llm)
        self.k8s_app_assumption_agent = K8sAppAssumptionAgent(llm)
        self.ce_instruct_agent = CEInstructAgent(llm)

    def process(
        self,
        input: ChaosEaterInput,
        work_dir: str,
        project_name: str = "chaos-eater",
        is_new_deployment: bool = True
    ) -> Tuple[List[LLMLog], ProcessedData, ChaosEaterInput]:
        pseudo_streaming_text("##### First, prprocess your inputs!")
        log = []
        #------------------------------------------------------
        # save the input manifests, then deploy them if needed
        #------------------------------------------------------
        preprocess_dir = f"{work_dir}/inputs"
        # save skaffold
        skaffold_path = f"{preprocess_dir}/{input.skaffold_yaml.fname}"
        os.makedirs(os.path.dirname(skaffold_path), exist_ok=True)
        skaffold_content = input.skaffold_yaml.content
        write_file(skaffold_path, skaffold_content)
        new_skaffold_yaml = File(
            path=skaffold_path,
            content=skaffold_content,
            work_dir=preprocess_dir,
            fname=input.skaffold_yaml.fname
        )
        # save files
        deployed_yamls = yaml.safe_load(skaffold_content)["manifests"]["rawYaml"]
        skaffold_yaml_dir = os.path.dirname(input.skaffold_yaml.fname)
        deployed_yamls = [f"{skaffold_yaml_dir}/{deployed_yaml}" for deployed_yaml in deployed_yamls]
        new_files = []
        k8s_yamls_ = []
        # save files
        for file in input.files:
            path = f"{preprocess_dir}/{file.fname}"
            os.makedirs(os.path.dirname(path), exist_ok=True)
            write_file(path, file.content)
            new_file = File(
                path=path,
                content=file.content,
                work_dir=preprocess_dir,
                fname=file.fname
            )
            new_files.append(new_file)
            if file.fname in deployed_yamls:
                k8s_yamls_.append(new_file)
        duplicated_input = ChaosEaterInput(
            skaffold_yaml=new_skaffold_yaml,
            files=new_files,
            ce_instructions=input.ce_instructions
        )
        st.write("##### K8s manifest(s) to be deployed:")
        for k8s_yaml in k8s_yamls_:
            st.write(f"```{k8s_yaml.fname}```")
            st.code(k8s_yaml.content)
        # deploy the resouces
        deployment_msg = st.empty()
        pseudo_streaming_text("##### Deploying resources...", obj=deployment_msg)
        if is_new_deployment:
            # deploy the k8s yamls with skaffold
            process = subprocess.Popen(
                f"skaffold run -l project={project_name}",
                shell=True,
                cwd=os.path.dirname(new_skaffold_yaml.path)
            )
        # wait for the deployment
        wait_for_resources_ready(label_selector=f"project={project_name}")
        # display each resouce status
        pseudo_streaming_text("##### Resource statuses", obj=deployment_msg)
        status_log = type_cmd(f"kubectl get all --selector=project={project_name}")
        st.code(status_log, language="powershell")

        #-----------------------------
        # summarize each k8s manifest
        #-----------------------------
        pseudo_streaming_text("##### Summary of each manifest:")
        summary_log, k8s_summaries = self.k8s_summary_agent.summarize_manifests(k8s_yamls=k8s_yamls_)
        log.append(summary_log)

        #---------------------------
        # analyze file dependencies
        #---------------------------
        pseudo_streaming_text("##### Dependencies between the manifests:")
        depdency_token_usage, k8s_dependencies = self.k8s_analysis_agent.analyze_manifest_dependencies(
            k8s_yamls=k8s_yamls_,
            k8s_summaries=k8s_summaries,
            work_dir=work_dir,
            project_name=project_name
        )
        log.append(depdency_token_usage)

        pseudo_streaming_text("##### Application of the manifests:")
        #---------------------------------------------
        # assume the application of the k8s manifests
        #---------------------------------------------
        app_token_usage, k8s_application = self.k8s_app_assumption_agent.assume_app(
            k8s_yamls=k8s_yamls_,
            k8s_summaries=k8s_summaries,
            k8s_dependencies=k8s_dependencies
        )
        log.append(app_token_usage)

        pseudo_streaming_text("##### Summary of your instructions for Chaos Engineering:")
        #----------------------------------------------
        # summarize instructions for Chaos Engineering
        #----------------------------------------------
        instruct_token_usage, ce_instructions = self.ce_instruct_agent.summarize_ce_instructions(input.ce_instructions)
        log.append(instruct_token_usage)

        #----------
        # epilogue
        #----------
        processed_data = ProcessedData(
            work_dir=preprocess_dir,
            input=duplicated_input,
            k8s_yamls=k8s_yamls_,
            k8s_summaries=k8s_summaries,
            k8s_dependencies=k8s_dependencies,
            k8s_app=k8s_application,
            ce_instructions=ce_instructions
        )
        save_json(f"{preprocess_dir}/processed_data.json", processed_data.dict())
        save_json(f"{preprocess_dir}/preprcessing_log.json", recursive_to_dict(log))
        return (
            log,
            processed_data,
            duplicated_input
        )