import os
import subprocess
import time
from kubernetes import client, config


if os.getenv('KUBERNETES_SERVICE_HOST'):
    config.load_incluster_config()
else:
    config.load_kube_config()

def kubectl_apply(manifest_path):
    try:
        result = subprocess.run(["kubectl", "apply", "-f", manifest_path], capture_output=True, text=True)
        if result.returncode == 0:
            print("Applied manifest successfully.")
            print(result.stdout)
        else:
            print("Error applying manifest:")
            print(result.stderr)
            return False
        return True
    except Exception as e:
        print(f"Error running kubectl apply: {e}")
        return False

def check_deployment_status_by_label(label_selector):
    api = client.AppsV1Api()
    deployments = api.list_deployment_for_all_namespaces(label_selector=label_selector)
    for deployment in deployments.items:
        available_replicas = deployment.status.available_replicas or 0
        desired_replicas = deployment.spec.replicas or 0
        print(f"Deployment {deployment.metadata.name} in namespace {deployment.metadata.namespace}: Available replicas {available_replicas}/{desired_replicas}")
        if available_replicas != desired_replicas:
            return False
    return True

def check_pod_status_by_label(label_selector):
    api = client.CoreV1Api()
    pods = api.list_pod_for_all_namespaces(label_selector=label_selector)
    for pod in pods.items:
        print(f"Pod {pod.metadata.name} in namespace {pod.metadata.namespace}: Phase {pod.status.phase}")
        if pod.status.phase != "Running":
            return False
    return True

def check_service_status_by_label(label_selector):
    api = client.CoreV1Api()
    services = api.list_service_for_all_namespaces(label_selector=label_selector)
    for service in services.items:
        print(f"Service {service.metadata.name} in namespace {service.metadata.namespace}: Cluster IP {service.spec.cluster_ip}")
        if not service.spec.cluster_ip:
            return False
    return True

def check_job_status_by_label(label_selector):
    api = client.BatchV1Api()
    jobs = api.list_job_for_all_namespaces(label_selector=label_selector)
    for job in jobs.items:
        print(f"Job {job.metadata.name} in namespace {job.metadata.namespace}: Succeeded {job.status.succeeded}")
        if job.status.succeeded < 1:
            return False
    return True

def check_statefulset_status_by_label(label_selector):
    api = client.AppsV1Api()
    statefulsets = api.list_stateful_set_for_all_namespaces(label_selector=label_selector)
    for statefulset in statefulsets.items:
        ready_replicas = statefulset.status.ready_replicas or 0
        desired_replicas = statefulset.spec.replicas or 0
        print(f"StatefulSet {statefulset.metadata.name} in namespace {statefulset.metadata.namespace}: Ready replicas {ready_replicas}/{desired_replicas}")
        if ready_replicas != desired_replicas:
            return False
    return True

def check_daemonset_status_by_label(label_selector):
    api = client.AppsV1Api()
    daemonsets = api.list_daemon_set_for_all_namespaces(label_selector=label_selector)
    for daemonset in daemonsets.items:
        number_available = daemonset.status.number_available or 0
        desired_number_scheduled = daemonset.status.desired_number_scheduled or 0
        print(f"DaemonSet {daemonset.metadata.name} in namespace {daemonset.metadata.namespace}: Available {number_available}/{desired_number_scheduled}")
        if number_available != desired_number_scheduled:
            return False
    return True

def check_resources_status(label_selector):
    if not check_deployment_status_by_label(label_selector):
        return False
    if not check_pod_status_by_label(label_selector):
        return False
    if not check_service_status_by_label(label_selector):
        return False
    if not check_job_status_by_label(label_selector):
        return False
    if not check_statefulset_status_by_label(label_selector):
        return False
    if not check_daemonset_status_by_label(label_selector):
        return False
    return True

def wait_for_resources_ready(label_selector, timeout=300):
    start_time = time.time()
    while time.time() - start_time < timeout:
        if check_resources_status(label_selector):
            print(f"All resources with label {label_selector} are ready.")
            time.sleep(1)
            return True
        time.sleep(1)
    print(f"Resources with label {label_selector} did not become ready within the timeout period.")
    return False

def remove_all_resources_by_labels(label_selector: str) -> None:
    subprocess.run(f"kubectl delete all -l {label_selector}", shell=True)