#!/bin/bash

#------------------
# default settings
#------------------
CLUSTER_NAME="chaos-eater-cluster"

#-----------------
# analyze options
#-----------------
while [[ "$#" -gt 0 ]]; do
    case $1 in
        -n|--name) CLUSTER_NAME="$2"; shift 2;;  # Handle both -n and --name
        *) echo "Unknown parameter passed: $1"; exit 1;;
    esac
done

#------------------
# cluster settings
#------------------
# Export PWD environment variable to make sure envsubst works correctly
export PWD=$(pwd)
# Create the kind cluster with our configuration, replacing the environment variable "PWD" with the root dir
envsubst < k8s/kind_config.yaml | kind create cluster --config=- --name "${CLUSTER_NAME}"
# Check if kind cluster creation was successful
if [ $? -ne 0 ]; then
    echo "Failed to create the kind cluster."
    exit 1
fi

#------------------
# kubectl commands
#------------------
# Switch to the created cluster's context
kubectl config use-context kind-${CLUSTER_NAME}

# Create namespace "chaos-eater"
kubectl create namespace chaos-eater

# Deploy pv/pvc
kubectl apply -f k8s/pv.yaml
kubectl apply -f k8s/pvc.yaml

# Grant superuser authorization to the "default" service account in the "chaos-eater" namespace
kubectl apply -f k8s/super_user_role_binding.yaml

# Enable `kubectl top` by deploying the metrics-server
kubectl apply -n kube-system -f https://github.com/kubernetes-sigs/metrics-server/releases/latest/download/components.yaml

# Build & load a docker image
docker build -t chaos-eater/k8sapi:1.0 -f docker/Dockerfile_k8sapi .
kind load docker-image chaos-eater/k8sapi:1.0 --name ${CLUSTER_NAME}

# Install Chaos Mesh
curl -sSL https://mirrors.chaos-mesh.org/v2.6.3/install.sh | bash -s -- --local kind --name ${CLUSTER_NAME}
# Function to check if chaos-dashboard is running
check_chaos_dashboard() {
    kubectl get pods -n chaos-mesh -l app.kubernetes.io/component=chaos-dashboard -o jsonpath='{.items[0].status.phase}' 2>/dev/null
}
# Wait for chaos-dashboard to be running
echo "Waiting for chaos-dashboard to be ready..."
while [[ "$(check_chaos_dashboard)" != "Running" ]]; do
    echo "Waiting for chaos-dashboard to be ready..."
    sleep 5
done

echo "Chaos dashboard is ready. Starting port-forward..."
# Enable Chaos Mesh dashboard via port-forwarding in the background
kubectl port-forward -n chaos-mesh svc/chaos-dashboard 2333:2333 &
# Get the PID of the background port-forward process
PORT_FORWARD_PID=$!
# Print the background job information and the PID
echo "Chaos Mesh dashboard is being port-forwarded at http://localhost:2333 in the background."
echo "To stop the port-forward process, use: kill ${PORT_FORWARD_PID}"

#----------
# epilogue
#----------
echo "A kind cluster named '${CLUSTER_NAME}' has been created successuly!"
