import copy
import gc
import logging

import torch
import numpy as np

from methods._trainer import _Trainer

logger = logging.getLogger()


class MoERanPAC(_Trainer):
    def __init__(self, *args, **kwargs):
        super(MoERanPAC, self).__init__(*args, **kwargs)

        self.task_id = 0
        self.label_to_task = {}

    def online_step(self, images, labels, idx):
        self.add_new_class(labels)
        _loss, _acc, _iter = 0.0, 0.0, 0

        # Train with multiple iterations as specified
        for _ in range(int(self.online_iter)):
            loss, acc = self.online_train([images.clone(), labels.clone()])
            _loss += loss
            _acc += acc
            _iter += 1

        self._collect_features_for_statistics(images, labels)

        return _loss / _iter, _acc / _iter

    def _collect_features_for_statistics(self, images, labels):
        images_copy = images.clone()
        labels_copy = labels.clone()
        
        # Map labels to exposed class indices
        for j in range(len(labels_copy)):
            labels_copy[j] = self.exposed_classes.index(labels_copy[j].item())

        unique_labels = torch.unique(labels_copy)
        for label in unique_labels:
            if label.item() not in self.label_to_task:
                self.label_to_task[label.item()] = self.task_id

        images_copy = images_copy.to(self.device)
        labels_copy = labels_copy.to(self.device)

        images_copy = self.test_transform_tensor(images_copy)

        with torch.no_grad():
            self.model.eval()
            self.model_without_ddp.collect_features_labels(images_copy, labels_copy)

        del images_copy, labels_copy
        gc.collect()

    def online_train(self, data):
        self.model.train()
        total_loss, total_correct, total_num_data = 0.0, 0.0, 0.0

        x, y = data

        for j in range(len(y)):
            y[j] = self.exposed_classes.index(y[j].item())

        logit_mask = torch.zeros_like(self.mask) - torch.inf
        cls_lst = torch.unique(y)
        for cc in cls_lst:
            logit_mask[cc] = 0

        x = x.to(self.device)
        y = y.to(self.device)

        x = self.train_transform(x)

        self.optimizer.zero_grad()
        if not self.no_batchmask:
            logit, loss = self.model_forward(x,y,mask=logit_mask)
        else:
            logit, loss = self.model_forward(x,y)

        _, preds = logit.topk(self.topk, 1, True, True)

        self.scaler.scale(loss).backward()
        self.scaler.step(self.optimizer)
        self.scaler.update()
        self.update_schedule()

        total_loss += loss.item()
        total_correct += torch.sum(preds == y.unsqueeze(1)).item()
        total_num_data += y.size(0)

        return total_loss, total_correct/total_num_data

    def model_forward(self, x, y, mask=None):
        with torch.cuda.amp.autocast(enabled=self.use_amp):
            if self.task_id == 0:
                logit = self.model(x)
            else:
                expert_ids = torch.full((y.size(0),), self.task_id - 1, device=y.device, dtype=torch.long)
                logit = self.model_without_ddp.forward_with_experts(x, expert_ids)

            if mask is not None:
                logit += mask
            else:
                logit += self.mask

            loss = self.criterion(logit, y)

        return logit, loss

    def online_evaluate(self, test_loader, task_id=None, end=False):
        total_correct, total_num_data, total_loss = 0.0, 0.0, 0.0
        correct_l = torch.zeros(self.n_classes)
        num_data_l = torch.zeros(self.n_classes)
        label = []

        if self.task_id > 0:
            self.model_without_ddp.update_classifier()

        self.model.eval()
        with torch.no_grad():
            for i, data in enumerate(test_loader):
                x, y = data
                
                # Map labels to exposed class indices
                for j in range(len(y)):
                    y[j] = self.exposed_classes.index(y[j].item())

                x = x.to(self.device)
                y = y.to(self.device)

                if self.task_id == 0:
                    logit_raw = self.model(x)
                    logit = logit_raw
                else:
                    logit_raw = self.model(x)
                    # pred_raw = torch.argmax(logit_raw, dim=-1)
                    # expert_ids = [self.label_to_task[p.item()]-1 for p in pred_raw]
                    # expert_ids = torch.tensor(expert_ids, device=y.device, dtype=torch.long)
                    # logit = self.model_without_ddp.forward_for_final_eval(x, expert_ids)

                    # Top2 ensemble
                    top1_pred = torch.argmax(logit_raw, dim=-1)
                    expert_ids = [self.label_to_task[p.item()]-1 for p in top1_pred]
                    expert_ids = torch.tensor(expert_ids, device=y.device, dtype=torch.long)
                    top1_logit = self.model_without_ddp.forward_for_final_eval(x, expert_ids)

                    top2_pred = torch.topk(logit_raw, 2, dim=-1)[1][:, 1]
                    expert_ids = [self.label_to_task[p.item()]-1 for p in top2_pred]
                    expert_ids = torch.tensor(expert_ids, device=y.device, dtype=torch.long)
                    top2_logit = self.model_without_ddp.forward_for_final_eval(x, expert_ids)

                    logit = (top1_logit + top2_logit) / 2

                logit = logit + self.mask
                loss = self.criterion(logit, y)
                pred = torch.argmax(logit, dim=-1)
                _, preds = logit.topk(self.topk, 1, True, True)
                total_correct += torch.sum(preds == y.unsqueeze(1)).item()
                total_num_data += y.size(0)

                xlabel_cnt, correct_xlabel_cnt = self._interpret_pred(y, pred)
                correct_l += correct_xlabel_cnt.detach().cpu()
                num_data_l += xlabel_cnt.detach().cpu()

                total_loss += loss.item()
                label += y.tolist()

        avg_acc = total_correct / total_num_data
        avg_loss = total_loss / len(test_loader)
        cls_acc = (correct_l / (num_data_l + 1e-5)).numpy().tolist()

        eval_dict = {"avg_loss": avg_loss, "avg_acc": avg_acc, "cls_acc": cls_acc}
        return eval_dict

    def online_before_task(self, task_id):
        if task_id == 0:
            if self.model_without_ddp.use_lora:
                self.model_without_ddp.freeze_backbone_except_lora()
                logger.info("First task: LoRA and classifier enabled for training")
            else:
                self.model_without_ddp.freeze_backbone_except_adapters()
                logger.info("First task: adapters and classifier enabled for training")
        else:
            self.model_without_ddp.init_new_expert(task_id)
            self.model_without_ddp.freeze_backbone_except_experts()

            mode = "LoRA" if self.model_without_ddp.use_lora else "adapter"
            logger.info(f"Task {task_id} ({mode} mode): collecting features for RanPAC statistics")

    def online_after_task(self, cur_iter):
        if self.task_id == 0:
            logger.info("Completing first task training, setting up random projection")

            if self.model_without_ddp.use_lora:
                self.model_without_ddp.merge_lora_weights()
            self.model_without_ddp.copy_classifier()
            self.model_without_ddp.update_classifier()
            self.model_without_ddp.freeze_backbone_except_experts()

            mode = "LoRA" if self.model_without_ddp.use_lora else "adapters"
            logger.info(f"Random projection initialized, {mode} processed")

        self.model_without_ddp.process_task_count()

        logger.info(f"Task {self.task_id} completed, statistics updated")
        self.task_id += 1