'''
Modified from https://github.com/pytorch/vision.git
'''
import math

import torch.nn as nn
import torch.nn.init as init
import torch.nn.functional as F

from utils.builder import get_builder
from args import args

# builder.conv2d(inp, oup, 3, stride, 1, bias=False),
# in_channels,
# out_channels,
# kernel_size,
# stride=1,
# padding=0,
# dilation=1,
# groups=1,
# bias=True,

#linear builder.conv1x1(512*block.expansion, num_classes)

class VGG(nn.Module):
    '''
    VGG model 
    '''
    def __init__(self, builder, features, num_classes):
        super(VGG, self).__init__()
        self.features = features
        self.classifier = nn.Sequential(
            nn.Dropout(),
            builder.conv1x1(512, 512),
            builder.activation(),
            nn.Dropout(),
            builder.conv1x1(512, 512),
            builder.activation(),
            builder.conv1x1(512, num_classes),
        )
         # Initialize weights
        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                n = m.kernel_size[0] * m.kernel_size[1] * m.out_channels
                m.weight.data.normal_(0, math.sqrt(2. / n))
                if m.bias is not None:
                    m.bias.data.zero_()

    def forward(self, x):
        x = self.features(x)
        # x = x.view(x.size(0), -1)
        x = self.classifier(x)
        x = x.view(x.size(0), -1)
        return x


def make_layers(builder, cfg, batch_norm=False):
    layers = []
    in_channels = 3
    for v in cfg:
        if v == 'M':
            layers += [nn.MaxPool2d(kernel_size=2, stride=2)]
        else:
            # conv2d = nn.Conv2d(in_channels, v, kernel_size=3, padding=1)
            conv2d = builder.conv3x3(in_channels, v)
            if batch_norm:
                layers += [conv2d, builder.batchnorm(v), builder.activation()]
            else:
                layers += [conv2d, builder.activation()]
            in_channels = v
    return nn.Sequential(*layers)


cfg = {
    'A': [64, 'M', 128, 'M', 256, 256, 'M', 512, 512, 'M', 512, 512, 'M'],
    'B': [64, 64, 'M', 128, 128, 'M', 256, 256, 'M', 512, 512, 'M', 512, 512, 'M'],
    'D': [64, 64, 'M', 128, 128, 'M', 256, 256, 256, 'M', 512, 512, 512, 'M', 512, 512, 512, 'M'],
    'E': [64, 64, 'M', 128, 128, 'M', 256, 256, 256, 256, 'M', 512, 512, 512, 512, 'M', 
          512, 512, 512, 512, 'M'],
}


# def vgg11(num_classes):
#     """VGG 11-layer model (configuration "A")"""
#     return VGG(get_builder(), make_layers(cfg['A']), num_classes)


# def vgg11_bn(num_classes):
#     """VGG 11-layer model (configuration "A") with batch normalization"""
#     return VGG(get_builder(), make_layers(cfg['A'], batch_norm=True), num_classes)


# def vgg13(num_classes):
#     """VGG 13-layer model (configuration "B")"""
#     return VGG(get_builder(), make_layers(cfg['B']), num_classes)


# def vgg13_bn(num_classes):
#     """VGG 13-layer model (configuration "B") with batch normalization"""
#     return VGG(get_builder(), make_layers(cfg['B'], batch_norm=True), num_classes)


def vgg16(num_classes):
    """VGG 16-layer model (configuration "D")"""
    builder = get_builder()

    return VGG(builder, make_layers(builder, cfg['D']), num_classes)


def vgg16_bn(num_classes):
    """VGG 16-layer model (configuration "D") with batch normalization"""
    builder = get_builder()
    return VGG(builder, make_layers(builder, cfg['D'], batch_norm=True), num_classes)