## ui.py
import tkinter as tk

class UI:
    def __init__(self, game):
        """
        Initialize the user interface and associate it with a game instance.

        Args:
            game (Game): The game instance to interact with.
        """
        self._game = game
        self._root = tk.Tk()
        self._root.title("Dice Game")
        self._configure_main_window()
        self._configure_dice_selection()

    def _configure_main_window(self):
        """
        Configure the main window of the application.
        """
        self._roll_button = tk.Button(self._root, text="Roll Dice", command=self._roll_dice)
        self._roll_button.pack(pady=20)

        self._result_label = tk.Label(self._root, text="", font=("Helvetica", 16))
        self._result_label.pack(pady=20)

    def _roll_dice(self):
        """
        Handle the roll dice button click event.
        """
        result = self._game.roll_dice()
        self.show_dice_result(result)

    def show_dice_result(self, result: int):
        """
        Show the result of the dice roll on the UI.

        Args:
            result (int): The result of the dice roll to display.
        """
        self._result_label.config(text=f"Result: {result}")

    def _configure_dice_selection(self):
        """
        Configure the UI to allow the user to select the number of dice.
        """
        self._dice_selection_label = tk.Label(self._root, text="Select number of dice:")
        self._dice_selection_label.pack(pady=5)

        self._dice_selection_spinbox = tk.Spinbox(self._root, from_=1, to=6)
        self._dice_selection_spinbox.pack(pady=5)

        self._dice_selection_button = tk.Button(self._root, text="Set Dice", command=self._set_dice)
        self._dice_selection_button.pack(pady=10)

    def _set_dice(self):
        """
        Set the number of dice based on the user's selection from the spinbox.
        """
        number_of_dice = int(self._dice_selection_spinbox.get())
        self._game.set_number_of_dice(number_of_dice)

    def start(self):
        """
        Start the main loop of the Tkinter UI.
        """
        self._root.mainloop()

# If this file is run directly, we will create a dummy Game instance to test the UI.
if __name__ == "__main__":
    class DummyGame:
        def roll_dice(self):
            return 4  # Dummy result for testing
        
        def set_number_of_dice(self, number: int):
            print(f"Number of dice set to: {number}")

    dummy_game = DummyGame()
    ui = UI(dummy_game)
    ui.start()
