from typing import List, Dict
from metagpt.actions import Action

class Feedback(Action):
    name: str = "Feedback"
    
    async def run(self, context: str, execution: str, execution_result: str):
        prompt = f"""
        Based on your own and other agents' recent actions and results. You are tasked with providing feedback. Analyze the information provided and give constructive feedback.

        Current context (User Request, History of agents' actions):
        {context}

        Your Latest Execution: 
        {execution}

        Your Latest Execution result:
        {execution_result}

        Please provide feedback addressing the following points:
        1. Self-reflection:
           a) Evaluate the effectiveness of your approach. Was it aligned with the task requirements?
           b) Identify strengths in your execution. What worked well?
           c) Recognize areas for improvement. What could you have done differently?
           d) Assess the quality and relevance of your result. Does it contribute significantly to the overall task?

        2. Analysis of other agents:
           a) Compare your approach with those of other agents. What unique perspectives or methods did they bring?
           b) Identify any synergies or conflicts between your action and those of other agents.
           c) Evaluate the collective progress towards the task goal. Are all agents contributing effectively?

        3. Task-oriented reflection:
           a) Consider how well the current approaches align with the overall task objective.
           b) Suggest any adjustments in strategy or focus that might benefit the group's performance.

        4. Learning and adaptation:
           a) Based on this analysis, what key lessons can be drawn for future actions?
           b) Propose specific changes or improvements you plan to implement in your next action.

        5. Collaboration insights:
           a) Identify opportunities for better collaboration or task division among agents.
           b) Suggest ways to leverage the diverse strengths of different agents more effectively.

        Please structure your feedback clearly, addressing each point systematically. Be constructive, specific, and actionable in your analysis and suggestions.

        Your feedback MUST be in the following JSON format: (within 100 words):
        {{
            "self_reflection": "Your self-reflection here",
            "agent_analysis": "Your analysis of other agents here",
            "task_reflection": "Your task-oriented reflection here",
            "learning_adaptation": "Your learning and adaptation here",
            "collaboration_insights": "Your collaboration insights here"
        }}
        """
        response = await self.llm.aask(prompt, response_format={ "type": "json_object" })
        return response

class Execute(Action):
    name: str = "Execute"
    
    async def run(self, self_reflection: str, context: str):
        prompt = f"""
        Based on the current context and your self reflection, you will execute the task by choosing your action.

        Remember:
        1. Do not rush to give the final answer to the user requirement. 
        2. Your action could be an intermediate step towards the final answer.
        3. A valid answer is one that correctly addresses the user's requirement in the proper format, even if it might be improved or expanded later.
        4. You can provide a valid answer without it being the final one - further improvements, alternatives, or expansions can still be explored in subsequent steps.

        ## Current context (User Request, History of agents' actions):
        {context}

        ## Your self-reflection:
        {self_reflection}
        
        ## Your action:
        Provide your execution content. If your action involves writing code, please enclose it in triple backticks with the language specified (e.g., ```python).

        ## Is this a valid answer? (Yes/No):
        Determine if your execution provides a valid answer to the user's requirement. A valid answer correctly addresses the requirement in the proper format, even if it could potentially be improved or expanded further.

        You MUST respond in the following JSON format:
        {{
            "execution_content": "Your execution content here",
            "is_valid": boolean
        }}
        """
        response = await self.llm.aask(prompt, response_format={ "type": "json_object" })
        return response

class Update(Action):
    name: str = "Update"
    
    async def run(
            self,
            user_requirement: str,
            prompt: str, 
            previous_updates: List[Dict], 
            latest_execution: str = None, 
            latest_execution_result: str = None,
            feedback: str = None):
        context = "\n".join([f"Agent {update['agent']}: {update['profile']}" for update in previous_updates]) if previous_updates else "No previous agent updates available."
        latest_execution = latest_execution if latest_execution else "No latest execution available."
        latest_execution_result = latest_execution_result if latest_execution_result else "No latest execution result available."
        feedback = feedback if feedback else "No feedback available."

        full_prompt = f"""
        You will update your profile based on the user requirement, the current context, last Execution and result and your own self-reflection.

        ## User Requirement:
        {user_requirement}

        ## Latest Execution:
        {latest_execution}

        ## Latest Execution Result:
        {latest_execution_result}

        ## Feedback:
        {feedback}
        
        ## Previous agents' profiles updates:
        {context}
        
        {prompt}

        Please update your profile direclty below:
         """
        response = await self.llm.aask(full_prompt)
        return response
    
class Skip(Action):
    name: str = "Skip"
    
    async def run(self):
        return "Agent decided to skip this turn."