## game.py

import random
import copy
from typing import List, Tuple, Optional
from constants import Constants  # Importing constants from constants.py

class Game:
    def __init__(self, grid_size: int = Constants.GRID_SIZE):
        self._grid: List[List[int]] = [[0 for _ in range(grid_size)] for _ in range(grid_size)]
        self._current_score: int = 0
        self._high_score: int = self._load_high_score()
        self._last_move: Optional[Tuple[List[List[int]], int]] = None
        self._add_new_tile()
        self._add_new_tile()

    def _load_high_score(self) -> int:
        try:
            with open("highscore.txt", "r") as f:
                return int(f.read())
        except (IOError, ValueError):
            return 0

    def _save_high_score(self):
        with open("highscore.txt", "w") as f:
            f.write(str(self._high_score))

    def _add_new_tile(self):
        empty_tiles = [(i, j) for i in range(len(self._grid)) for j in range(len(self._grid[i])) if self._grid[i][j] == 0]
        if not empty_tiles:
            return
        i, j = random.choice(empty_tiles)
        self._grid[i][j] = 2 if random.random() < 0.9 else 4

    def _compress(self, grid: List[List[int]]) -> Tuple[List[List[int]], bool]:
        changed = False
        new_grid = [[0] * len(grid) for _ in range(len(grid))]
        for i in range(len(grid)):
            cnt = 0
            for j in range(len(grid[i])):
                if grid[i][j] != 0:
                    new_grid[i][cnt] = grid[i][j]
                    if cnt != j:
                        changed = True
                    cnt += 1
        return new_grid, changed

    def _merge(self, grid: List[List[int]]) -> Tuple[List[List[int]], bool]:
        changed = False
        for i in range(len(grid)):
            for j in range(len(grid[i])-1):
                if grid[i][j] == grid[i][j + 1] and grid[i][j] != 0:
                    grid[i][j] *= 2
                    grid[i][j + 1] = 0
                    self._current_score += grid[i][j]
                    changed = True
        return grid, changed

    def _reverse(self, grid: List[List[int]]) -> List[List[int]]:
        return [row[::-1] for row in grid]

    def _transpose(self, grid: List[List[int]]) -> List[List[int]]:
        return [list(row) for row in zip(*grid)]

    def move(self, direction: str) -> bool:
        if direction not in ['up', 'down', 'left', 'right']:
            return False

        self._last_move = (copy.deepcopy(self._grid), self._current_score)
        grid_changed = False

        if direction in ['up', 'down']:
            self._grid = self._transpose(self._grid)

        if direction in ['up', 'left']:
            self._grid, compressed = self._compress(self._grid)
            self._grid, merged = self._merge(self._grid)
            self._grid, _ = self._compress(self._grid)
            grid_changed = compressed or merged
        else:
            self._grid = self._reverse(self._grid)
            self._grid, compressed = self._compress(self._grid)
            self._grid, merged = self._merge(self._grid)
            self._grid, _ = self._compress(self._grid)
            self._grid = self._reverse(self._grid)
            grid_changed = compressed or merged

        if direction in ['up', 'down']:
            self._grid = self._transpose(self._grid)

        if grid_changed:
            self._add_new_tile()
            if self._current_score > self._high_score:
                self._high_score = self._current_score
                self._save_high_score()
            return True
        else:
            return False

    def undo(self) -> bool:
        if self._last_move is not None:
            self._grid, self._current_score = self._last_move
            self._last_move = None
            return True
        return False

    def check_game_over(self) -> bool:
        for i in range(len(self._grid)):
            for j in range(len(self._grid[i])):
                if self._grid[i][j] == 0:
                    return False
                if i < len(self._grid) - 1 and self._grid[i][j] == self._grid[i + 1][j]:
                    return False
                if j < len(self._grid[i]) - 1 and self._grid[i][j] == self._grid[i][j + 1]:
                    return False
        return True

    def reset_game(self):
        self._grid = [[0 for _ in range(len(self._grid))] for _ in range(len(self._grid))]
        self._current_score = 0
        self._last_move = None
        self._add_new_tile()
        self._add_new_tile()
