## logic.py

class Logic:
    def is_valid_move(self, position, board, board_size=15):
        """
        Check if a move is valid (i.e., within the board limits and on an empty spot).

        :param position: A tuple (row, col) representing the move's position.
        :param board: A 2D list representing the current state of the board.
        :param board_size: The size of the board (default is 15 for Gomoku).
        :return: True if the move is valid, False otherwise.
        """
        row, col = position
        return 0 <= row < board_size and 0 <= col < board_size and board[row][col] == '.'

    def get_winner(self, board, last_move, board_size=15, win_condition=5):
        """
        Determine if there is a winner based on the last move.

        :param board: A 2D list representing the current state of the board.
        :param last_move: A tuple (row, col) representing the last move made.
        :param board_size: The size of the board (default is 15 for Gomoku).
        :param win_condition: The number of consecutive marks needed to win (default is 5 for Gomoku).
        :return: The winner ('X' or 'O') if found, otherwise None.
        """
        if not last_move:
            return None

        last_row, last_col = last_move
        player = board[last_row][last_col]
        if player == '.':
            return None

        directions = [(0, 1), (1, 0), (1, 1), (1, -1)]  # horizontal, vertical, diagonal, anti-diagonal

        for delta in directions:
            consecutive_count = 1  # Start with the last move

            # Check in the positive direction
            for step in range(1, win_condition):
                row, col = last_row + delta[0] * step, last_col + delta[1] * step
                if 0 <= row < board_size and 0 <= col < board_size and board[row][col] == player:
                    consecutive_count += 1
                else:
                    break
                if consecutive_count == win_condition:
                    return player

            consecutive_count = 1  # Reset count before checking in the negative direction

            # Check in the negative direction
            for step in range(1, win_condition):
                row, col = last_row - delta[0] * step, last_col - delta[1] * step
                if 0 <= row < board_size and 0 <= col < board_size and board[row][col] == player:
                    consecutive_count += 1
                else:
                    break
                if consecutive_count == win_condition:
                    return player

        return None
