import tkinter as tk
import tkinter.messagebox as messagebox
from game import Game

class UI:
    def __init__(self, game: Game):
        """
        Initialize the UI class with a reference to the Game class.

        :param game: An instance of the Game class.
        """
        self.game = game
        self.root = tk.Tk()
        self.root.title("Gomoku")
        self.buttons = [[None for _ in range(self.game.board_size)] for _ in range(self.game.board_size)]

    def init_ui(self):
        """
        Initialize the user interface and start the main UI loop.
        """
        self._create_board()
        self.root.mainloop()

    def _create_board(self):
        """
        Create the board with buttons for each cell.
        """
        for row in range(self.game.board_size):
            for col in range(self.game.board_size):
                button = tk.Button(self.root, text='', font=('Arial', 12), height=2, width=4,
                                   command=lambda r=row, c=col: self._button_click(r, c))
                button.grid(row=row, column=col)
                self.buttons[row][col] = button

    def _button_click(self, row: int, col: int):
        """
        Handle button click event to make a move.

        :param row: The row index of the button clicked.
        :param col: The column index of the button clicked.
        """
        if self.game.make_move((row, col)):
            self.update_board()
            winner = self.game.check_winner()
            if winner:
                self.display_winner(winner)
            elif self.game.game_state == 'draw':
                self.display_draw()

    def update_board(self):
        """
        Update the board UI to reflect the current state of the game.
        """
        for row in range(self.game.board_size):
            for col in range(self.game.board_size):
                text = self.game.board[row][col]
                self.buttons[row][col].config(text=text)

    def display_winner(self, winner: str):
        """
        Display the winner and ask if players want to start a new game.

        :param winner: The winner of the game ('X' or 'O').
        """
        result = messagebox.askquestion("Game Over", f"Player {winner} wins! Start a new game?")
        if result == 'yes':
            self.game.start_game()
            self.update_board()

    def display_draw(self):
        """
        Display a message that the game is a draw and ask if players want to start a new game.
        """
        result = messagebox.askquestion("Game Over", "The game is a draw! Start a new game?")
        if result == 'yes':
            self.game.start_game()
            self.update_board()

    def display_history(self):
        """
        Display the history of moves made in the game.
        """
        history_str = '\n'.join([f"Player {player} moved to ({row}, {col})"
                                  for row, col, player in self.game.history])
        messagebox.showinfo("Game History", history_str)
