## database.py

import sqlite3
from typing import List, Tuple

class Database:
    def __init__(self, db_path: str = 'gomoku_history.db'):
        """
        Initialize the Database class with the path to the SQLite database.

        :param db_path: The file path to the SQLite database.
        """
        self.db_path = db_path
        self.connection = None
        self.initialize_database()

    def initialize_database(self):
        """
        Initialize the database by setting up the connection and creating tables.
        """
        try:
            self._connect()
            self._create_table()
        except sqlite3.Error as e:
            print(f"An error occurred during database initialization: {e}")
        finally:
            self._disconnect()

    def _create_table(self):
        """
        Create the game history table if it does not exist.
        """
        cursor = self.connection.cursor()
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS history (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                row INTEGER NOT NULL,
                col INTEGER NOT NULL,
                player CHAR(1) NOT NULL
            )
        ''')
        self.connection.commit()

    def _connect(self):
        """
        Establish a connection to the SQLite database.
        """
        if self.connection is None:
            self.connection = sqlite3.connect(self.db_path)

    def _disconnect(self):
        """
        Close the connection to the SQLite database.
        """
        if self.connection:
            self.connection.close()
            self.connection = None

    def save_move(self, position: Tuple[int, int], player: str):
        """
        Save a move to the database.

        :param position: A tuple (row, col) representing the move's position.
        :param player: The player ('X' or 'O') who made the move.
        """
        try:
            self._connect()
            cursor = self.connection.cursor()
            cursor.execute('''
                INSERT INTO history (row, col, player) VALUES (?, ?, ?)
            ''', (position[0], position[1], player))
            self.connection.commit()
        except sqlite3.Error as e:
            print(f"An error occurred while saving the move: {e}")
        finally:
            self._disconnect()

    def load_history(self) -> List[Tuple[int, int, str]]:
        """
        Load the game history from the database.

        :return: A list of tuples (row, col, player) representing the moves made.
        """
        try:
            self._connect()
            cursor = self.connection.cursor()
            cursor.execute('SELECT row, col, player FROM history ORDER BY id')
            history = cursor.fetchall()
            return history
        except sqlite3.Error as e:
            print(f"An error occurred while loading the history: {e}")
            return []
        finally:
            self._disconnect()

    def clear_history(self):
        """
        Clear the game history from the database.
        """
        try:
            self._connect()
            cursor = self.connection.cursor()
            cursor.execute('DELETE FROM history')
            self.connection.commit()
        except sqlite3.Error as e:
            print(f"An error occurred while clearing the history: {e}")
        finally:
            self._disconnect()
