#!/usr/bin/env python
# -*- coding: utf-8 -*-

from typing import NamedTuple, List, Tuple
from abc import ABC, abstractmethod
from contextlib import contextmanager
import ast
import builtins
import contextlib
import io
import json
import os
import signal
import sys
import tempfile
import types
import unittest
from typing import List
       
class ExecuteResult(NamedTuple):
    is_passing: bool
    feedback: str
    state: Tuple[bool]

class Executor(ABC):
    @abstractmethod
    def execute(self, func: str, tests: List[str], timeout: int = 5) -> ExecuteResult:
        ...

    @abstractmethod
    def evaluate(self, name: str, func: str, test: str, timeout: int = 5) -> bool:
        ...

import os
import json
from threading import Thread

def timeout_handler(_, __):
    raise TimeoutError()


def to_jsonl(dict_data, file_path):
    with open(file_path, 'a') as file:
        json_line = json.dumps(dict_data)
        file.write(json_line + os.linesep)


class PropagatingThread(Thread):
    def run(self):
        self.exc = None
        try:
            if hasattr(self, '_Thread__target'):
                # Thread uses name mangling prior to Python 3.
                self.ret = self._Thread__target(*self._Thread__args, **self._Thread__kwargs)
            else:
                self.ret = self._target(*self._args, **self._kwargs)
        except BaseException as e:
            self.exc = e

    def join(self, timeout=None):
        super(PropagatingThread, self).join(timeout)
        if self.exc:
            raise self.exc
        return self.ret
    

def function_with_timeout(func, args, timeout):
    result_container = []

    def wrapper():
        result_container.append(func(*args))

    thread = PropagatingThread(target=wrapper)
    thread.start()
    thread.join(timeout)

    if thread.is_alive():
        raise TimeoutError()
    else:
        return result_container[0]
    

@contextlib.contextmanager
def create_tempdir():
    with tempfile.TemporaryDirectory() as dirname:
        with chdir(dirname):
            yield dirname

@contextlib.contextmanager
def chdir(dirname):
    curdir = os.getcwd()
    try:
        os.chdir(dirname)
        yield
    finally:
        os.chdir(curdir)

@contextlib.contextmanager
def swallow_io():
    stream = io.StringIO()
    with contextlib.redirect_stdout(stream), contextlib.redirect_stderr(stream):
        yield

@contextlib.contextmanager
def time_limit(seconds):
    def signal_handler(signum, frame):
        raise TimeoutError("Test execution timed out")
    
    signal.signal(signal.SIGALRM, signal_handler)
    signal.alarm(seconds)
    try:
        yield
    finally:
        signal.alarm(0)

        
class UniPyExecutor(Executor):
    def execute(self, func: str, tests: List[str], timeout: int = 5, verbose: bool = True) -> ExecuteResult:
        # Combine function code and test cases
        full_code = func + "\n" + "\n".join(tests)
        
        try:
            with create_tempdir():
                module_name = "__test__"
                new_module = types.ModuleType(module_name)
                
                new_module.__dict__.update({
                    '__builtins__': __builtins__,
                    '__file__': f"{module_name}.py",
                    '__package__': None,
                    '__doc__': None,
                    'sys': sys,
                    'os': os,
                    'environ': os.environ,
                })
                
                exec(compile(full_code, f"{module_name}.py", 'exec'), new_module.__dict__)
                sys.modules[module_name] = new_module
                TestCases = getattr(new_module, 'TestCases')
                
                loader = unittest.TestLoader()
                suite = loader.loadTestsFromTestCase(TestCases)
                test_result = unittest.TestResult()
                
                with swallow_io(), time_limit(seconds=timeout):
                    suite.run(test_result)
                
                is_passing = test_result.wasSuccessful()
                feedback = self._generate_feedback(test_result)
                state = self._generate_state(test_result, len(tests))
                
        except Exception as e:
            is_passing = False
            feedback = f"An error occurred during test execution: {str(e)}"
            state = [False] * len(tests)
        
        return is_passing, feedback, tuple(state)

    def _generate_feedback(self, test_result: unittest.TestResult) -> str:
        if test_result.wasSuccessful():
            return "All tests passed successfully."
        
        feedback = []
        for failure in test_result.failures:
            feedback.append(f"FAIL: {failure[0]}\n{failure[1]}")
        for error in test_result.errors:
            feedback.append(f"ERROR: {error[0]}\n{error[1]}")
        
        return "\n\n".join(feedback)

    def _generate_state(self, test_result: unittest.TestResult, num_tests: int) -> List[bool]:
        state = [True] * num_tests
        for failure in test_result.failures + test_result.errors:
            test_name = failure[0].id().split('.')[-1]
            test_index = int(test_name.split('_')[-1]) - 1
            if 0 <= test_index < num_tests:
                state[test_index] = False
        return state

    def evaluate(self, name: str, func: str, test: str, timeout: int = 5) -> bool:
        """
        Evaluates the implementation on Human-Eval Python.
        """
        code = f"{func}\n{test}\ncheck({name})"
        try:
            with time_limit(seconds=timeout):
                with create_tempdir():
                    module_name = "__eval__"
                    new_module = types.ModuleType(module_name)
                    
                    new_module.__dict__.update({
                        '__builtins__': __builtins__,
                        '__file__': f"{module_name}.py",
                        '__package__': None,
                        '__doc__': None,
                        'sys': sys,
                        'os': os,
                        'environ': os.environ,
                    })
                    
                    exec(compile(code, f"{module_name}.py", 'exec'), new_module.__dict__)
            return True
        except Exception:
            return False
