## game.py

import pygame
import random
from config import Config

class Game:
    def __init__(self):
        # Initialize game configuration
        config = Config()
        self.width = config.width
        self.height = config.height
        self.food_size = config.food_size
        self.snake_size = config.snake_size
        self.colors = config.colors

        # Initialize game state
        self.score = 0
        self.high_score = 0
        self.game_over = False
        self.snake = [(self.width // 2, self.height // 2)]  # Use integer division for coordinates
        self.food = None
        self.direction = (0, -self.snake_size)  # Start moving upwards

        # Initialize pygame components
        if not pygame.get_init():
            raise SystemExit("Failed to initialize Pygame")
        pygame.init()
        self.font = pygame.font.SysFont('arial', 35)
        self.screen = pygame.display.set_mode((self.width, self.height))
        pygame.display.set_caption('Snake Game')

        # Place the first food on the screen
        self._place_food()

    def run(self):
        clock = pygame.time.Clock()

        while not self.game_over:
            self.handle_events()
            self.update()
            self.draw()
            clock.tick(10)  # Control the game speed

        self.reset()

    def handle_events(self):
        key_directions = {
            pygame.K_UP: (0, -self.snake_size),
            pygame.K_DOWN: (0, self.snake_size),
            pygame.K_LEFT: (-self.snake_size, 0),
            pygame.K_RIGHT: (self.snake_size, 0),
        }
        for event in pygame.event.get():
            if event.type == pygame.QUIT:
                self.game_over = True
            elif event.type == pygame.KEYDOWN:
                new_direction = key_directions.get(event.key)
                if new_direction:
                    # Check if the new direction is not the opposite of the current direction
                    if (new_direction[0] != -self.direction[0] or
                            new_direction[1] != -self.direction[1]):
                        self.direction = new_direction

    def update(self):
        # Move the snake
        head_x, head_y = self.snake[0]
        new_head = (head_x + self.direction[0], head_y + self.direction[1])
        self.snake.insert(0, new_head)

        # Check if the snake has eaten the food
        if self.snake[0] == self.food:
            self.score += 1
            self.high_score = max(self.high_score, self.score)
            self._place_food()
        else:
            self.snake.pop()

        # Check for collisions with walls or self
        if (self.snake[0][0] < 0 or self.snake[0][0] >= self.width or
                self.snake[0][1] < 0 or self.snake[0][1] >= self.height or
                self.snake[0] in self.snake[1:]):
            self.game_over = True

    def draw(self):
        self.screen.fill(self.colors['white'])

        # Draw the snake
        for segment in self.snake:
            pygame.draw.rect(self.screen, self.colors['green'],
                             pygame.Rect(segment[0], segment[1], self.snake_size, self.snake_size))

        # Draw the food
        pygame.draw.rect(self.screen, self.colors['red'],
                         pygame.Rect(self.food[0], self.food[1], self.food_size, self.food_size))

        # Draw the score
        score_text = self.font.render(f'Score: {self.score}', True, self.colors['blue'])
        self.screen.blit(score_text, [0, 0])

        pygame.display.update()

    def reset(self):
        self.score = 0
        self.game_over = False
        self.snake = [(self.width // 2, self.height // 2)]  # Use integer division for coordinates
        self.direction = (0, -self.snake_size)
        self._place_food()

    def _place_food(self):
        # Place food on the screen on a random position
        while True:
            x = random.randint(0, (self.width - self.food_size) // self.food_size) * self.food_size
            y = random.randint(0, (self.height - self.food_size) // self.food_size) * self.food_size
            self.food = (x, y)
            if self.food not in self.snake:
                break
