# dataset/lidar_noise.py
import numpy as np

def rolling_shutter(xyz, ego_vel, dt_ms: float):
    """xyz: (N,3), ego_vel: (3,), dt_ms=Δt (0 → skip)"""
    if dt_ms == 0:
        return xyz
    az = np.arctan2(xyz[:, 1], xyz[:, 0])               # -π~π
    frac = ((az + 2*np.pi) % (2*np.pi)) / (2*np.pi)      # 0~1
    shift = ego_vel * (dt_ms * 1e-3) * frac[:, None]
    return xyz + shift

def angle_jitter(xyz, sigma_deg: float, origin=np.zeros(3)):
    """σ=0 → skip; isotropic az/el Gaussian jitter in deg."""
    if sigma_deg == 0:
        return xyz
    v = xyz - origin
    r = np.linalg.norm(v, axis=1)
    az = np.arctan2(v[:, 1], v[:, 0]) + np.random.normal(scale=np.deg2rad(sigma_deg), size=v.shape[0])
    el = np.arcsin(v[:, 2] / r)       + np.random.normal(scale=np.deg2rad(sigma_deg), size=v.shape[0])
    return np.stack([r*np.cos(el)*np.cos(az),
                     r*np.cos(el)*np.sin(az),
                     r*np.sin(el)], 1) + origin

def ray_drop(xyz, remis, labels, p: float):
    """Randomly drop p% of rays **consistently** across xyz / intensity / label."""
    if p == 0:
        return xyz, remis, labels
    keep = np.random.rand(len(xyz)) > p
    return xyz[keep], remis[keep], labels[keep]
