import gymnasium as gym
import numpy as np
import time
import torch as th
import wandb

from minigrid.core.world_object import Door, Key, Wall, Ball, Box, Goal
from matplotlib import pyplot as plt

from src.algo.buffers.ppo_buffer import PPORolloutBuffer
from src.utils.loggers import StatisticsLogger, LocalLogger
from src.utils.common_func import set_random_seed
from src.utils.enum_types import ModelType, EnvSrc

from stable_baselines3.common.base_class import BaseAlgorithm
from stable_baselines3.common.callbacks import BaseCallback
from stable_baselines3.common.policies import ActorCriticPolicy, BasePolicy
from stable_baselines3.common.type_aliases import GymEnv, MaybeCallback, Schedule
from stable_baselines3.common.utils import obs_as_tensor, safe_mean
from stable_baselines3.common.vec_env import VecEnv

from typing import Any, Dict, Optional, Tuple, Type, Union
from gymnasium import spaces


class PPORollout(BaseAlgorithm):

    def __init__(
        self,
        policy: Union[str, Type[ActorCriticPolicy]],
        env: Union[GymEnv, str],
        run_id: int,
        learning_rate: Union[float, Schedule],
        n_steps: int,
        batch_size: int,
        gamma: float,
        gae_lambda: float,
        ent_coef: float,
        pg_coef: float,
        vf_coef: float,
        int_rew_source: ModelType,
        int_rew_coef: float,
        int_rew_norm : int,
        int_rew_momentum: Optional[float],
        int_rew_eps : float,
        int_rew_clip : float,
        adv_momentum : float,
        image_noise_scale : float,
        enable_plotting : int,
        can_see_walls : int,
        ext_rew_coef: float,
        adv_norm: int,
        adv_eps: float,
        max_grad_norm: float,
        use_sde: bool,
        sde_sample_freq: int,
        # policy_base: Type[BasePolicy] = ActorCriticPolicy,
        policy_kwargs: Optional[Dict[str, Any]] = None,
        verbose: int = 0,
        seed: Optional[int] = None,
        device: Union[th.device, str] = "auto",
        _init_setup_model: bool = True,
        env_source: Optional[EnvSrc] = None,
        env_render: Optional[int] = None,
        fixed_seed: Optional[int] = None,
        plot_interval: int = 10,
        plot_colormap: str = "Blues",
        log_explored_states: Optional[int] = None,
        local_logger: Optional[LocalLogger] = None,
        use_wandb: bool = False,
    ):
        super(PPORollout, self).__init__(
            policy=policy,
            env=env,
            learning_rate=learning_rate,
            policy_kwargs=policy_kwargs,
            verbose=verbose,
            device=device,
            use_sde=use_sde,
            sde_sample_freq=sde_sample_freq,
            support_multi_env=True,
            seed=seed,
        )
        self.run_id = run_id
        self.n_steps = n_steps
        self.batch_size = batch_size
        self.gamma = gamma
        self.gae_lambda = gae_lambda
        self.ent_coef = ent_coef
        self.pg_coef = pg_coef
        self.vf_coef = vf_coef
        self.max_grad_norm = max_grad_norm
        self.num_timesteps = 0
        self.int_rew_source = int_rew_source
        self.int_rew_coef = int_rew_coef
        self.int_rew_norm = int_rew_norm
        self.int_rew_eps = int_rew_eps
        self.adv_momentum = adv_momentum
        self.int_rew_clip = int_rew_clip
        self.image_noise_scale = image_noise_scale
        self.enable_plotting = enable_plotting
        self.can_see_walls = can_see_walls
        self.int_rew_momentum = int_rew_momentum
        self.ext_rew_coef = ext_rew_coef
        self.adv_norm = adv_norm
        self.adv_eps = adv_eps
        self.env_source = env_source
        self.env_render = env_render
        self.fixed_seed = fixed_seed
        self.plot_interval = plot_interval
        self.plot_colormap = plot_colormap
        self.log_explored_states = log_explored_states
        self.local_logger = local_logger
        self.use_wandb = use_wandb

        if _init_setup_model:
            self._setup_model()

    def _setup_model(self) -> None:
        self._setup_lr_schedule()
        set_random_seed(self.seed)
        self.policy = self.policy_class(  # pytype:disable=not-instantiable
            self.observation_space,
            self.action_space,
            self.lr_schedule,
            use_sde=self.use_sde,
            **self.policy_kwargs  # pytype:disable=not-instantiable
        )
        self.policy = self.policy.to(self.device)
        self.ppo_rollout_buffer = PPORolloutBuffer(
            self.n_steps,
            self.observation_space,
            self.action_space,
            self.device,
            gamma=self.gamma,
            gae_lambda=self.gae_lambda,
            n_envs=self.n_envs,
            features_dim=self.policy.features_dim,
            dim_policy_traj=self.policy.dim_policy_features,
            dim_model_traj=self.policy.dim_model_features,
            int_rew_coef=self.int_rew_coef,
            ext_rew_coef=self.ext_rew_coef,
            int_rew_norm=self.int_rew_norm,
            int_rew_clip=self.int_rew_clip,
            int_rew_eps=self.int_rew_eps,
            adv_momentum=self.adv_momentum,
            adv_norm=self.adv_norm,
            adv_eps=self.adv_eps,
            gru_layers=self.policy.gru_layers,
            int_rew_momentum=self.int_rew_momentum,
            use_status_predictor=self.policy.use_status_predictor,
        )


    def on_training_start(self):
        if isinstance(self._last_obs, Dict):
            self._last_obs = self._last_obs["rgb"]

        if self.env_source in [EnvSrc.MiniGrid, EnvSrc.NGC]:
            # Set advanced options for MiniGrid envs
            self.env.can_see_walls = self.can_see_walls
            self.env.image_noise_scale = self.image_noise_scale
            self.env.image_rng = np.random.default_rng(seed=self.run_id + 1313)

            # Initialize seeds for each MiniGrid env
            np.random.seed(self.run_id)
            seeds = np.random.rand(self.n_envs)
            seeds = [int(s * 0x7fffffff) for s in seeds]
            np.random.seed(self.run_id)
            # self.env.set_seeds(seeds)
            self.env.waiting = True
            for i in range(self.n_envs):
                self.env.send_reset(env_id=i, seed=seeds[i])
            for i in range(self.n_envs):
                self._last_obs[i] = self.env.recv_obs(env_id=i)
            self.env.waiting = False

            # Init variables for logging
            self.width = self.env.get_attr('width')[0]
            self.height = self.env.get_attr('height')[0]
            self.global_visit_counts = np.zeros([self.width, self.height], dtype=np.int32)
            self.global_reward_map_maxs = np.zeros([self.width, self.height], dtype=np.float64)
            self.global_reward_map_sums = np.zeros([self.width, self.height], dtype=np.float64)
            self.global_reward_map_nums = np.zeros([self.width, self.height], dtype=np.int32)
            self.global_value_map_sums = np.zeros([self.width, self.height], dtype=np.float64)
            self.global_value_map_nums = np.zeros([self.width, self.height], dtype=np.int32)

        self.global_episode_rewards = np.zeros(self.n_envs, dtype=np.float32)
        self.global_episode_intrinsic_rewards = np.zeros(self.n_envs, dtype=np.float32)

        self.global_episode_unique_states = np.zeros(self.n_envs, dtype=np.float32)
        self.global_episode_visited_states = [dict() for _ in range(self.n_envs)]
        self.global_lifelong_unique_states = 0
        self.global_lifelong_visited_states = dict()
        self.global_episode_visited_positions = [dict() for _ in range(self.n_envs)]
        self.global_episode_visited_pos_sum = np.zeros(self.n_envs, dtype=np.float32)

        self.global_episode_steps = np.zeros(self.n_envs, dtype=np.int32)
        if self.policy.use_status_predictor:
            self.global_has_keys = np.zeros(self.n_envs, dtype=np.int32)
            self.global_open_doors = np.zeros(self.n_envs, dtype=np.int32)
            self.curr_key_status = np.zeros(self.n_envs, dtype=np.float32)
            self.curr_door_status = np.zeros(self.n_envs, dtype=np.float32)
            self.curr_target_dists = np.zeros((self.n_envs, 3), dtype=np.float32)
        else:
            self.global_has_keys = None
            self.global_open_doors = None
            self.curr_key_status = None
            self.curr_door_status = None
            self.curr_target_dists = None

        self.episodic_obs_emb_history = [None for _ in range(self.n_envs)]
        self.episodic_trj_emb_history = [None for _ in range(self.n_envs)]

        if self.int_rew_source in [ModelType.DEIR, ModelType.PlainDiscriminator]:
            self.policy.int_rew_model.init_obs_queue(self._last_obs)

        def float_zeros(tensor_shape):
            return th.zeros(tensor_shape, device=self.device, dtype=th.float32)

        self._last_policy_mems = float_zeros([self.n_envs, self.policy.gru_layers, self.policy.dim_policy_features])
        self._last_model_mems = float_zeros([self.n_envs, self.policy.gru_layers, self.policy.dim_model_features])


    def init_on_rollout_start(self):
        # Log statistics data per each rollout
        self.rollout_stats = StatisticsLogger(mode='rollout')
        self.rollout_done_episodes = 0
        self.rollout_done_episode_steps = 0
        self.rollout_sum_rewards = 0
        self.rollout_episode_unique_states = 0
        self.rollout_done_episode_unique_states = 0


    def log_before_transition(self, values):
        if self.env_source == EnvSrc.MiniGrid:
            self._last_state_hash_vals = self.env.env_method('hash')

        # Update Key and Door Status
        agent_positions = None
        if self.policy.use_status_predictor:
            agent_positions = np.array(self.env.get_attr('agent_pos'))
            agent_carryings = self.env.get_attr('carrying')
            env_grids = self.env.get_attr('grid')

            self.curr_door_pos = np.zeros((self.n_envs, 2), dtype=np.int32)
            self.curr_key_pos = np.copy(agent_positions).reshape(self.n_envs, 2)
            self.curr_goal_pos = np.zeros((self.n_envs, 2), dtype=np.int32)
            for env_id in range(self.n_envs):
                # The only possible carrying in DoorKey is the key
                self.global_has_keys[env_id] = int(isinstance(agent_carryings[env_id], Key))

                # Door, Key, Goal positions
                for env_id, grid in enumerate(env_grids[env_id].grid):
                    col = env_id % self.width
                    row = env_id // self.width
                    if isinstance(grid, Door):
                        self.curr_door_pos[env_id] = np.array((col, row))
                        self.global_open_doors[env_id] = int(grid.is_open)
                    elif isinstance(grid, Key):
                        self.curr_key_pos[env_id] = np.array((col, row))
                    elif isinstance(grid, Goal):
                        self.curr_goal_pos[env_id] = np.array((col, row))

            self.curr_key_status = np.copy(self.global_has_keys)
            self.curr_door_status = np.copy(self.global_open_doors)
            self.rollout_stats.add(
                key_status=np.mean(self.global_has_keys),
                door_status=np.mean(self.global_open_doors),
            )

        # Update agent position and visit count
        if self.policy.use_status_predictor or self.enable_plotting:
            if agent_positions is None:
                agent_positions = np.array(self.env.get_attr('agent_pos'))
            for i in range(self.n_envs):
                c, r = agent_positions[i]
                self.global_visit_counts[r, c] += 1
                self.global_value_map_sums[r, c] += values[i].item()
                self.global_value_map_nums[r, c] += 1

            # Current agent position
            self.curr_agent_pos = np.copy(agent_positions)

            # Define the target of position prediction loss
            if self.policy.use_status_predictor:
                # Manhattan Distance to the Door
                key_dists = np.abs(self.curr_agent_pos - self.curr_key_pos)
                key_dists = np.sum(key_dists, axis=1) / (self.width + self.height)
                door_dists = np.abs(self.curr_agent_pos - self.curr_door_pos)
                door_dists = np.sum(door_dists, axis=1) / (self.width + self.height)
                goal_dists = np.abs(self.curr_agent_pos - self.curr_goal_pos)
                goal_dists = np.sum(goal_dists, axis=1) / (self.width + self.height)
                self.curr_target_dists = np.stack([key_dists, door_dists, goal_dists], axis=1)


    def log_after_transition(self, rewards, intrinsic_rewards):
        self.global_episode_rewards += rewards
        self.global_episode_intrinsic_rewards += intrinsic_rewards
        self.global_episode_steps += 1

        # Logging episodic/lifelong visited states, reward map
        if self.log_explored_states:
            # 0 - Not to log
            # 1 - Log both episodic and lifelong states
            # 2 - Log episodic visited states only
            if self.env_source == EnvSrc.MiniGrid:
                agent_positions = np.array(self.env.get_attr('agent_pos'))
                for env_id in range(self.n_envs):
                    c, r = agent_positions[env_id]

                    # count the visited positions
                    pos = c * self.width + r
                    pos_visit_count = self.global_episode_visited_positions[env_id]
                    if pos not in pos_visit_count:
                        pos_visit_count[pos] = 1
                        self.global_episode_visited_pos_sum[env_id] += 1
                    else:
                        pos_visit_count[pos] += 1

                    env_hash = self._last_state_hash_vals[env_id]
                    if env_hash in self.global_episode_visited_states[env_id]:
                        self.global_episode_visited_states[env_id][env_hash] += 1
                    else:
                        self.global_episode_visited_states[env_id][env_hash] = 1
                        self.global_episode_unique_states[env_id] += 1
                        self.rollout_episode_unique_states += 1

                    if self.log_explored_states == 1:
                        if env_hash in self.global_lifelong_visited_states:
                            self.global_lifelong_visited_states[env_hash] += 1
                        else:
                            self.global_lifelong_visited_states[env_hash] = 1
                            self.global_lifelong_unique_states += 1

                    if self.enable_plotting:
                        self.global_reward_map_maxs[r, c] = np.maximum(
                            self.global_reward_map_maxs[r, c],
                            intrinsic_rewards[env_id]
                        )
                        self.global_reward_map_sums[r, c] += intrinsic_rewards[env_id]
                        self.global_reward_map_nums[r, c] += 1

            elif self.env_source == EnvSrc.ProcGen:
                for env_id in range(self.n_envs):
                    # In Procgen games, the counted "states" are observations
                    env_hash = tuple(self._last_obs[env_id].reshape(-1).tolist())

                    if env_hash in self.global_episode_visited_states[env_id]:
                        self.global_episode_visited_states[env_id][env_hash] += 1
                    else:
                        self.global_episode_visited_states[env_id][env_hash] = 1
                        self.global_episode_unique_states[env_id] += 1
                        self.rollout_episode_unique_states += 1

                    if self.log_explored_states == 1:
                        if env_hash in self.global_lifelong_visited_states:
                            self.global_lifelong_visited_states[env_hash] += 1
                        else:
                            self.global_lifelong_visited_states[env_hash] = 1
                            self.global_lifelong_unique_states += 1


    def clear_on_episode_end(self, dones, policy_mems, model_mems):
        for env_id in range(self.n_envs):
            if dones[env_id]:
                if policy_mems is not None: policy_mems[env_id] *= 0.0
                if model_mems is not None: model_mems[env_id] *= 0.0
                self.episodic_obs_emb_history[env_id] = None
                self.episodic_trj_emb_history[env_id] = None
                self.rollout_sum_rewards += self.global_episode_rewards[env_id]
                self.rollout_done_episode_steps += self.global_episode_steps[env_id]
                self.rollout_done_episode_unique_states += self.global_episode_unique_states[env_id]
                self.rollout_done_episodes += 1
                self.global_episode_rewards[env_id] = 0
                self.global_episode_intrinsic_rewards[env_id] = 0
                self.global_episode_unique_states[env_id] = 0
                self.global_episode_visited_states[env_id] = dict()  # logging use
                self.global_episode_visited_positions[env_id] = dict()  # logging use
                self.global_episode_visited_pos_sum[env_id] = 0  # logging use
                self.global_episode_steps[env_id] = 0
                if self.policy.use_status_predictor:
                    self.global_has_keys[env_id] = 0
                    self.global_open_doors[env_id] = 0
                    self.curr_key_status[env_id] = 0
                    self.curr_door_status[env_id] = 0


    def log_on_rollout_end(self, log_interval):
        # running average of reward achieve ratio
        self.rew_achieve = 0.95*self.rew_achieve + 0.05*self.rollout_sum_rewards / (self.rollout_done_episodes + 1e-8)

        if log_interval is not None and self.iteration % log_interval == 0:
            log_data = {
                "iterations": self.iteration,
                "time/fps": int(self.num_timesteps / (time.time() - self.start_time)),
                "time/time_elapsed": int(time.time() - self.start_time),
                "time/total_timesteps": self.num_timesteps,
                "rollout/ep_rew_mean": self.rollout_sum_rewards / (self.rollout_done_episodes + 1e-8),
                "rollout/ep_len_mean": self.rollout_done_episode_steps / (self.rollout_done_episodes + 1e-8),
                # unique states / positions
                "rollout/ep_unique_states": self.rollout_done_episode_unique_states / (
                            self.rollout_done_episodes + 1e-8),
                "rollout/ll_unique_states": self.global_lifelong_unique_states,
                "rollout/ep_unique_states_per_step": self.rollout_episode_unique_states / (
                            self.ppo_rollout_buffer.buffer_size * self.n_envs),
                "rollout/ll_unique_states_per_step": self.global_lifelong_unique_states / self.num_timesteps,
                # intrinsic rewards
                "rollout/int_rew_coef": self.ppo_rollout_buffer.int_rew_coef,
                "rollout/int_rew_buffer_mean": self.ppo_rollout_buffer.int_rew_mean,
                "rollout/int_rew_buffer_std": self.ppo_rollout_buffer.int_rew_std,
            }

            if len(self.ep_info_buffer) > 0 and len(self.ep_info_buffer[0]) > 0:
                log_data.update({
                    "rollout/ep_info_rew_mean": safe_mean([ep_info["r"] for ep_info in self.ep_info_buffer]),
                    "rollout/ep_info_len_mean": safe_mean([ep_info["l"] for ep_info in self.ep_info_buffer]),
                })
            else:
                log_data.update({
                    "rollout/ep_info_rew_mean": 0.0,
                })

            if self.int_rew_coef > 0:
                log_data.update({
                    "rollout/int_rew_mean": np.mean(self.ppo_rollout_buffer.intrinsic_rewards),
                    "rollout/int_rew_std": np.std(self.ppo_rollout_buffer.intrinsic_rewards),
                    "rollout/pos_int_rew_mean": np.maximum(self.ppo_rollout_buffer.intrinsic_rewards, 0.0).mean(),
                    "rollout/neg_int_rew_mean": np.minimum(self.ppo_rollout_buffer.intrinsic_rewards, 0.0).mean(),
                })

            if self.adv_norm > 0:
                log_data.update({
                    "rollout/adv_mean": np.mean(self.ppo_rollout_buffer.adv_mean),
                    "rollout/adv_std": np.std(self.ppo_rollout_buffer.adv_std),
                })

            # Update with other stats
            log_data.update(self.rollout_stats.to_dict())

            # Logging with wandb
            if self.use_wandb:
                wandb.log(log_data)
            # Logging with local logger
            if self.local_logger is not None:
                self.local_logger.write(log_data, log_type='rollout')

        # Log visualized data
        if self.enable_plotting and self.use_wandb:
            if self.iteration > 0 and self.iteration % self.plot_interval == 0:
                # Plot visit counts
                _, ax = plt.subplots()
                im = ax.imshow(
                    self.global_visit_counts,
                    cmap=self.plot_colormap,
                    interpolation='nearest')
                ax.figure.colorbar(im, ax=ax)
                plt.title(f'visit counts (iters:{self.iteration}, steps:{self.num_timesteps})')
                wandb.log({"visit counts": plt})
                plt.close()

                # Plot reward map
                _, ax = plt.subplots()
                im = ax.imshow(
                    self.global_reward_map_sums / (self.global_reward_map_nums + 1e-8),
                    cmap=self.plot_colormap,
                    interpolation='nearest')
                ax.figure.colorbar(im, ax=ax)
                plt.title(f'reward map (iters:{self.iteration}, steps:{self.num_timesteps})')
                wandb.log({"reward map": plt})
                plt.close()

                # Plot sum reward map
                _, ax = plt.subplots()
                im = ax.imshow(
                    self.global_reward_map_sums,
                    cmap=self.plot_colormap,
                    interpolation='nearest')
                ax.figure.colorbar(im, ax=ax)
                plt.title(f'sum reward map (iters:{self.iteration}, steps:{self.num_timesteps})')
                wandb.log({"sum reward map": plt})
                plt.close()

                # Plot max reward map
                _, ax = plt.subplots()
                im = ax.imshow(
                    self.global_reward_map_maxs,
                    cmap=self.plot_colormap,
                    interpolation='nearest')
                ax.figure.colorbar(im, ax=ax)
                plt.title(f'max reward map (iters:{self.iteration}, steps:{self.num_timesteps})')
                wandb.log({"max reward map": plt})
                plt.close()

                # Plot value map
                _, ax = plt.subplots()
                im = ax.imshow(
                    self.global_value_map_sums / (self.global_value_map_nums + 1e-8),
                    cmap=self.plot_colormap,
                    interpolation='nearest')
                ax.figure.colorbar(im, ax=ax)
                plt.title(f'value map (iters:{self.iteration}, steps:{self.num_timesteps})')
                wandb.log({"value map": plt})
                plt.close()

                # Clear counts
                self.global_visit_counts = np.zeros([self.width, self.height], dtype=np.int32)
                self.global_reward_map_maxs = np.zeros([self.width, self.height], dtype=np.float64)
                self.global_reward_map_sums = np.zeros([self.width, self.height], dtype=np.float64)
                self.global_reward_map_nums = np.zeros([self.width, self.height], dtype=np.int32)
                self.global_value_map_sums = np.zeros([self.width, self.height], dtype=np.float64)
                self.global_value_map_nums = np.zeros([self.width, self.height], dtype=np.int32)


    def create_intrinsic_rewards(self, new_obs, actions, dones):
        if self.int_rew_source == ModelType.NoModel:
            intrinsic_rewards = np.zeros([self.n_envs], dtype=float)
            model_mems = None
            return intrinsic_rewards, model_mems

        # Prepare input tensors for IR generation
        with th.no_grad():
            curr_obs_tensor = obs_as_tensor(self._last_obs, self.device)
            next_obs_tensor = obs_as_tensor(new_obs, self.device)
            curr_act_tensor = th.as_tensor(actions, dtype=th.int64, device=self.device)
            done_tensor = th.as_tensor(dones, dtype=th.int64, device=self.device)

            if self.policy.use_model_rnn:
                last_model_mem_tensor = self._last_model_mems
                if self.int_rew_source in [ModelType.RND, ModelType.NGU, ModelType.NovelD]:
                    if self.policy.rnd_use_policy_emb:
                        last_model_mem_tensor = self._last_policy_mems
            else:
                last_model_mem_tensor = None

            if self.policy.use_status_predictor:
                key_status_tensor = th.as_tensor(self.curr_key_status, dtype=th.int64, device=self.device)
                door_status_tensor = th.as_tensor(self.curr_door_status, dtype=th.int64, device=self.device)
                target_dists_tensor = th.as_tensor(self.curr_target_dists, dtype=th.float32, device=self.device)
            else:
                key_status_tensor = None
                door_status_tensor = None
                target_dists_tensor = None

        # DEIR / Plain discriminator model
        if self.int_rew_source in [ModelType.DEIR, ModelType.PlainDiscriminator]:
            intrinsic_rewards, model_mems = self.policy.int_rew_model.get_intrinsic_rewards(
                curr_obs=curr_obs_tensor,
                next_obs=next_obs_tensor,
                last_mems=last_model_mem_tensor,
                obs_history=self.episodic_obs_emb_history,
                trj_history=self.episodic_trj_emb_history,
                curr_dones=done_tensor,
                plain_dsc=bool(self.int_rew_source == ModelType.PlainDiscriminator),
            )
            # Insert obs into the Discriminator's obs queue
            # Algorithm A2 in the Technical Appendix
            if self.int_rew_source in [ModelType.DEIR, ModelType.PlainDiscriminator]:
                self.policy.int_rew_model.update_obs_queue(
                    iteration=self.iteration,
                    intrinsic_rewards=intrinsic_rewards,
                    ir_mean=self.ppo_rollout_buffer.int_rew_stats.mean,
                    new_obs=new_obs,
                    stats_logger=self.rollout_stats
                )
        # Plain forward / inverse model
        elif self.int_rew_source in [ModelType.PlainForward, ModelType.PlainInverse]:
            intrinsic_rewards, model_mems = self.policy.int_rew_model.get_intrinsic_rewards(
                curr_obs=curr_obs_tensor,
                next_obs=next_obs_tensor,
                last_mems=last_model_mem_tensor,
                curr_act=curr_act_tensor,
                curr_dones=done_tensor,
                obs_history=self.episodic_obs_emb_history,
                key_status=key_status_tensor,
                door_status=door_status_tensor,
                target_dists=target_dists_tensor,
                stats_logger=self.rollout_stats
            )
        # ICM
        elif self.int_rew_source == ModelType.ICM:
            intrinsic_rewards, model_mems = self.policy.int_rew_model.get_intrinsic_rewards(
                curr_obs=curr_obs_tensor,
                next_obs=next_obs_tensor,
                last_mems=last_model_mem_tensor,
                curr_act=curr_act_tensor,
                curr_dones=done_tensor,
                stats_logger=self.rollout_stats
            )
        # RND
        elif self.int_rew_source == ModelType.RND:
            intrinsic_rewards, model_mems = self.policy.int_rew_model.get_intrinsic_rewards(
                curr_obs=curr_obs_tensor,
                last_mems=last_model_mem_tensor,
                curr_dones=done_tensor,
                stats_logger=self.rollout_stats
            )
        # NGU
        elif self.int_rew_source == ModelType.NGU:
            intrinsic_rewards, model_mems = self.policy.int_rew_model.get_intrinsic_rewards(
                curr_obs=curr_obs_tensor,
                next_obs=next_obs_tensor,
                last_mems=last_model_mem_tensor,
                curr_act=curr_act_tensor,
                curr_dones=done_tensor,
                obs_history=self.episodic_obs_emb_history,
                stats_logger=self.rollout_stats
            )
        # NovelD
        elif self.int_rew_source == ModelType.NovelD:
            return self.policy.int_rew_model.get_intrinsic_rewards(
                curr_obs=curr_obs_tensor,
                next_obs=next_obs_tensor,
                last_mems=last_model_mem_tensor,
                curr_dones=done_tensor,
                stats_logger=self.rollout_stats
            )
        else:
            raise NotImplementedError
        return intrinsic_rewards, model_mems


    def collect_rollouts(
        self,
        env: VecEnv,
        callback: BaseCallback,
        ppo_rollout_buffer: PPORolloutBuffer,
        n_rollout_steps: int,
    ) -> bool:
        assert self._last_obs is not None, "No previous observation was provided"
        n_steps = 0
        ppo_rollout_buffer.reset()
        # Sample new weights for the state dependent exploration
        if self.use_sde:
            self.policy.reset_noise(env.num_envs)

        callback.on_rollout_start()
        self.init_on_rollout_start()

        while n_steps < n_rollout_steps:
            
            if self.use_sde and self.sde_sample_freq > 0 and n_steps % self.sde_sample_freq == 0:
                # Sample a new noise matrix
                self.policy.reset_noise(env.num_envs)

            with th.no_grad():
                # Convert to pytorch tensor or to TensorDict
                obs_tensor = obs_as_tensor(self._last_obs, self.device)
                actions, values, log_probs, policy_mems = \
                    self.policy.forward(obs_tensor, self._last_policy_mems)
                actions = actions.cpu().numpy()

            # Rescale and perform action
            clipped_actions = actions
            # Clip the actions to avoid out of bound error
            if isinstance(self.action_space, gym.spaces.Box):
                clipped_actions = np.clip(actions, self.action_space.low, self.action_space.high)

            # Log before a transition
            self.log_before_transition(values)

            # Transition
            new_obs, rewards, dones, infos = env.step(clipped_actions)



            if isinstance(new_obs, Dict):
                new_obs = new_obs["rgb"]
            if self.env_render:
                env.render()

            with th.no_grad():
                # Compute value for the last timestep
                new_obs_tensor = obs_as_tensor(new_obs, self.device)
                _, new_values, _, _ = self.policy.forward(new_obs_tensor, policy_mems)

            # IR Generation
            intrinsic_rewards, model_mems = \
                self.create_intrinsic_rewards(new_obs, actions, dones)

            # Log after the transition and IR generation
            self.log_after_transition(rewards, intrinsic_rewards)

            # Clear episodic memories when an episode ends
            # for most the cases, but here it seems to require only termination signals.
            self.clear_on_episode_end(dones, policy_mems, model_mems)

            # Update global stats
            self.num_timesteps += self.n_envs
            self._update_info_buffer(infos)
            n_steps += 1

            callback.update_locals(locals())
            if callback.on_step() is False:
                return False

            # Add to PPO buffer
            if isinstance(self.action_space, gym.spaces.Discrete):
                actions = actions.reshape(-1, 1)
            ppo_rollout_buffer.add(
                self._last_obs,
                new_obs,
                self._last_policy_mems,
                self._last_model_mems,
                actions,
                rewards,
                intrinsic_rewards,
                self._last_episode_starts,
                dones,
                values,
                log_probs,
                self.curr_key_status,
                self.curr_door_status,
                self.curr_target_dists,
            )
            self._last_obs = new_obs
            self._last_episode_starts = dones
            if policy_mems is not None:
                self._last_policy_mems = policy_mems.detach().clone()
            if model_mems is not None:
                self._last_model_mems = model_mems.detach().clone()

        ppo_rollout_buffer.compute_intrinsic_rewards()
        ppo_rollout_buffer.compute_returns_and_advantage(new_values, dones)
        callback.on_rollout_end()
        return True


    def learn(
        self,
        total_timesteps: int,
        callback: MaybeCallback = None,
        log_interval: int = 1,
        eval_env: Optional[GymEnv] = None,
        eval_freq: int = -1,
        n_eval_episodes: int = 5,
        tb_log_name: str = "CustomOnPolicyAlgorithm",
        eval_log_path: Optional[str] = None,
        reset_num_timesteps: bool = True,
    ) -> "PPORollout":
        self.iteration = 0

        total_timesteps, callback = self._setup_learn(
            total_timesteps, callback, reset_num_timesteps, tb_log_name
        )
        self.total_timesteps = total_timesteps
        callback.on_training_start(locals(), globals())

        self.on_training_start()
        print('Collecting rollouts ...')

        # initial settings for reward guided exploitation
        self.rew_achieve = 0
        self.ent_coef_init = self.ent_coef
        if isinstance(self.action_space, spaces.Discrete):
            self.entropy_max = -self.action_space.n * 1.0/self.action_space.n * np.log(1.0/self.action_space.n)

        while self.num_timesteps < total_timesteps:
            collect_start_time = time.time()
            self.policy.eval()
            continue_training = self.collect_rollouts(
                self.env,
                callback,
                self.ppo_rollout_buffer,
                n_rollout_steps=self.ppo_rollout_buffer.buffer_size)
            self.policy.train()
            collect_end_time = time.time()

            # Uploading rollout infos
            self.iteration += 1
            self._update_current_progress_remaining(self.num_timesteps, total_timesteps)
            self.log_on_rollout_end(log_interval)

            # Train the agent's policy and the IR model
            if continue_training is False:
                break
            train_start_time = time.time()
            self.train()
            train_end_time = time.time()

            # Print to the console
            rews = [ep_info["r"] for ep_info in self.ep_info_buffer]
            rew_mean = 0.0 if len(rews) == 0 else np.mean(rews)
            print(f'run: {self.run_id:2d}  '
                  f'iters: {self.iteration}  '
                  f'frames: {self.num_timesteps}  '
                  f'rew: {rew_mean:.6f}  '
                  f'rollout: {collect_end_time - collect_start_time:.3f} sec  '
                  f'train: {train_end_time - train_start_time:.3f} sec')

        callback.on_training_end()
        return self
