import torch
from tqdm import tqdm
import pdb
import numpy as np 
import pandas as pd 
from transformers import AutoModelForCausalLM

from utils.lowrank_layers import SVDLinear

from collections import defaultdict


def get_mapping_dict(compression_stats):
    """
    Creates a dictionary that maps layer number and layer name to the number of singular values to retain. Saved and
    required when initializing a new llama model
    """

    mapping = defaultdict(dict)
    mapping_to_masks =  defaultdict(dict)
    for row in compression_stats:
        mapping[row['layer_idx']][row['layer_name']] = row['topk']
        mapping_to_masks[row['layer_idx']][row['layer_name']] = row['mask']

    return mapping, mapping_to_masks


def convert_linear_to_compressed(module):
    """
    Convert the low-rank linear model used in training to a compressed model, to be used during evalution 
    """

    E_train_mask = module.calculate_mask(is_training=False)
    m,n,r =  module.UE.size(0), module.V_t.size(1), E_train_mask.sum().item()
    compression_ratio = (m*r + n*r)/(m*n)

    # if compression is achieved, create lowrank layer 
    if compression_ratio < 1.:
        UE = module.UE[:, E_train_mask]
        V_t = module.V_t[E_train_mask, :]
        new_module = LinearLowRank(UE, V_t)
        
    else: 
        print(f'Compress not achieved for module ignore low-rank conversion:{module}')
        W_new = module.UE @ module.V_t
        new_module = torch.nn.Linear(W_new.shape[1], W_new.shape[0], bias=False)
        new_module.weight.data = W_new.contiguous()
        r = None

    return new_module, r

class LinearLowRank(torch.nn.Module):
    def __init__(self, UE, V_t, init_conig={}):
        super(LinearLowRank, self).__init__()
        """
        More efficient in the forward pass by avoiding first materialization of the weight

        Inputs: Linear layer to perform ASVD on.
        Approach: Parameter + gumbel sigmoid to generate mask
        """
        if not init_conig:
            self.in_features = int(V_t.shape[1])
            self.out_features = int(UE.shape[0])
            self.rank = int(V_t.shape[0])

            self.V_t = torch.nn.Linear(V_t.shape[1], V_t.shape[0], bias=False)
            self.UE = torch.nn.Linear(UE.shape[1], UE.shape[0], bias=False)
            self.V_t.weight.data = V_t.contiguous()
            self.UE.weight.data = UE.contiguous()
            print(f'Created UE: in={UE.shape[1]}, out={ UE.shape[0]}')
            print(f'Created V_t: in={V_t.shape[1]}, out={V_t.shape[0]}')
        else:
            self.in_features = init_conig['in_features']
            self.out_features = init_conig['out_features']
            self.rank = init_conig['rank']

            self.V_t = torch.nn.Linear(self.in_features, self.rank, bias=False)
            self.UE = torch.nn.Linear(self.rank, self.out_features, bias=False)

    def forward(self, inputs):
        x = self.V_t(inputs)
        return self.UE(x)

    def __str__(self):
        return f"LinearLowRank(in_features={self.in_features}, out_features={self.out_features}, rank={self.rank})"

    def __repr__(self):
        return self.__str__()
    

def replace_with_compressed_layer(model):
    """
    Replace all the low-rank decomposed full-rank layers with layers that only contain the selected singular values.

    """
    full_name_dict = {module: name for name, module in model.named_modules()}
    linear_info = {}
    modules = [model]
    while len(modules) > 0:
        submodule = modules.pop()
        for name, raw_linear in submodule.named_children():
            if hasattr(raw_linear, 'E_train'):
                full_name = full_name_dict[raw_linear]
                linear_info[raw_linear] = {
                    "father": submodule,
                    "name": name,
                    "full_name": full_name,
                }
            else:
                modules.append(raw_linear)

    for total_len, _ in enumerate(model.named_modules()):
        pass
    
    replace_config = {} 
    i = 0
    for name, module in tqdm(model.named_modules(), total=total_len, desc='Replacing Linear with Low-Rank Layers', mininterval=5):
        if module in linear_info:
            info = linear_info[module]

            compressed_module, r = convert_linear_to_compressed(module)

            setattr(info["father"], info["name"], compressed_module)

            del linear_info[module]
            del module
            
            i += 1
            if i % 10 == 0:
                torch.cuda.empty_cache()

            # if no compression done, ignore adding to config
            if r == None:
                continue

            tokens = name.split('.')
            layer_idx, layer_name = int(tokens[2]), tokens[-1]

            if layer_idx not in replace_config: 
                replace_config[layer_idx] = {} 
            
            replace_config[layer_idx][layer_name] = r

    torch.cuda.empty_cache()
    print('Replaced low-rank layers with compressed low-rank layers.')
    return model, replace_config 

class SVDLinearWithMask(torch.nn.Module):
    def __init__(self, current_layer, mask, svd_vector=None, alpha=0.5):
        super(SVDLinearWithMask, self).__init__()
        """
        Converts the linear layer into a layer where the weight matrix is decomposed using SVD into singular vectors and singular values

        Inputs:
            - current_layer: Linear layer that has to be compressed
            - mask: binary list
            - svd_vector: dict of asvd weights 
            - alpha: alpha value of svd
        """
        self.in_features = current_layer.weight.shape[1]
        self.out_features = current_layer.weight.shape[0]

        mask = torch.BoolTensor(mask)
        self.rank = int(mask.sum().item())

        weight = current_layer.weight.float()
        if svd_vector is not None: 
            svd_vector = svd_vector.to(weight.device)**alpha
            weight = weight * svd_vector.unsqueeze(0)
        
        device = weight.device
        if torch.cuda.is_available():
            weight = weight.cuda()

        #self.rank = int(self.in_features*self.out_features/(self.in_features + self.out_features)) # start from no compression   
        U, E, V = torch.svd_lowrank(weight,
                                    q=len(mask),
                                    niter=2)
        U, E, V = U.to(device), E.to(device), V.to(device)
        mask = mask.to(device)
        U, E, V = U[:, mask], E[mask], V[:, mask] # select required singular values
        
        if svd_vector is not None: 
            V = V / svd_vector.unsqueeze(1)

        self.V_t = torch.nn.Linear(self.in_features, V.shape[1], bias=False)
        self.UE = torch.nn.Linear(U.shape[1], self.out_features, bias=False)

        UE = (U * E.unsqueeze(0)).to(U.dtype)

        self.UE.weight.data = UE.contiguous() 
        self.V_t.weight.data = V.T.contiguous() 

    def forward(self, inputs):
        x = self.V_t(inputs)
        return self.UE(x)

    def __str__(self):
        return f"LinearLowRank(in_features={self.in_features}, out_features={self.out_features}, rank={self.rank})"

    def __repr__(self):
        return self.__str__()
    

def replace_with_compressed_layer_13b(args, svd_info={}, compression_map={}):
    """
    Replace all linear layers in a PyTorch model with low-rank approximations using SVD.

    """
    model = AutoModelForCausalLM.from_pretrained(args.model_name, cache_dir=args.cache_dir)
    compression_map = {str(k): v for k, v in compression_map.items()}

    full_name_dict = {module: name for name, module in model.named_modules()}
    linear_info = {}
    modules = [model]
    while len(modules) > 0:
        submodule = modules.pop()
        for name, raw_linear in submodule.named_children():
            if isinstance(raw_linear, torch.nn.Linear):
                full_name = full_name_dict[raw_linear]
                linear_info[raw_linear] = {
                    "father": submodule,
                    "name": name,
                    "full_name": full_name,
                }
            else:
                modules.append(raw_linear)

    for total_len, _ in enumerate(model.named_modules()):
        pass

    i = 0
    lowrank_config = defaultdict(dict)
    for name, module in tqdm(model.named_modules(), total=total_len, desc='Replacing Linear with Low-Rank Layers', mininterval=5):
        if 'lm_head' in name:
            print('Ignored low-rank decomposition on logits layer')
        
        elif module in linear_info:
            info = linear_info[module]

            if svd_info: 
                svd_vector = svd_info[info['full_name']]
            else: 
                svd_vector = None

            if compression_map:
                layer_idx = name.split('.')[2] 
                found_name = name.split('.')[-1]
                assert layer_idx.isnumeric(), f"layer_idx: {layer_idx} is not numeric"

                if layer_idx in compression_map and found_name in compression_map[layer_idx]: 
                    mask = compression_map[layer_idx][found_name]

                    if sum(mask) != len(mask): 
                        new_module = SVDLinearWithMask(module, mask, svd_vector)
                        setattr(info["father"], info["name"], new_module)
                        print(f"Used compression_map, {layer_idx}, {found_name}, rank: {sum(mask)}", new_module)
                        lowrank_config[layer_idx][found_name] = sum(mask)
                    else:
                        print(f'No compression in {layer_idx}, {found_name}: learnt rank {sum(mask)} = true rank {len(mask)}')

            i += 1
            if i > 10 and args.debug:
                break

    print('Replaced linear layers with low-rank layers.')
    return model, lowrank_config
