from collections import OrderedDict
from torch.utils.data import DataLoader
from torch.utils.data.dataloader import default_collate
from torch.utils.data.dataset import Dataset as TorchDataset
from glob import glob
import torch
from torchmeta.utils.data.dataset import CombinationMetaDataset
import random
import warnings
import numpy as np
from dppy.finite_dpps import FiniteDPP
from torch.utils.data.sampler import RandomSampler
from src.datasets.task_sampler.disjoint_sampler import DisjointMetaDataloader


def get_num_samples(targets, num_classes, dtype=None):
    batch_size = targets.size(0)
    with torch.no_grad():
        ones = torch.ones_like(targets, dtype=dtype)
        num_samples = ones.new_zeros((batch_size, num_classes))
        num_samples.scatter_add_(1, targets, ones)
    return num_samples


def get_prototypes(embeddings, targets, num_classes):
    """Compute the prototypes (the mean vector of the embedded training/support
    points belonging to its class) for each classes in the task.
    Parameters
    ----------
    embeddings : `torch.FloatTensor` instance
        A tensor containing the embeddings of the support points. This tensor
        has shape `(batch_size, num_examples, embedding_size)`.
    targets : `torch.LongTensor` instance
        A tensor containing the targets of the support points. This tensor has
        shape `(batch_size, num_examples)`.
    num_classes : int
        Number of classes in the task.
    Returns
    -------
    prototypes : `torch.FloatTensor` instance
        A tensor containing the prototypes for each class. This tensor has shape
        `(batch_size, num_classes, embedding_size)`.
    """
    batch_size, embedding_size = embeddings.size(0), embeddings.size(-1)

    num_samples = get_num_samples(targets, num_classes, dtype=embeddings.dtype)
    num_samples.unsqueeze_(-1)
    num_samples = torch.max(num_samples, torch.ones_like(num_samples))

    prototypes = embeddings.new_zeros((batch_size, num_classes, embedding_size))
    indices = targets.unsqueeze(-1).expand_as(embeddings)
    prototypes.scatter_add_(1, indices, embeddings).div_(num_samples)
    return prototypes


class DPPSampler(RandomSampler):
    def __init__(self, data_source, tasks, batch_size):
        self.tasks = tasks
        self.batch_size = batch_size
        if not isinstance(data_source, CombinationMetaDataset):
            raise TypeError('Expected `data_source` to be an instance of '
                            '`CombinationMetaDataset`, but found '
                            '{0}'.format(type(data_source)))
        # Temporarily disable the warning if the length of the length of the
        # dataset exceeds the machine precision. This avoids getting this
        # warning shown with MetaDataLoader, even though MetaDataLoader itself
        # does not use the length of the dataset.
        with warnings.catch_warnings():
            warnings.simplefilter('ignore')
            super(DPPSampler, self).__init__(data_source,
                                             replacement=True)

    def __iter__(self):
        num_classes_per_task = self.data_source.num_classes_per_task
        num_classes = len(self.data_source.dataset)
        if len(self.tasks):
            x = self.tasks
            for _ in range(len(self.tasks)):
                y = random.sample(x, 1)
                x = [item for item in x if item not in y]
                yield tuple(y[0])
            for _ in range(int(self.batch_size-len(self.tasks))):
                yield tuple(random.sample(range(num_classes), num_classes_per_task))
        else:
            for _ in range(self.batch_size):
                yield tuple(random.sample(range(num_classes), num_classes_per_task))


class MetaDataLoader(DataLoader):
    def __init__(self, dataset, batch_size=1, shuffle=True, sampler=None,
                 batch_sampler=None, num_workers=0, collate_fn=None,
                 pin_memory=False, drop_last=False, timeout=0,
                 worker_init_fn=None, task=None):
        if collate_fn is None:
            collate_fn = no_collate

        if isinstance(dataset, CombinationMetaDataset) and (sampler is None):
            sampler = DPPSampler(dataset, task, batch_size)
        shuffle = False

        super(MetaDataLoader, self).__init__(dataset, batch_size=batch_size,
                                             shuffle=shuffle, sampler=sampler, batch_sampler=batch_sampler,
                                             num_workers=num_workers, collate_fn=collate_fn,
                                             pin_memory=pin_memory, drop_last=drop_last, timeout=timeout,
                                             worker_init_fn=worker_init_fn)


class BatchMetaCollate(object):

    def __init__(self, collate_fn):
        super().__init__()
        self.collate_fn = collate_fn

    def collate_task(self, task):
        if isinstance(task, TorchDataset):
            return self.collate_fn([task[idx] for idx in range(len(task))])
        elif isinstance(task, OrderedDict):
            return OrderedDict([(key, self.collate_task(subtask))
                                for (key, subtask) in task.items()])
        else:
            raise NotImplementedError()

    def __call__(self, batch):
        return self.collate_fn([self.collate_task(task) for task in batch])


def no_collate(batch):
    return batch


class BatchMetaDataLoaderDPP(MetaDataLoader):
    def __init__(self, dataset, batch_size=1, shuffle=True, sampler=None, num_workers=0,
                 pin_memory=False, drop_last=False, timeout=0, worker_init_fn=None, task=None):
        collate_fn = BatchMetaCollate(default_collate)

        super(BatchMetaDataLoaderDPP, self).__init__(dataset,
                                                     batch_size=batch_size, shuffle=shuffle, sampler=sampler, num_workers=num_workers,
                                                     collate_fn=collate_fn, pin_memory=pin_memory, drop_last=drop_last,
                                                     timeout=timeout, worker_init_fn=worker_init_fn, task=task)


class DPP(object):
    def __init__(self, dataset, batch_size=32, shuffle=True, num_workers=1, pin_memory=True, num_ways=5, mode='static', dataset_name='omniglot'):
        self.dataset = dataset
        self.batch_size = batch_size
        self.shuffle = shuffle
        self.num_workers = num_workers
        self.pin_memory = pin_memory
        self.num_ways = num_ways
        self.index = 0
        self.dataset_name = dataset_name
        self.disjoint_dataloader = DisjointMetaDataloader(self.dataset,
                                                          batch_size=self.batch_size,
                                                          shuffle=self.shuffle,
                                                          num_workers=self.num_workers,
                                                          pin_memory=self.pin_memory)
        self.prototypes = {}
        if mode == 'static':
            self.model = self.init_static_model()
        else:
            self.model = self.metalearner.model

    def init_static_model(self):
        from src.protonet.model import Protonet_Omniglot, Protonet_MiniImagenet
        model = Protonet_Omniglot() if self.dataset_name == 'omniglot' else Protonet_MiniImagenet()
        for model_path in enumerate(glob(f"protonet_{self.dataset_name}/0/*/config.json")):
            with open(model_path, 'rb') as f:
                model.load_state_dict(torch.load(f, map_location=torch.device(
                    'cuda' if torch.cuda.is_available() else 'cpu')))
        return model

    def init_metalearner(self, metalearner):
        self.metalearner = metalearner

    def get_task_embedding(self):
        for batch in self.disjoint_dataloader:
            train_inputs, train_targets, tasks = batch['train']
            with torch.no_grad():
                train_embeddings = self.model(train_inputs)
                prototypes = get_prototypes(train_embeddings, train_targets, self.num_ways)
            for task_id, task in enumerate(tasks):
                for class_id, index in enumerate(task):
                    self.prototypes[index.item()] = np.array(
                        prototypes[task_id][class_id].cpu().tolist())

    def apply_dpp(self, Phi):
        print(type(Phi))
        print(Phi.shape)
        rng = np.random.RandomState(1)
        DPP = FiniteDPP('likelihood', **{'L': Phi.dot(Phi.T)})
        for _ in range(self.batch_size):
            DPP.sample_exact_k_dpp(size=self.num_ways, random_state=rng)
        return DPP.list_of_samples

    def get_diverse_tasks(self):
        self.get_task_embedding()
        return self.apply_dpp(Phi=np.array(list(self.prototypes.values())))

    def __iter__(self):
        return self

    def __next__(self):
        task_for_batch = self.get_diverse_tasks()
        self.index += 1
        print(f"Diverse tasks: {task_for_batch}")
        for batch in BatchMetaDataLoaderDPP(self.dataset,
                                            batch_size=self.batch_size,
                                            shuffle=self.shuffle,
                                            num_workers=self.num_workers,
                                            pin_memory=self.pin_memory, task=task_for_batch):
            return batch
            break
