# Modified from:
#   DiT:  https://github.com/facebookresearch/DiT/blob/main/sample.py
import torch
torch.backends.cuda.matmul.allow_tf32 = True
torch.backends.cudnn.allow_tf32 = True
torch.set_float32_matmul_precision('high')
setattr(torch.nn.Linear, 'reset_parameters', lambda self: None)
setattr(torch.nn.LayerNorm, 'reset_parameters', lambda self: None)
from torchvision.utils import save_image

import sys
sys.path.append(".")

import time
import argparse
from tokenizer.tokenizer_image.vq_model import VQ_models
from autoregressive.models.medusa_model.medusa_model_legacy import MedusaModel
from torch.nn import functional as F
from autoregressive.models.eagle_model.ea_model import EaModel
from autoregressive.models.eagle_model.kv_cache import initialize_past_key_values
from autoregressive.models.eagle_model.utils import *

import os
from tqdm import tqdm
import json


def cfg_logit_process(combined_logits, cfg_scale=4.0, cfg_interval=-1):
    if combined_logits.dim() == 4:
        combined_logits = combined_logits.transpose(0, 1)
    cond_logits, uncond_logits = torch.split(combined_logits, len(combined_logits) // 2, dim=0)
    logits = uncond_logits + (cond_logits - uncond_logits) * cfg_scale
    if combined_logits.dim() == 4:
        logits = logits.transpose(0, 1)
    return logits


def top_k_top_p_filtering(
    logits,
    top_k: int = 0,
    top_p: float = 1.0,
    filter_value: float = -float("Inf"),
    min_tokens_to_keep: int = 1,
):
    """Filter a distribution of logits using top-k and/or nucleus (top-p) filtering
    Args:
        logits: logits distribution shape (batch size, vocabulary size)
        if top_k > 0: keep only top k tokens with highest probability (top-k filtering).
        if top_p < 1.0: keep the top tokens with cumulative probability >= top_p (nucleus filtering).
            Nucleus filtering is described in Holtzman et al. (http://arxiv.org/abs/1904.09751)
        Make sure we keep at least min_tokens_to_keep per batch example in the output
    From: https://gist.github.com/thomwolf/1a5a29f6962089e871b94cbd09daf317
    """
    if top_k > 0:
        top_k = min(max(top_k, min_tokens_to_keep), logits.size(-1))  # Safety check
        # Remove all tokens with a probability less than the last token of the top-k
        indices_to_remove = logits < torch.topk(logits, top_k)[0][..., -1, None]
        logits[indices_to_remove] = filter_value

    if top_p < 1.0:
        sorted_logits, sorted_indices = torch.sort(logits, descending=True)
        cumulative_probs = torch.cumsum(F.softmax(sorted_logits, dim=-1), dim=-1)

        # Remove tokens with cumulative probability above the threshold (token with 0 are kept)
        sorted_indices_to_remove = cumulative_probs > top_p
        if min_tokens_to_keep > 1:
            # Keep at least min_tokens_to_keep (set to min_tokens_to_keep-1 because we add the first one below)
            sorted_indices_to_remove[..., :min_tokens_to_keep] = 0
        # Shift the indices to the right to keep also the first token above the threshold
        sorted_indices_to_remove[..., 1:] = sorted_indices_to_remove[..., :-1].clone()
        sorted_indices_to_remove[..., 0] = 0

        # scatter sorted tensors to original indexing
        indices_to_remove = sorted_indices_to_remove.scatter(1, sorted_indices, sorted_indices_to_remove)
        logits[indices_to_remove] = filter_value
    return logits


def sample(logits, temperature: float=1.0, top_k: int=0, top_p: float=1.0, sample_logits=True, repetition_penalty=1.0):        
    logits = logits[:, -1, :] / max(temperature, 1e-5)
    if top_k > 0 or top_p < 1.0:
        logits = top_k_top_p_filtering(logits, top_k=top_k, top_p=top_p)
    probs = F.softmax(logits, dim=-1)
    if sample_logits:
        idx = torch.multinomial(probs, num_samples=1)
    else:
        _, idx = torch.topk(probs, k=1, dim=-1)
    return idx, probs

def generate(model, c_indices, max_new_tokens, cfg_scale=4.0, cfg_interval=-1, accept_k=1, **sampling_kwargs):
    model.eval()
    accept_length_list = []
    with torch.no_grad():
        device = c_indices.device
        if cfg_scale > 1.0:
            cond_null = torch.ones_like(c_indices) * model.base_model.model.num_classes
            cond_combined = torch.cat([c_indices, cond_null])
        else:
            cond_combined = c_indices
        
        max_batch_size = c_indices.shape[0]
        seq = torch.empty((max_batch_size, max_new_tokens), dtype=torch.int, device=device)
        padding = (torch.zeros(1,1,dtype=torch.long)-1).to(cond_combined.device)
        model.ea_layer.reset_kv()

        if sampling_kwargs['temperature'] > 1e-5:
            logits_processor = prepare_logits_processor(temperature=sampling_kwargs['temperature'], top_k=sampling_kwargs['top_k'], top_p=sampling_kwargs['top_p'], repetition_penalty=sampling_kwargs['repetition_penalty'])
        else:
            logits_processor = None

        if hasattr(model, "past_key_values"):
            past_key_values = model.past_key_values
            past_key_values_data = model.past_key_values_data
            current_length_data = model.current_length_data
            # Reset the past key and value states
            current_length_data.zero_()
        else:
            (
                past_key_values,
                past_key_values_data,
                current_length_data,
            ) = initialize_past_key_values(model.base_model)
            model.past_key_values = past_key_values
            model.past_key_values_data = past_key_values_data
            model.current_length_data = current_length_data
        new_token = 1
        cur_length = 1

        reset_tree_mode(model)
        
        draft_tokens, retrieve_indices,tree_mask,tree_position_ids, logits, hidden_state, sample_token = initialize_tree(
            cond_combined, model, past_key_values, logits_processor, cfg_scale, cfg_interval
        )

        # ! Checked until here

        max_steps = max_new_tokens
        input_ids = cond_combined.unsqueeze(1)
        for idx in range(max_steps): # idx: new decoding steps
            model.base_model.model.tree_mask = tree_mask

            tree_draft_tokens = torch.cat([draft_tokens, draft_tokens]).to(input_ids.device)
            if new_token > max_new_tokens - 6:
                model.base_model.model.tree_mask = None
                position_ids = torch.zeros(1) + input_ids.shape[1]
                if position_ids > max_new_tokens:
                    input_ids = torch.cat([input_ids, draft_tokens[:, :1]], dim=1)
                    break
                position_ids = position_ids.to(input_ids.device).long()
                outputs, tree_logits, hidden_state_new = model(
                    input_ids=tree_draft_tokens[:, :1],
                    output_orig=True,
                    past_key_values=past_key_values,
                    position_ids=position_ids,
                )
                tree_logits = cfg_logit_process(tree_logits, cfg_scale, cfg_interval)
                input_ids = torch.cat([input_ids, draft_tokens[:, :1]], dim=1)
                draft_tokens, probs = sample(tree_logits, **sampling_kwargs)
                continue

            logits, hidden_state_new, outputs = tree_decoding(
                model,
                tree_draft_tokens,
                past_key_values,
                tree_position_ids,
                input_ids,
                retrieve_indices,
                cfg_scale,
                cfg_interval,
            )
            draft_tokens=torch.cat((draft_tokens,padding),dim=1)
            candidates=draft_tokens[0,retrieve_indices]
            best_candidate, accept_length, sample_p = evaluate_posterior_loose(
                logits, candidates, logits_processor, accept_k
            )
            input_ids, draft_tokens, retrieve_indices,tree_mask,tree_position_ids, new_token, hidden_state, sample_token = update_inference_inputs(
                input_ids,
                candidates,
                best_candidate,
                accept_length,
                retrieve_indices,
                logits_processor,
                new_token,
                past_key_values_data,
                current_length_data,
                model,
                hidden_state_new,
                sample_p,
                cfg_scale,
                cfg_interval,
            )
            # accept_length_tree = input_ids.shape[1] - cur_length
            # cur_length = accept_length_tree + cur_length
            # accept_length_list.append(accept_length_tree)
            accept_length_list.append(accept_length.item()+1)
            if new_token > max_new_tokens:
                break
        return input_ids[:, 1:max_new_tokens+1], accept_length_list



def main(args):
    # Setup PyTorch:
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False
    torch.set_grad_enabled(False)
    device = "cuda" if torch.cuda.is_available() else "cpu"

    # create and load model
    vq_model = VQ_models[args.vq_model](
        codebook_size=args.codebook_size,
        codebook_embed_dim=args.codebook_embed_dim)
    vq_model.to(device)
    vq_model.eval()
    checkpoint = torch.load(args.vq_ckpt, map_location="cpu")
    vq_model.load_state_dict(checkpoint["model"])
    del checkpoint
    print(f"image tokenizer is loaded")

    # create and load gpt model
    precision = {'none': torch.float32, 'bf16': torch.bfloat16, 'fp16': torch.float16}[args.precision]
    latent_size = args.image_size // args.downsample_size
    # gpt_model = MedusaModel.from_pretrained(
    #     args.gpt_ckpt,
    # ).to(device=device, dtype=precision)
    gpt_model = EaModel.from_pretrained(
        base_model_path=args.gpt_base_ckpt,
        ea_model_path=args.gpt_ckpt,
        total_token=args.total_token,
    ).to(device=device, dtype=precision)
    
    print(f"gpt model is loaded")

    if args.compile:
        print(f"compiling the model...")
        gpt_model = torch.compile(
            gpt_model,
            mode="reduce-overhead",
            fullgraph=True
        ) # requires PyTorch 2.0 (optional)
    else:
        print(f"no need to compile model in demo") 
    
    gpt_base_name = args.gpt_ckpt.split('/')[-3] + '_temp_' + str(args.temperature) + '_topk_' + str(args.top_k) + '_topp_' + str(args.top_p) + '_penalty_' + str(args.repetition_penalty) + '_cfg_scale_' + str(args.cfg_scale)
    os.makedirs("samples/{}_accept_k_{}".format(gpt_base_name, args.accept_k), exist_ok=True)

    # Labels to condition the model with (feel free to change):
    # class_labels = [207, 360, 387, 974, 88, 979, 417, 279]
    for idx in tqdm(range(1000)):
        for j in range(args.num_samples):
            class_labels = [idx]
            c_indices = torch.tensor(class_labels, device=device)
            qzshape = [len(class_labels), args.codebook_embed_dim, latent_size, latent_size]

            t1 = time.time()
            index_sample, accepted_length_list = generate(
                gpt_model, c_indices, latent_size ** 2,
                cfg_scale=args.cfg_scale, cfg_interval=args.cfg_interval,
                accept_k=args.accept_k,
                temperature=args.temperature, top_k=args.top_k,
                top_p=args.top_p, sample_logits=True, 
                repetition_penalty=args.repetition_penalty
                )
            sampling_time = time.time() - t1
            # print(f"gpt sampling takes about {sampling_time:.2f} seconds.")    
            
            t2 = time.time()
            samples = vq_model.decode_code(index_sample, qzshape) # output value is between [-1, 1]
            decoder_time = time.time() - t2
            # print(f"decoder takes about {decoder_time:.2f} seconds.")

            # Save and display images:
            save_image(samples, "samples/{}_accept_k_{}/sample_{}_{}_{}.png".format(gpt_base_name, args.accept_k, args.gpt_type, idx, j), nrow=4, normalize=True, value_range=(-1, 1))
            sampling_stat = {}
            sampling_stat['idx'] = idx
            sampling_stat['j'] = j
            sampling_stat['sampling_time'] = sampling_time
            sampling_stat['decoder_time'] = decoder_time
            sampling_stat['accepted_length'] = accepted_length_list
            sampling_stat['mean_accepted_length'] = sum(accepted_length_list)/len(accepted_length_list)
            with open("samples/{}_accept_k_{}/sampling_stat_{}.jsonl".format(gpt_base_name, args.accept_k, args.gpt_type), 'a') as f:
                f.write(json.dumps(sampling_stat) + '\n')


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--gpt-ckpt", type=str, default=None)
    parser.add_argument("--gpt-base-ckpt", type=str, default='/path/to/LlamaGen-3B')
    parser.add_argument("--gpt-type", type=str, choices=['c2i', 't2i'], default="c2i", help="class-conditional or text-conditional")
    parser.add_argument("--from-fsdp", action='store_true')
    parser.add_argument("--cls-token-num", type=int, default=1, help="max token number of condition input")
    parser.add_argument("--precision", type=str, default='bf16', choices=["none", "fp16", "bf16"]) 
    parser.add_argument("--compile", action='store_true', default=False)
    parser.add_argument("--vq-model", type=str, choices=list(VQ_models.keys()), default="VQ-16")
    parser.add_argument("--vq-ckpt", type=str, default=None, help="ckpt path for vq model")
    parser.add_argument("--codebook-size", type=int, default=16384, help="codebook size for vector quantization")
    parser.add_argument("--codebook-embed-dim", type=int, default=8, help="codebook dimension for vector quantization")
    parser.add_argument("--image-size", type=int, choices=[256, 384, 512], default=384)
    parser.add_argument("--downsample-size", type=int, choices=[8, 16], default=16)
    parser.add_argument("--num-classes", type=int, default=1000)
    parser.add_argument("--cfg-scale", type=float, default=4.0)
    parser.add_argument("--cfg-interval", type=float, default=-1)
    parser.add_argument("--seed", type=int, default=0)
    parser.add_argument("--top-k", type=int, default=2000,help="top-k value to sample with")
    parser.add_argument("--temperature", type=float, default=1.0, help="temperature value to sample with")
    parser.add_argument("--top-p", type=float, default=1.0, help="top-p value to sample with")
    parser.add_argument("--total-token", type=int, default=60)
    parser.add_argument("--repetition_penalty", type=float, default=1.0)
    parser.add_argument("--accept-k", type=int, default=1)
    parser.add_argument("--num-samples", type=int, default=1)
    args = parser.parse_args()
    main(args)