import torch
torch.backends.cuda.matmul.allow_tf32 = True
torch.backends.cudnn.allow_tf32 = True
torch.set_float32_matmul_precision('high')
setattr(torch.nn.Linear, 'reset_parameters', lambda self: None)     # disable default parameter init for faster speed
setattr(torch.nn.LayerNorm, 'reset_parameters', lambda self: None)  # disable default parameter init for faster speed
from torchvision.utils import save_image

import sys
sys.path.append(".")

import os
import time
import argparse
from tokenizer.tokenizer_image.vq_model import VQ_models
# from language.t5 import T5Embedder
from autoregressive.models.modeling_llama import LlamaForCausalLM
from torch.nn import functional as F
os.environ["TOKENIZERS_PARALLELISM"] = "false"
import numpy as np
import pandas as pd
import json
from tqdm import tqdm

def cfg_logit_process(combined_logits, cfg_scale=4.0, cfg_interval=-1):
    cond_logits, uncond_logits = torch.split(combined_logits, len(combined_logits) // 2, dim=0)
    logits = uncond_logits + (cond_logits - uncond_logits) * cfg_scale
    return logits


def top_k_top_p_filtering(
    logits,
    top_k: int = 0,
    top_p: float = 1.0,
    filter_value: float = -float("Inf"),
    min_tokens_to_keep: int = 1,
):
    """Filter a distribution of logits using top-k and/or nucleus (top-p) filtering
    Args:
        logits: logits distribution shape (batch size, vocabulary size)
        if top_k > 0: keep only top k tokens with highest probability (top-k filtering).
        if top_p < 1.0: keep the top tokens with cumulative probability >= top_p (nucleus filtering).
            Nucleus filtering is described in Holtzman et al. (http://arxiv.org/abs/1904.09751)
        Make sure we keep at least min_tokens_to_keep per batch example in the output
    From: https://gist.github.com/thomwolf/1a5a29f6962089e871b94cbd09daf317
    """
    if top_k > 0:
        top_k = min(max(top_k, min_tokens_to_keep), logits.size(-1))  # Safety check
        # Remove all tokens with a probability less than the last token of the top-k
        indices_to_remove = logits < torch.topk(logits, top_k)[0][..., -1, None]
        # logits = torch.ones_like(logits).to(logits.device)
        logits[indices_to_remove] = filter_value

    if top_p < 1.0:
        sorted_logits, sorted_indices = torch.sort(logits, descending=True)
        cumulative_probs = torch.cumsum(F.softmax(sorted_logits, dim=-1), dim=-1)

        # Remove tokens with cumulative probability above the threshold (token with 0 are kept)
        sorted_indices_to_remove = cumulative_probs > top_p
        if min_tokens_to_keep > 1:
            # Keep at least min_tokens_to_keep (set to min_tokens_to_keep-1 because we add the first one below)
            sorted_indices_to_remove[..., :min_tokens_to_keep] = 0
        # Shift the indices to the right to keep also the first token above the threshold
        sorted_indices_to_remove[..., 1:] = sorted_indices_to_remove[..., :-1].clone()
        sorted_indices_to_remove[..., 0] = 0

        # scatter sorted tensors to original indexing
        indices_to_remove = sorted_indices_to_remove.scatter(1, sorted_indices, sorted_indices_to_remove)
        logits[indices_to_remove] = filter_value
    return logits


def sample(logits, temperature: float=1.0, top_k: int=0, top_p: float=1.0, sample_logits=True):        
    logits = logits[:, -1, :] / max(temperature, 1e-5)
    if top_k > 0 or top_p < 1.0:
        logits = top_k_top_p_filtering(logits, top_k=top_k, top_p=top_p)
    probs = F.softmax(logits, dim=-1)
    if sample_logits:
        idx = torch.multinomial(probs, num_samples=1)
    else:
        _, idx = torch.topk(probs, k=1, dim=-1)
    return idx, probs

def generate(model, c_indices, max_new_tokens, emb_masks=None, cfg_scale=4.0, **sampling_kwargs):
    model.eval()
    with torch.no_grad():
        device = c_indices.device
        if model.model.model_type == 'c2i':
            if cfg_scale > 1.0:
                cond_null = torch.ones_like(c_indices) * model.model.num_classes
                cond_combined = torch.cat([c_indices, cond_null])
            else:
                cond_combined = c_indices
        elif model.model.model_type == 't2i':
            if cfg_scale > 1.0:
                cond_null = torch.zeros_like(c_indices) + model.model.cls_embedding.uncond_embedding
                cond_combined = torch.cat([c_indices, cond_null])
            else:
                cond_combined = c_indices
            T = cond_combined.shape[1]      
        else:
            raise NotImplementedError(f"input type {model.model.input_type} is not supported")
        max_batch_size = c_indices.shape[0]
        
        if emb_masks is not None:
            assert emb_masks.shape[0] == max_batch_size
            assert emb_masks.shape[-1] == T
            if cfg_scale > 1.0:
                attention_mask = torch.cat([emb_masks, emb_masks])
            else:
                attention_mask =  emb_masks
        
        seq = torch.empty((max_batch_size, max_new_tokens), dtype=torch.int, device=device)
        output = model(cond_idx=cond_combined, attention_mask=attention_mask)
        combined_logits = output.logits
        past_key_values = output.past_key_values
        cond_logits, uncond_logits = torch.split(combined_logits, len(combined_logits) // 2, dim=0)
        logits = uncond_logits + (cond_logits - uncond_logits) * cfg_scale
        next_token, _ = sample(logits, **sampling_kwargs)
        seq[:, 0] = next_token.squeeze(-1)
        for i in range(max_new_tokens-1):
            attention_mask = torch.cat([attention_mask, torch.ones_like(attention_mask)[:, :1]], dim=-1)
            input_ids = torch.cat([next_token, next_token])
            output = model(input_ids=input_ids, past_key_values=past_key_values, attention_mask=attention_mask)
            combined_logits = output.logits
            past_key_values = output.past_key_values
            cond_logits, uncond_logits = torch.split(combined_logits, len(combined_logits) // 2, dim=0)
            logits = uncond_logits + (cond_logits - uncond_logits) * cfg_scale
            next_token, _ = sample(logits, **sampling_kwargs)
            seq[:, i+1] = next_token.squeeze(-1)
            
    return seq



def main(args):
    # Setup PyTorch:
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False
    torch.set_grad_enabled(False)
    device = "cuda" if torch.cuda.is_available() else "cpu"

    # create and load model
    vq_model = VQ_models[args.vq_model](
        codebook_size=args.codebook_size,
        codebook_embed_dim=args.codebook_embed_dim)
    vq_model.to(device)
    vq_model.eval()
    checkpoint = torch.load(args.vq_ckpt, map_location="cpu")
    vq_model.load_state_dict(checkpoint["model"])
    del checkpoint
    print(f"image tokenizer is loaded")

    # create and load gpt model
    precision = {'none': torch.float32, 'bf16': torch.bfloat16, 'fp16': torch.float16}[args.precision]
    latent_size = args.image_size // args.downsample_size
    gpt_model = LlamaForCausalLM.from_pretrained(
        args.gpt_ckpt,
    ).to(device=device, dtype=precision)
    # uncond_embedding_weight = torch.load(os.path.join(args.gpt_ckpt, 'uncond_embedding.pt')).to(device=device, dtype=precision)
    # gpt_model.model.cls_embedding.uncond_embedding = uncond_embedding_weight

    print(f"gpt model is loaded")

    if args.compile:
        print(f"compiling the model...")
        gpt_model = torch.compile(
            gpt_model,
            mode="reduce-overhead",
            fullgraph=True
        ) # requires PyTorch 2.0 (optional)
    else:
        print(f"no need to compile model in demo") 
    
    # assert os.path.exists(args.t5_path)
    # t5_model = T5Embedder(
    #     device=device, 
    #     local_cache=True, 
    #     cache_dir=args.t5_path, 
    #     dir_or_name=args.t5_model_type,
    #     torch_dtype=precision,
    #     model_max_length=args.t5_feature_max_len,
    # )
    # prompts = [
    #     "A portrait photo of a kangaroo wearing an orange hoodie and blue sunglasses standing on the grassin front of the Sydney Opera House holding a sign on the chest that says Welcome Friends!",
    #     "A blue Porsche 356 parked in front of a yellow brick wall.",
    #     "A photo of an astronaut riding a horse in the forest. There is a river in front of them with water lilies.",
    #     "A map of the United States made out of sushi. It is on a table next to a glass of red wine."
    # ]
    # prompts = ["A cutting board topped with bread, meat and vegetables"]

    # caption_embs, emb_masks = t5_model.get_text_embeddings(prompts)
    ckpt_string_name = os.path.basename(args.gpt_ckpt).replace(".pth", "").replace(".pt", "")
    prompt_name = args.prompt_csv.split('/')[-1].split('.')[0].lower()
    folder_name = f"{ckpt_string_name}-{prompt_name}-size-{args.image_size}-size-{args.image_size}-{args.vq_model}-" \
                  f"topk-{args.top_k}-topp-{args.top_p}-temperature-{args.temperature}-" \
                  f"cfg-{args.cfg_scale}_for_speedup"
    sample_folder_dir = f"{args.sample_dir}/{folder_name}"
    os.makedirs(f"{sample_folder_dir}/images", exist_ok=True)
    print(f"Saving .png samples at {sample_folder_dir}/images")
    prompt_feature_list = [os.path.join(args.prompt_csv, f'{index:06d}.npy') for index in range(args.num_samples)]
    df = pd.read_csv('./evaluations/t2i/coco_captions.csv', sep='\t')
    
    # warmup
    for i in range(5):
        prompt_batch = []
        emb_masks = []
        index = i
        prompt_elem = prompt_feature_list[index] if index < len(prompt_feature_list) else prompt_feature_list[-1]
        caption_emb = np.load(prompt_elem)
        caption_emb = torch.from_numpy(caption_emb).to(device=device, dtype=precision)
        caption_padding = args.cls_token_num - caption_emb.shape[1]
        if caption_padding > 0:
            zero_padding = torch.zeros(caption_emb.shape[0], caption_padding, caption_emb.shape[2], device=device, dtype=precision)
            caption_emb = torch.cat([zero_padding, caption_emb], dim=1)
        prompt_batch.append(caption_emb)
        emb_mask = torch.ones(caption_emb.shape[0], caption_emb.shape[1], device=device, dtype=precision)
        emb_mask[:, :caption_padding] = 0
        emb_masks.append(emb_mask)
        emb_masks = torch.cat(emb_masks)
        caption_embs = torch.cat(prompt_batch)
        
        new_caption_embs, new_emb_masks = caption_embs, emb_masks
        c_indices = new_caption_embs * new_emb_masks[:,:, None]
        c_emb_masks = new_emb_masks
        
        qzshape = [len(c_indices), args.codebook_embed_dim, latent_size, latent_size]
        index_sample = generate(
            gpt_model, c_indices, latent_size ** 2, 
            c_emb_masks, 
            cfg_scale=args.cfg_scale,
            temperature=args.temperature, top_k=args.top_k,
            top_p=args.top_p, sample_logits=True, 
            )
        samples = vq_model.decode_code(index_sample, qzshape) # output value is between [-1, 1]
        

    for i in tqdm(range(args.num_samples)):
        prompt_batch = []
        emb_masks = []
        index = i
        prompt_elem = prompt_feature_list[index] if index < len(prompt_feature_list) else prompt_feature_list[-1]
        caption_emb = np.load(prompt_elem)
        caption_emb = torch.from_numpy(caption_emb).to(device=device, dtype=precision)
        caption_padding = args.cls_token_num - caption_emb.shape[1]
        if caption_padding > 0:
            zero_padding = torch.zeros(caption_emb.shape[0], caption_padding, caption_emb.shape[2], device=device, dtype=precision)
            caption_emb = torch.cat([zero_padding, caption_emb], dim=1)
        prompt_batch.append(caption_emb)
        emb_mask = torch.ones(caption_emb.shape[0], caption_emb.shape[1], device=device, dtype=precision)
        emb_mask[:, :caption_padding] = 0
        emb_masks.append(emb_mask)
        emb_masks = torch.cat(emb_masks)
        caption_embs = torch.cat(prompt_batch)
        
        new_caption_embs, new_emb_masks = caption_embs, emb_masks
        c_indices = new_caption_embs * new_emb_masks[:,:, None]
        c_emb_masks = new_emb_masks

        qzshape = [len(c_indices), args.codebook_embed_dim, latent_size, latent_size]
        t1 = time.time()
        index_sample = generate(
            gpt_model, c_indices, latent_size ** 2, 
            c_emb_masks, 
            cfg_scale=args.cfg_scale,
            temperature=args.temperature, top_k=args.top_k,
            top_p=args.top_p, sample_logits=True, 
            )
        sampling_time = time.time() - t1  
        
        t2 = time.time()
        samples = vq_model.decode_code(index_sample, qzshape) # output value is between [-1, 1]

        decoder_time = time.time() - t2
        save_image(samples, f"{sample_folder_dir}/images/{index:06d}.png", normalize=True, value_range=(-1, 1))
        sampling_stats = {}
        sampling_stats["sampling_time"] = sampling_time
        sampling_stats["decoder_time"] = decoder_time
        with open(f"{sample_folder_dir}/sampling_stats.json", "a") as f:
            f.write(json.dumps(sampling_stats) + "\n")



if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--prompt-csv", type=str, default='/path/to/mscoco_text_features')
    parser.add_argument("--t5-path", type=str, default='pretrained_models/t5-ckpt')
    parser.add_argument("--t5-model-type", type=str, default='flan-t5-xl')
    parser.add_argument("--t5-feature-max-len", type=int, default=120)
    parser.add_argument("--t5-feature-dim", type=int, default=2048)
    parser.add_argument("--no-left-padding", action='store_true', default=False)
    parser.add_argument("--gpt-ckpt", type=str, default=None)
    parser.add_argument("--gpt-type", type=str, choices=['c2i', 't2i'], default="t2i", help="class->image or text->image")  
    parser.add_argument("--cls-token-num", type=int, default=120, help="max token number of condition input")
    parser.add_argument("--precision", type=str, default='bf16', choices=["none", "fp16", "bf16"]) 
    parser.add_argument("--compile", action='store_true', default=False)
    parser.add_argument("--vq-model", type=str, choices=list(VQ_models.keys()), default="VQ-16")
    parser.add_argument("--vq-ckpt", type=str, default=None, help="ckpt path for vq model")
    parser.add_argument("--codebook-size", type=int, default=16384, help="codebook size for vector quantization")
    parser.add_argument("--codebook-embed-dim", type=int, default=8, help="codebook dimension for vector quantization")
    parser.add_argument("--image-size", type=int, choices=[256, 384, 512], default=256)
    parser.add_argument("--downsample-size", type=int, choices=[8, 16], default=16)
    parser.add_argument("--num-classes", type=int, default=1000)
    parser.add_argument("--cfg-scale", type=float, default=7.5)
    parser.add_argument("--seed", type=int, default=0)
    parser.add_argument("--top-k", type=int, default=1000, help="top-k value to sample with")
    parser.add_argument("--temperature", type=float, default=1.0, help="temperature value to sample with")
    parser.add_argument("--top-p", type=float, default=1.0, help="top-p value to sample with")
    parser.add_argument("--sample-dir", type=str, default="samples_speedup_new")
    parser.add_argument("--num-samples", type=int, default=100)
    args = parser.parse_args()
    main(args)
