# Modified from:
#   DiT:  https://github.com/facebookresearch/DiT/blob/main/sample.py
import torch
torch.backends.cuda.matmul.allow_tf32 = True
torch.backends.cudnn.allow_tf32 = True
torch.set_float32_matmul_precision('high')
setattr(torch.nn.Linear, 'reset_parameters', lambda self: None)
setattr(torch.nn.LayerNorm, 'reset_parameters', lambda self: None)
from torchvision.utils import save_image

import sys
sys.path.append(".")
from tqdm import tqdm
import os
import json

import time
import argparse
from tokenizer.tokenizer_image.vq_model import VQ_models
from autoregressive.models.modeling_llama import LlamaForCausalLM
from torch.nn import functional as F

def verify_and_replace(model_preds, drafted_seq):
    # Convert inputs to lists (if they are not already lists)
    device = model_preds.device
    model_preds = model_preds[0].tolist()
    drafted_seq = drafted_seq[0].tolist()
    drafted_seq = drafted_seq[1:]
    
    # Initialize the length of verified tokens
    verified_length = 1
    
    # Compare each element
    for pred_token, draft_token in zip(model_preds, drafted_seq):
        if pred_token == draft_token:
            verified_length += 1
        else:
            break
    
    # The final sequence should only include tokens up to the first mismatch
    final_sequence_tensor = torch.tensor([model_preds[:verified_length]], device=device)

    return final_sequence_tensor, verified_length

def past_key_values_until(past_key_values, until):
    if until == -1:
        return past_key_values
    sliced_past_key_values = []
    for past_key, past_value in past_key_values:
        # Slice the sequence length dimension
        sliced_past_key = past_key[:, :, :until, :]
        sliced_past_value = past_value[:, :, :until, :]
        
        # Append the sliced tensors to the new list
        sliced_past_key_values.append((sliced_past_key, sliced_past_value))
    
    return sliced_past_key_values
    

def top_k_top_p_filtering(
    logits,
    top_k: int = 0,
    top_p: float = 1.0,
    filter_value: float = -float("Inf"),
    min_tokens_to_keep: int = 1,
):
    """Filter a distribution of logits using top-k and/or nucleus (top-p) filtering
    Args:
        logits: logits distribution shape (batch size, vocabulary size)
        if top_k > 0: keep only top k tokens with highest probability (top-k filtering).
        if top_p < 1.0: keep the top tokens with cumulative probability >= top_p (nucleus filtering).
            Nucleus filtering is described in Holtzman et al. (http://arxiv.org/abs/1904.09751)
        Make sure we keep at least min_tokens_to_keep per batch example in the output
    From: https://gist.github.com/thomwolf/1a5a29f6962089e871b94cbd09daf317
    """
    if top_k > 0:
        top_k = min(max(top_k, min_tokens_to_keep), logits.size(-1))  # Safety check
        # Remove all tokens with a probability less than the last token of the top-k
        indices_to_remove = logits < torch.topk(logits, top_k)[0][..., -1, None]
        logits[indices_to_remove] = filter_value

    if top_p < 1.0:
        sorted_logits, sorted_indices = torch.sort(logits, descending=True)
        cumulative_probs = torch.cumsum(F.softmax(sorted_logits, dim=-1), dim=-1)

        # Remove tokens with cumulative probability above the threshold (token with 0 are kept)
        sorted_indices_to_remove = cumulative_probs > top_p
        if min_tokens_to_keep > 1:
            # Keep at least min_tokens_to_keep (set to min_tokens_to_keep-1 because we add the first one below)
            sorted_indices_to_remove[..., :min_tokens_to_keep] = 0
        # Shift the indices to the right to keep also the first token above the threshold
        sorted_indices_to_remove[..., 1:] = sorted_indices_to_remove[..., :-1].clone()
        sorted_indices_to_remove[..., 0] = 0

        # scatter sorted tensors to original indexing
        indices_to_remove = sorted_indices_to_remove.scatter(1, sorted_indices, sorted_indices_to_remove)
        logits[indices_to_remove] = filter_value
    return logits


def sample(logits, temperature: float=1.0, top_k: int=0, top_p: float=1.0, sample_logits=True):        
    logits = logits[:, -1, :] / max(temperature, 1e-5)
    if top_k > 0 or top_p < 1.0:
        logits = top_k_top_p_filtering(logits, top_k=top_k, top_p=top_p)
    probs = F.softmax(logits, dim=-1)
    if sample_logits:
        idx = torch.multinomial(probs, num_samples=1)
    else:
        _, idx = torch.topk(probs, k=1, dim=-1)
    return idx, probs

def generate(model, c_indices, max_new_tokens, cfg_scale=4.0, cfg_interval=-1, **sampling_kwargs):
    model.eval()
    with torch.no_grad():
        device = c_indices.device
        if model.model.model_type == 'c2i':
            if cfg_scale > 1.0:
                cond_null = torch.ones_like(c_indices) * model.model.num_classes
                cond_combined = torch.cat([c_indices, cond_null])
            else:
                cond_combined = c_indices
        else:
            raise NotImplementedError(f"input type {model.model.input_type} is not supported")
        
        max_batch_size = c_indices.shape[0]
        seq = torch.empty((max_batch_size, max_new_tokens), dtype=torch.int, device=device)
        output = model(cond_idx=cond_combined)
        combined_logits = output.logits
        past_key_values = output.past_key_values
        cond_logits, uncond_logits = torch.split(combined_logits, len(combined_logits) // 2, dim=0)
        logits = uncond_logits + (cond_logits - uncond_logits) * cfg_scale
        next_token, _ = sample(logits, **sampling_kwargs)
        seq[:, 0] = next_token.squeeze(-1)
        for i in range(max_new_tokens-1):
            input_ids = torch.cat([next_token, next_token])
            output = model(input_ids=input_ids, past_key_values=past_key_values)
            combined_logits = output.logits
            past_key_values = output.past_key_values
            cond_logits, uncond_logits = torch.split(combined_logits, len(combined_logits) // 2, dim=0)
            logits = uncond_logits + (cond_logits - uncond_logits) * cfg_scale
            next_token, _ = sample(logits, **sampling_kwargs)
            seq[:, i+1] = next_token.squeeze(-1)
            
    return seq

def spec_generate(model, spec_model, c_indices, max_new_tokens, cfg_scale=4.0, cfg_interval=-1, **sampling_kwargs):
    # ! currently, only support greedy decoding
    drafting_length=5
    model.eval()
    spec_model.eval()
    accepted_length_list = []
    with torch.no_grad():
        device = c_indices.device
        if model.model.model_type == 'c2i':
            if cfg_scale > 1.0:
                cond_null = torch.ones_like(c_indices) * model.model.num_classes
                cond_combined = torch.cat([c_indices, cond_null])
            else:
                cond_combined = c_indices
        else:
            raise NotImplementedError(f"input type {model.model.input_type} is not supported")
        
        max_batch_size = c_indices.shape[0]
        seq = torch.empty((max_batch_size, max_new_tokens), dtype=torch.int, device=device)
        output = model(cond_idx=cond_combined)
        combined_logits = output.logits
        past_key_values = output.past_key_values
        spec_output = spec_model(cond_idx=cond_combined.clone().detach())
        spec_past_key_values = spec_output.past_key_values
        cond_logits, uncond_logits = torch.split(combined_logits, len(combined_logits) // 2, dim=0)
        logits = uncond_logits + (cond_logits - uncond_logits) * cfg_scale
        next_token= torch.argmax(logits[:, -1, :], dim=-1).unsqueeze(-1)
        seq[:, 0] = next_token.squeeze(-1)
        cur_length = 1
        while cur_length < max_new_tokens:
            if cur_length + drafting_length >= max_new_tokens:
                drafting_length = max_new_tokens - cur_length
            # draft the next token using spec model for drafting_length steps
            drafted_seq = torch.empty((max_batch_size, drafting_length + 1), dtype=torch.int, device=device)
            drafted_seq[:, 0] = next_token.squeeze(-1)
            for i in range(drafting_length):
                input_ids = torch.cat([next_token, next_token])
                output = spec_model(input_ids=input_ids, past_key_values=spec_past_key_values)
                combined_logits = output.logits
                spec_past_key_values = output.past_key_values
                cond_logits, uncond_logits = torch.split(combined_logits, len(combined_logits) // 2, dim=0)
                logits = uncond_logits + (cond_logits - uncond_logits) * cfg_scale
                next_token = torch.argmax(logits[:, -1, :], dim=-1).unsqueeze(-1)
                drafted_seq[:, i+1] = next_token.squeeze(-1)
            # verify the drafted sequence using model
            input_ids = torch.cat([drafted_seq, drafted_seq])
            
            output = model(input_ids=input_ids, past_key_values=past_key_values)
            combined_logits = output.logits
            past_key_values = output.past_key_values
            cond_logits, uncond_logits = torch.split(combined_logits, len(combined_logits) // 2, dim=0)
            logits = uncond_logits + (cond_logits - uncond_logits) * cfg_scale
            model_preds = torch.argmax(logits[:, -(drafting_length+1):, :], dim=-1)
            verified_seq, verified_length = verify_and_replace(model_preds, drafted_seq)
            # use the drafted sequence until the model rejects it from the beginning
            if cur_length + verified_length >= max_new_tokens:
                verified_length = max_new_tokens - cur_length
                verified_seq = verified_seq[:, :verified_length]
            seq[:, cur_length:cur_length+verified_length] = verified_seq.squeeze(-1)
            cur_length += verified_length
            past_key_values = past_key_values_until(past_key_values, cur_length)
            spec_past_key_values = past_key_values_until(spec_past_key_values, cur_length)
            accepted_length_list.append(verified_length)     
            next_token = verified_seq[:, -1].unsqueeze(-1)
            
    return seq, accepted_length_list

def main(args):
    # Setup PyTorch:
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False
    torch.set_grad_enabled(False)
    device = "cuda" if torch.cuda.is_available() else "cpu"

    # create and load model
    vq_model = VQ_models[args.vq_model](
        codebook_size=args.codebook_size,
        codebook_embed_dim=args.codebook_embed_dim)
    vq_model.to(device)
    vq_model.eval()
    checkpoint = torch.load(args.vq_ckpt, map_location="cpu")
    vq_model.load_state_dict(checkpoint["model"])
    del checkpoint
    print(f"image tokenizer is loaded")

    # create and load gpt model
    precision = {'none': torch.float32, 'bf16': torch.bfloat16, 'fp16': torch.float16}[args.precision]
    latent_size = args.image_size // args.downsample_size
    gpt_model = LlamaForCausalLM.from_pretrained(
        args.gpt_ckpt,
    ).to(device=device, dtype=precision)
    spec_model = LlamaForCausalLM.from_pretrained(
        args.spec_ckpt,
    ).to(device=device, dtype=precision)
    gpt_model_name = args.gpt_ckpt.split("/")[-1]
    spec_model_name = args.spec_ckpt.split("/")[-1]
    
    print(f"gpt model is loaded")

    if args.compile:
        print(f"compiling the model...")
        gpt_model = torch.compile(
            gpt_model,
            mode="reduce-overhead",
            fullgraph=True
        ) # requires PyTorch 2.0 (optional)
    else:
        print(f"no need to compile model in demo") 

    # Labels to condition the model with (feel free to change):
    # class_labels = [207, 360, 387, 974, 88, 979, 417, 279]
    os.makedirs("samples/{}_{}".format(gpt_model_name, spec_model_name), exist_ok=True)
    for idx in tqdm(range(1000)):
        class_labels = [idx]
        c_indices = torch.tensor(class_labels, device=device)
        qzshape = [len(class_labels), args.codebook_embed_dim, latent_size, latent_size]

        t1 = time.time()
        index_sample, accepted_length_list = spec_generate(
            gpt_model, spec_model, c_indices, latent_size ** 2,
            cfg_scale=args.cfg_scale, cfg_interval=args.cfg_interval,
            temperature=args.temperature, top_k=args.top_k,
            top_p=args.top_p, sample_logits=True, 
            )
        sampling_time = time.time() - t1
        # print(f"gpt sampling takes about {sampling_time:.2f} seconds.")    
        
        t2 = time.time()
        samples = vq_model.decode_code(index_sample, qzshape) # output value is between [-1, 1]
        decoder_time = time.time() - t2
        # print(f"decoder takes about {decoder_time:.2f} seconds.")

        # Save and display images:
        # save_image(samples, "sample_{}.png".format(args.gpt_type), nrow=4, normalize=True, value_range=(-1, 1))
        # print(f"image is saved to sample_{args.gpt_type}.png")
        save_image(samples, "samples/{}_{}/sample_{}_{}.png".format(gpt_model_name, spec_model_name, args.gpt_type, idx), nrow=4, normalize=True, value_range=(-1, 1))
        # print(f"image is saved to sample_{args.gpt_type}.png")
        sampling_stat = {}
        sampling_stat['idx'] = idx
        sampling_stat['sampling_time'] = sampling_time
        sampling_stat['decoder_time'] = decoder_time
        sampling_stat['accepted_length'] = accepted_length_list
        sampling_stat['mean_accepted_length'] = sum(accepted_length_list)/len(accepted_length_list)
        with open("samples/{}_{}/sampling_stat_{}.jsonl".format(gpt_model_name, spec_model_name, args.gpt_type), 'a') as f:
            f.write(json.dumps(sampling_stat) + '\n')


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--gpt-ckpt", type=str, default=None)
    parser.add_argument("--spec-ckpt", type=str, default=None)
    parser.add_argument("--gpt-type", type=str, choices=['c2i', 't2i'], default="c2i", help="class-conditional or text-conditional")
    parser.add_argument("--cls-token-num", type=int, default=1, help="max token number of condition input")
    parser.add_argument("--precision", type=str, default='bf16', choices=["none", "fp16", "bf16"]) 
    parser.add_argument("--compile", action='store_true', default=False)
    parser.add_argument("--vq-model", type=str, choices=list(VQ_models.keys()), default="VQ-16")
    parser.add_argument("--vq-ckpt", type=str, default=None, help="ckpt path for vq model")
    parser.add_argument("--codebook-size", type=int, default=16384, help="codebook size for vector quantization")
    parser.add_argument("--codebook-embed-dim", type=int, default=8, help="codebook dimension for vector quantization")
    parser.add_argument("--image-size", type=int, choices=[256, 384, 512], default=384)
    parser.add_argument("--downsample-size", type=int, choices=[8, 16], default=16)
    parser.add_argument("--num-classes", type=int, default=1000)
    parser.add_argument("--cfg-scale", type=float, default=4.0)
    parser.add_argument("--cfg-interval", type=float, default=-1)
    parser.add_argument("--seed", type=int, default=0)
    parser.add_argument("--top-k", type=int, default=2000,help="top-k value to sample with")
    parser.add_argument("--temperature", type=float, default=1.0, help="temperature value to sample with")
    parser.add_argument("--top-p", type=float, default=1.0, help="top-p value to sample with")
    args = parser.parse_args()
    main(args)