import torch
torch.backends.cuda.matmul.allow_tf32 = True
torch.backends.cudnn.allow_tf32 = True
torch.set_float32_matmul_precision('high')
setattr(torch.nn.Linear, 'reset_parameters', lambda self: None)     # disable default parameter init for faster speed
setattr(torch.nn.LayerNorm, 'reset_parameters', lambda self: None)  # disable default parameter init for faster speed
import torch.nn.functional as F
import torch.distributed as dist

import os
import math
import json
import argparse
import pandas as pd
from tqdm import tqdm
from PIL import Image

import sys
sys.path.append('.')

from tokenizer.tokenizer_image.vq_model import VQ_models
from torch.nn import functional as F
from autoregressive.models.eagle_model.ea_model import EaModel
from autoregressive.models.eagle_model.kv_cache import initialize_past_key_values
from autoregressive.models.eagle_model.utils import *
import numpy as np
os.environ["TOKENIZERS_PARALLELISM"] = "false"

def cfg_logit_process(combined_logits, cfg_scale=4.0, cfg_interval=-1):
    cond_logits, uncond_logits = torch.split(combined_logits, len(combined_logits) // 2, dim=0)
    logits = uncond_logits + (cond_logits - uncond_logits) * cfg_scale
    return logits


def top_k_top_p_filtering(
    logits,
    top_k: int = 0,
    top_p: float = 1.0,
    filter_value: float = -float("Inf"),
    min_tokens_to_keep: int = 1,
):
    """Filter a distribution of logits using top-k and/or nucleus (top-p) filtering
    Args:
        logits: logits distribution shape (batch size, vocabulary size)
        if top_k > 0: keep only top k tokens with highest probability (top-k filtering).
        if top_p < 1.0: keep the top tokens with cumulative probability >= top_p (nucleus filtering).
            Nucleus filtering is described in Holtzman et al. (http://arxiv.org/abs/1904.09751)
        Make sure we keep at least min_tokens_to_keep per batch example in the output
    From: https://gist.github.com/thomwolf/1a5a29f6962089e871b94cbd09daf317
    """
    if top_k > 0:
        top_k = min(max(top_k, min_tokens_to_keep), logits.size(-1))  # Safety check
        # Remove all tokens with a probability less than the last token of the top-k
        indices_to_remove = logits < torch.topk(logits, top_k)[0][..., -1, None]
        # logits = torch.ones_like(logits).to(logits.device)
        logits[indices_to_remove] = filter_value

    if top_p < 1.0:
        sorted_logits, sorted_indices = torch.sort(logits, descending=True)
        cumulative_probs = torch.cumsum(F.softmax(sorted_logits, dim=-1), dim=-1)

        # Remove tokens with cumulative probability above the threshold (token with 0 are kept)
        sorted_indices_to_remove = cumulative_probs > top_p
        if min_tokens_to_keep > 1:
            # Keep at least min_tokens_to_keep (set to min_tokens_to_keep-1 because we add the first one below)
            sorted_indices_to_remove[..., :min_tokens_to_keep] = 0
        # Shift the indices to the right to keep also the first token above the threshold
        sorted_indices_to_remove[..., 1:] = sorted_indices_to_remove[..., :-1].clone()
        sorted_indices_to_remove[..., 0] = 0

        # scatter sorted tensors to original indexing
        indices_to_remove = sorted_indices_to_remove.scatter(1, sorted_indices, sorted_indices_to_remove)
        logits[indices_to_remove] = filter_value
    return logits


def sample(logits, temperature: float=1.0, top_k: int=0, top_p: float=1.0, sample_logits=True):        
    logits = logits[:, -1, :] / max(temperature, 1e-5)
    if top_k > 0 or top_p < 1.0:
        logits = top_k_top_p_filtering(logits, top_k=top_k, top_p=top_p)
    probs = F.softmax(logits, dim=-1)
    if sample_logits:
        idx = torch.multinomial(probs, num_samples=1)
    else:
        _, idx = torch.topk(probs, k=1, dim=-1)
    return idx, probs

def generate(model, c_indices, max_new_tokens, nearest_latent, emb_masks=None, cfg_scale=4.0, warmup_steps=0, adaptive_func='const', coeff_a=1, coeff_b=1, **sampling_kwargs):
    model.eval()
    accept_length_list = []

    with torch.no_grad():
        device = c_indices.device
        if cfg_scale > 1.0:
            cond_null = torch.zeros_like(c_indices) + model.base_model.model.cls_embedding.uncond_embedding
            cond_combined = torch.cat([c_indices, cond_null])
        else:
            cond_combined = c_indices
        T = cond_combined.shape[1] 
        
        max_batch_size = c_indices.shape[0]
        
        if emb_masks is not None:
            assert emb_masks.shape[0] == max_batch_size
            assert emb_masks.shape[-1] == T
            if cfg_scale > 1.0:
                attention_mask = torch.cat([emb_masks, emb_masks])
            else:
                attention_mask =  emb_masks
        
        # seq = torch.empty((max_batch_size, max_new_tokens), dtype=torch.int, device=device)
        padding = (torch.zeros(1,1,dtype=torch.long)-1).to(cond_combined.device)
        model.ea_layer.reset_kv()

        if sampling_kwargs['temperature'] > 1e-5:
            logits_processor = prepare_logits_processor(temperature=sampling_kwargs['temperature'], top_k=sampling_kwargs['top_k'], top_p=sampling_kwargs['top_p'])
        else:
            logits_processor = None

        if hasattr(model, "past_key_values"):
            past_key_values = model.past_key_values
            past_key_values_data = model.past_key_values_data
            current_length_data = model.current_length_data
            # Reset the past key and value states
            current_length_data.zero_()
        else:
            (
                past_key_values,
                past_key_values_data,
                current_length_data,
            ) = initialize_past_key_values(model.base_model)
            model.past_key_values = past_key_values
            model.past_key_values_data = past_key_values_data
            model.current_length_data = current_length_data
        new_token = 1
        cur_length = 1

        reset_tree_mode(model)
        
        draft_tokens, retrieve_indices,tree_mask,tree_position_ids, logits, hidden_state, sample_token = initialize_tree(
            cond_combined, model, past_key_values, logits_processor, cfg_scale, -1, attention_mask,
        )

        # ! Checked until here

        max_steps = max_new_tokens
        input_ids = torch.zeros((max_batch_size, args.cls_token_num), dtype=torch.long).to(cond_combined.device)
        for idx in range(max_steps): # idx: new decoding steps
            model.base_model.model.tree_mask = tree_mask

            tree_draft_tokens = torch.cat([draft_tokens, draft_tokens]).to(input_ids.device)

            logits, hidden_state_new, outputs = tree_decoding(
                model,
                tree_draft_tokens,
                past_key_values,
                tree_position_ids,
                input_ids,
                retrieve_indices,
                cfg_scale,
                -1,
                attention_mask
            )
            draft_tokens=torch.cat((draft_tokens,padding),dim=1)
            candidates=draft_tokens[0,retrieve_indices]
            best_candidate, accept_length, sample_p = evaluate_posterior_with_nearest_latent(
                logits, candidates, logits_processor, nearest_latent, adaptive_func, coeff_a, coeff_b, warmup_steps, input_ids.shape[1]
            )
            input_ids, draft_tokens, retrieve_indices,tree_mask,tree_position_ids, new_token, hidden_state, sample_token = update_inference_inputs(
                input_ids,
                candidates,
                best_candidate,
                accept_length,
                retrieve_indices,
                logits_processor,
                new_token,
                past_key_values_data,
                current_length_data,
                model,
                hidden_state_new,
                sample_p,
                cfg_scale,
                -1,
            )
            # accept_length_tree = input_ids.shape[1] - cur_length
            # cur_length = accept_length_tree + cur_length
            # accept_length_list.append(accept_length_tree)
            if torch.is_tensor(accept_length):
                accept_length_list.append(accept_length.item()+1)
            else:
                accept_length_list.append(accept_length+1)
            if new_token > max_new_tokens:
                break
        return input_ids[:, 120:max_new_tokens+120], accept_length_list

def main(args):
    # Setup PyTorch:
    assert torch.cuda.is_available(), "Sampling with DDP requires at least one GPU. sample.py supports CPU-only usage"
    torch.set_grad_enabled(False)

    # Setup DDP:
    dist.init_process_group("nccl")
    rank = dist.get_rank()
    device = rank % torch.cuda.device_count()
    seed = args.global_seed * dist.get_world_size() + rank
    torch.manual_seed(seed)
    torch.cuda.set_device(device)
    print(f"Starting rank={rank}, seed={seed}, world_size={dist.get_world_size()}.")

    # create and load model
    vq_model = VQ_models[args.vq_model](
        codebook_size=args.codebook_size,
        codebook_embed_dim=args.codebook_embed_dim)
    vq_model.to(device)
    vq_model.eval()
    checkpoint = torch.load(args.vq_ckpt, map_location="cpu")
    vq_model.load_state_dict(checkpoint["model"])
    del checkpoint
    print(f"image tokenizer is loaded")

    # create and load gpt model
    precision = {'none': torch.float32, 'bf16': torch.bfloat16, 'fp16': torch.float16}[args.precision]
    latent_size = args.image_size // args.downsample_size
    gpt_model = EaModel.from_pretrained(
        base_model_path=args.gpt_base_ckpt,
        ea_model_path=args.gpt_ckpt,
        total_token=args.total_token,
    ).to(device=device, dtype=precision)

    print(f"gpt model is loaded")

    if args.compile:
        print(f"compiling the model...")
        gpt_model = torch.compile(
            gpt_model,
            mode="reduce-overhead",
            fullgraph=True
        ) # requires PyTorch 2.0 (optional)
    else:
        print(f"no need to compile model in demo") 
    
    # assert os.path.exists(args.t5_path)
    # t5_model = T5Embedder(
    #     device=device, 
    #     local_cache=True, 
    #     cache_dir=args.t5_path, 
    #     dir_or_name=args.t5_model_type,
    #     torch_dtype=precision,
    #     model_max_length=args.t5_feature_max_len,
    # )
    # print(f"t5 model is loaded")

    # Create folder to save samples:
    ckpt_string_name = os.path.basename(args.gpt_ckpt).replace(".pth", "").replace(".pt", "")
    prompt_name = args.prompt_csv.split('/')[-1].split('.')[0].lower()
    folder_name = f"{ckpt_string_name}-{prompt_name}-size-{args.image_size}-size-{args.image_size}-{args.vq_model}-" \
                  f"topk-{args.top_k}-topp-{args.top_p}-temperature-{args.temperature}-" \
                  f"cfg-{args.cfg_scale}-seed-{args.global_seed}-warmup-{args.warmup_steps}-adative-{args.adaptive_func}-a-{args.coeff_a}-b-{args.coeff_b}"
    sample_folder_dir = f"{args.sample_dir}/{folder_name}"
    if rank == 0:
        os.makedirs(f"{sample_folder_dir}/images", exist_ok=True)
        print(f"Saving .png samples at {sample_folder_dir}/images")
    dist.barrier()
    
    prompt_feature_list = [os.path.join(args.prompt_csv, f'{index:06d}.npy') for index in range(30000)]
    df = pd.read_csv('./evaluations/t2i/coco_captions.csv', sep='\t')
    prompt_list = df['Prompt'].tolist()

    # Figure out how many samples we need to generate on each GPU and how many iterations we need to run:
    n = args.per_proc_batch_size
    global_batch_size = n * dist.get_world_size()
    num_fid_samples = min(args.num_fid_samples, len(prompt_feature_list))
    # To make things evenly-divisible, we'll sample a bit more than we need and then discard the extra samples:
    total_samples = int(math.ceil(num_fid_samples / global_batch_size) * global_batch_size)
    if rank == 0:
        print(f"Total number of images that will be sampled: {total_samples}")
    assert total_samples % dist.get_world_size() == 0, "total_samples must be divisible by world_size"
    samples_needed_this_gpu = int(total_samples // dist.get_world_size())
    assert samples_needed_this_gpu % n == 0, "samples_needed_this_gpu must be divisible by the per-GPU batch size"
    iterations = int(samples_needed_this_gpu // n)
    pbar = range(iterations)
    pbar = tqdm(pbar) if rank == 0 else pbar
    total = 0
    if args.adaptive_func == 'jsd_cos' or args.adaptive_func == 'tvd_cos':
        nearest_latent = torch.from_numpy(np.load('/path/to/t2i_top_16383_similar_indices.npy')).to(device)
    else:
        nearest_latent = torch.from_numpy(gpt_model.nearest_latent).to(device)
    for _ in pbar:
        # Select text prompt
        missing_png = False
        for i in range(n):
            index = i * dist.get_world_size() + rank + total
            if not os.path.exists(f"{sample_folder_dir}/images/{index:06d}.png"):
                missing_png = True
                break
        if not missing_png:
            total += global_batch_size
            continue
        
        prompt_batch = []
        emb_masks = []
        for i in range(n):
            index = i * dist.get_world_size() + rank + total
            prompt_elem = prompt_feature_list[index] if index < len(prompt_feature_list) else prompt_feature_list[-1]
            caption_emb = np.load(prompt_elem)
            caption_emb = torch.from_numpy(caption_emb).to(device=device, dtype=precision)
            caption_padding = args.cls_token_num - caption_emb.shape[1]
            if caption_padding > 0:
                zero_padding = torch.zeros(caption_emb.shape[0], caption_padding, caption_emb.shape[2], device=device, dtype=precision)
                caption_emb = torch.cat([zero_padding, caption_emb], dim=1)
            prompt_batch.append(caption_emb)
            emb_mask = torch.ones(caption_emb.shape[0], caption_emb.shape[1], device=device, dtype=precision)
            emb_mask[:, :caption_padding] = 0
            emb_masks.append(emb_mask)
        emb_masks = torch.cat(emb_masks)
        caption_embs = torch.cat(prompt_batch)
        
        new_caption_embs, new_emb_masks = caption_embs, emb_masks

        c_indices = new_caption_embs * new_emb_masks[:,:, None]
        c_emb_masks = new_emb_masks

        qzshape = [len(c_indices), args.codebook_embed_dim, latent_size, latent_size]
        t1 = time.time()
        index_sample, accepted_length_list = generate(
            gpt_model, c_indices, latent_size ** 2, 
            nearest_latent,
            c_emb_masks, 
            cfg_scale=args.cfg_scale,
            warmup_steps=args.warmup_steps, adaptive_func=args.adaptive_func,
            coeff_a=args.coeff_a, coeff_b=args.coeff_b,
            temperature=args.temperature, top_k=args.top_k,
            top_p=args.top_p, sample_logits=True, 
            )
        sampling_time = time.time() - t1
        t2 = time.time()
        samples = vq_model.decode_code(index_sample, qzshape) # output value is between [-1, 1]
        decoder_time = time.time() - t2
        samples = torch.clamp(127.5 * samples + 128.0, 0, 255).permute(0, 2, 3, 1).to("cpu", dtype=torch.uint8).numpy()
        
        # Save samples to disk as individual .png files
        for i, sample in enumerate(samples):
            index = i * dist.get_world_size() + rank + total
            Image.fromarray(sample).save(f"{sample_folder_dir}/images/{index:06d}.png")
        total += global_batch_size
        sampling_stats = {}
        sampling_stats["sampling_time"] = sampling_time
        sampling_stats["decoder_time"] = decoder_time
        sampling_stats["mean_accept_length"] = sum(accepted_length_list) / len(accepted_length_list)
        sampling_stats["accept_length"] = accepted_length_list
        with open(f"{sample_folder_dir}/sampling_stats.json", "a") as f:
            f.write(json.dumps(sampling_stats) + "\n")

    # Make sure all processes have finished saving their samples before attempting to convert to .npz
    dist.barrier()
    if rank == 0:
        # Save infer result in a jsonl file
        json_items = []
        for idx, prompt in enumerate(prompt_list):
            image_path = os.path.join(sample_folder_dir, "images", f"{idx:06d}.png")
            json_items.append({"text": prompt, "image_path": image_path})
        res_jsonl_path = os.path.join(sample_folder_dir, "result.jsonl")
        print(f"Save jsonl to {res_jsonl_path}...")
        with open(res_jsonl_path, "w") as f:
            for item in json_items:
                f.write(json.dumps(item) + "\n")

        # Save captions to txt
        caption_path = os.path.join(sample_folder_dir, "captions.txt")
        print(f"Save captions to {caption_path}...")
        with open(caption_path, "w") as f:
            for item in prompt_list:
                f.write(f"{item}\n")
        print("Done.")
    
    dist.barrier()
    dist.destroy_process_group()



if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--prompt-csv", type=str, default='/path/to/mscoco_text_features')
    parser.add_argument("--t5-feature-max-len", type=int, default=120)
    parser.add_argument("--t5-feature-dim", type=int, default=2048)
    parser.add_argument("--no-left-padding", action='store_true', default=False)
    parser.add_argument("--gpt-ckpt", type=str, default=None)
    parser.add_argument("--gpt-base-ckpt", type=str, default='/path/to/LlamaGen-T2I')
    parser.add_argument("--total-token", type=int, default=60)
    parser.add_argument("--gpt-type", type=str, choices=['c2i', 't2i'], default="t2i", help="class->image or text->image")  
    parser.add_argument("--cls-token-num", type=int, default=120, help="max token number of condition input")
    parser.add_argument("--precision", type=str, default='bf16', choices=["none", "fp16", "bf16"]) 
    parser.add_argument("--compile", action='store_true', default=False)
    parser.add_argument("--vq-model", type=str, choices=list(VQ_models.keys()), default="VQ-16")
    parser.add_argument("--vq-ckpt", type=str, default=None, help="ckpt path for vq model")
    parser.add_argument("--codebook-size", type=int, default=16384, help="codebook size for vector quantization")
    parser.add_argument("--codebook-embed-dim", type=int, default=8, help="codebook dimension for vector quantization")
    parser.add_argument("--image-size", type=int, choices=[256, 384, 512], default=256)
    parser.add_argument("--downsample-size", type=int, choices=[8, 16], default=16)
    parser.add_argument("--num-classes", type=int, default=1000)
    parser.add_argument("--cfg-scale", type=float, default=7.5)
    parser.add_argument("--sample-dir", type=str, default="samples_coco", help="samples_coco or samples_parti")
    parser.add_argument("--per-proc-batch-size", type=int, default=1)
    parser.add_argument("--num-fid-samples", type=int, default=30000)
    parser.add_argument("--global-seed", type=int, default=0)
    parser.add_argument("--top-k", type=int, default=1000, help="top-k value to sample with")
    parser.add_argument("--temperature", type=float, default=1.0, help="temperature value to sample with")
    parser.add_argument("--top-p", type=float, default=1.0, help="top-p value to sample with")
    parser.add_argument("--warmup-steps", type=int, default=0)
    parser.add_argument("--adaptive-func", type=str, default="constant")
    parser.add_argument("--coeff-a", type=float, default=1)
    parser.add_argument("--coeff-b", type=float, default=1)
    args = parser.parse_args()
    main(args)
