import torch
torch.backends.cuda.matmul.allow_tf32 = True
torch.backends.cudnn.allow_tf32 = True
torch.set_float32_matmul_precision('high')
setattr(torch.nn.Linear, 'reset_parameters', lambda self: None)     # disable default parameter init for faster speed
setattr(torch.nn.LayerNorm, 'reset_parameters', lambda self: None)  # disable default parameter init for faster speed
import torch.nn.functional as F
import torch.distributed as dist

import os
import math
import json
import argparse
import pandas as pd
from tqdm import tqdm
from PIL import Image

import sys
sys.path.append('.')

from tokenizer.tokenizer_image.vq_model import VQ_models
from autoregressive.models.modeling_llama import LlamaForCausalLM
import numpy as np
os.environ["TOKENIZERS_PARALLELISM"] = "false"

def cfg_logit_process(combined_logits, cfg_scale=4.0, cfg_interval=-1):
    cond_logits, uncond_logits = torch.split(combined_logits, len(combined_logits) // 2, dim=0)
    logits = uncond_logits + (cond_logits - uncond_logits) * cfg_scale
    return logits


def top_k_top_p_filtering(
    logits,
    top_k: int = 0,
    top_p: float = 1.0,
    filter_value: float = -float("Inf"),
    min_tokens_to_keep: int = 1,
):
    """Filter a distribution of logits using top-k and/or nucleus (top-p) filtering
    Args:
        logits: logits distribution shape (batch size, vocabulary size)
        if top_k > 0: keep only top k tokens with highest probability (top-k filtering).
        if top_p < 1.0: keep the top tokens with cumulative probability >= top_p (nucleus filtering).
            Nucleus filtering is described in Holtzman et al. (http://arxiv.org/abs/1904.09751)
        Make sure we keep at least min_tokens_to_keep per batch example in the output
    From: https://gist.github.com/thomwolf/1a5a29f6962089e871b94cbd09daf317
    """
    if top_k > 0:
        top_k = min(max(top_k, min_tokens_to_keep), logits.size(-1))  # Safety check
        # Remove all tokens with a probability less than the last token of the top-k
        indices_to_remove = logits < torch.topk(logits, top_k)[0][..., -1, None]
        # logits = torch.ones_like(logits).to(logits.device)
        logits[indices_to_remove] = filter_value

    if top_p < 1.0:
        sorted_logits, sorted_indices = torch.sort(logits, descending=True)
        cumulative_probs = torch.cumsum(F.softmax(sorted_logits, dim=-1), dim=-1)

        # Remove tokens with cumulative probability above the threshold (token with 0 are kept)
        sorted_indices_to_remove = cumulative_probs > top_p
        if min_tokens_to_keep > 1:
            # Keep at least min_tokens_to_keep (set to min_tokens_to_keep-1 because we add the first one below)
            sorted_indices_to_remove[..., :min_tokens_to_keep] = 0
        # Shift the indices to the right to keep also the first token above the threshold
        sorted_indices_to_remove[..., 1:] = sorted_indices_to_remove[..., :-1].clone()
        sorted_indices_to_remove[..., 0] = 0

        # scatter sorted tensors to original indexing
        indices_to_remove = sorted_indices_to_remove.scatter(1, sorted_indices, sorted_indices_to_remove)
        logits[indices_to_remove] = filter_value
    return logits


def sample(logits, temperature: float=1.0, top_k: int=0, top_p: float=1.0, sample_logits=True):        
    logits = logits[:, -1, :] / max(temperature, 1e-5)
    if top_k > 0 or top_p < 1.0:
        logits = top_k_top_p_filtering(logits, top_k=top_k, top_p=top_p)
    probs = F.softmax(logits, dim=-1)
    if sample_logits:
        idx = torch.multinomial(probs, num_samples=1)
    else:
        _, idx = torch.topk(probs, k=1, dim=-1)
    return idx, probs

def generate(model, c_indices, max_new_tokens, emb_masks=None, cfg_scale=4.0, **sampling_kwargs):
    model.eval()
    with torch.no_grad():
        device = c_indices.device
        if model.model.model_type == 'c2i':
            if cfg_scale > 1.0:
                cond_null = torch.ones_like(c_indices) * model.model.num_classes
                cond_combined = torch.cat([c_indices, cond_null])
            else:
                cond_combined = c_indices
        elif model.model.model_type == 't2i':
            if cfg_scale > 1.0:
                cond_null = torch.zeros_like(c_indices) + model.model.cls_embedding.uncond_embedding
                cond_combined = torch.cat([c_indices, cond_null])
            else:
                cond_combined = c_indices
            T = cond_combined.shape[1]      
        else:
            raise NotImplementedError(f"input type {model.model.input_type} is not supported")
        max_batch_size = c_indices.shape[0]
        
        if emb_masks is not None:
            assert emb_masks.shape[0] == max_batch_size
            assert emb_masks.shape[-1] == T
            if cfg_scale > 1.0:
                attention_mask = torch.cat([emb_masks, emb_masks])
            else:
                attention_mask =  emb_masks
        
        seq = torch.empty((max_batch_size, max_new_tokens), dtype=torch.int, device=device)
        output = model(cond_idx=cond_combined, attention_mask=attention_mask)
        combined_logits = output.logits
        past_key_values = output.past_key_values
        cond_logits, uncond_logits = torch.split(combined_logits, len(combined_logits) // 2, dim=0)
        logits = uncond_logits + (cond_logits - uncond_logits) * cfg_scale
        next_token, _ = sample(logits, **sampling_kwargs)
        seq[:, 0] = next_token.squeeze(-1)
        for i in range(max_new_tokens-1):
            attention_mask = torch.cat([attention_mask, torch.ones_like(attention_mask)[:, :1]], dim=-1)
            input_ids = torch.cat([next_token, next_token])
            output = model(input_ids=input_ids, past_key_values=past_key_values, attention_mask=attention_mask)
            combined_logits = output.logits
            past_key_values = output.past_key_values
            cond_logits, uncond_logits = torch.split(combined_logits, len(combined_logits) // 2, dim=0)
            logits = uncond_logits + (cond_logits - uncond_logits) * cfg_scale
            next_token, _ = sample(logits, **sampling_kwargs)
            seq[:, i+1] = next_token.squeeze(-1)
            
    return seq

def main(args):
    # Setup PyTorch:
    assert torch.cuda.is_available(), "Sampling with DDP requires at least one GPU. sample.py supports CPU-only usage"
    torch.set_grad_enabled(False)

    # Setup DDP:
    dist.init_process_group("nccl")
    rank = dist.get_rank()
    device = rank % torch.cuda.device_count()
    seed = args.global_seed * dist.get_world_size() + rank
    torch.manual_seed(seed)
    torch.cuda.set_device(device)
    print(f"Starting rank={rank}, seed={seed}, world_size={dist.get_world_size()}.")

    # create and load model
    vq_model = VQ_models[args.vq_model](
        codebook_size=args.codebook_size,
        codebook_embed_dim=args.codebook_embed_dim)
    vq_model.to(device)
    vq_model.eval()
    checkpoint = torch.load(args.vq_ckpt, map_location="cpu")
    vq_model.load_state_dict(checkpoint["model"])
    del checkpoint
    print(f"image tokenizer is loaded")

    # create and load gpt model
    precision = {'none': torch.float32, 'bf16': torch.bfloat16, 'fp16': torch.float16}[args.precision]
    latent_size = args.image_size // args.downsample_size
    gpt_model = LlamaForCausalLM.from_pretrained(
        args.gpt_ckpt,
    ).to(device=device, dtype=precision)

    print(f"gpt model is loaded")

    if args.compile:
        print(f"compiling the model...")
        gpt_model = torch.compile(
            gpt_model,
            mode="reduce-overhead",
            fullgraph=True
        ) # requires PyTorch 2.0 (optional)
    else:
        print(f"no need to compile model in demo") 
    
    # assert os.path.exists(args.t5_path)
    # t5_model = T5Embedder(
    #     device=device, 
    #     local_cache=True, 
    #     cache_dir=args.t5_path, 
    #     dir_or_name=args.t5_model_type,
    #     torch_dtype=precision,
    #     model_max_length=args.t5_feature_max_len,
    # )
    # print(f"t5 model is loaded")

    # Create folder to save samples:
    ckpt_string_name = os.path.basename(args.gpt_ckpt).replace(".pth", "").replace(".pt", "")
    prompt_name = args.prompt_csv.split('/')[-1].split('.')[0].lower()
    folder_name = f"{ckpt_string_name}-{prompt_name}-size-{args.image_size}-size-{args.image_size}-{args.vq_model}-" \
                  f"topk-{args.top_k}-topp-{args.top_p}-temperature-{args.temperature}-" \
                  f"cfg-{args.cfg_scale}-seed-{args.global_seed}"
    sample_folder_dir = f"{args.sample_dir}/{folder_name}"
    if rank == 0:
        os.makedirs(f"{sample_folder_dir}/images", exist_ok=True)
        print(f"Saving .png samples at {sample_folder_dir}/images")
    dist.barrier()
    
    prompt_feature_list = [os.path.join(args.prompt_csv, f'{index:06d}.npy') for index in range(30000)]
    df = pd.read_csv('./evaluations/t2i/coco_captions.csv', sep='\t')
    prompt_list = df['Prompt'].tolist()

    # Figure out how many samples we need to generate on each GPU and how many iterations we need to run:
    n = args.per_proc_batch_size
    global_batch_size = n * dist.get_world_size()
    num_fid_samples = min(args.num_fid_samples, len(prompt_feature_list))
    # To make things evenly-divisible, we'll sample a bit more than we need and then discard the extra samples:
    total_samples = int(math.ceil(num_fid_samples / global_batch_size) * global_batch_size)
    if rank == 0:
        print(f"Total number of images that will be sampled: {total_samples}")
    assert total_samples % dist.get_world_size() == 0, "total_samples must be divisible by world_size"
    samples_needed_this_gpu = int(total_samples // dist.get_world_size())
    assert samples_needed_this_gpu % n == 0, "samples_needed_this_gpu must be divisible by the per-GPU batch size"
    iterations = int(samples_needed_this_gpu // n)
    pbar = range(iterations)
    pbar = tqdm(pbar) if rank == 0 else pbar
    total = 0
    for _ in pbar:
        # Select text prompt
        missing_png = False
        for i in range(n):
            index = i * dist.get_world_size() + rank + total
            if not os.path.exists(f"{sample_folder_dir}/images/{index:06d}.png"):
                missing_png = True
                break
        if not missing_png:
            total += global_batch_size
            continue
        prompt_batch = []
        emb_masks = []
        for i in range(n):
            index = i * dist.get_world_size() + rank + total
            prompt_elem = prompt_feature_list[index] if index < len(prompt_feature_list) else prompt_feature_list[-1]
            caption_emb = np.load(prompt_elem)
            caption_emb = torch.from_numpy(caption_emb).to(device=device, dtype=precision)
            caption_padding = args.cls_token_num - caption_emb.shape[1]
            if caption_padding > 0:
                zero_padding = torch.zeros(caption_emb.shape[0], caption_padding, caption_emb.shape[2], device=device, dtype=precision)
                caption_emb = torch.cat([zero_padding, caption_emb], dim=1)
            prompt_batch.append(caption_emb)
            emb_mask = torch.ones(caption_emb.shape[0], caption_emb.shape[1], device=device, dtype=precision)
            emb_mask[:, :caption_padding] = 0
            emb_masks.append(emb_mask)
        emb_masks = torch.cat(emb_masks)
        caption_embs = torch.cat(prompt_batch)
        
        new_caption_embs, new_emb_masks = caption_embs, emb_masks

        c_indices = new_caption_embs * new_emb_masks[:,:, None]
        c_emb_masks = new_emb_masks

        qzshape = [len(c_indices), args.codebook_embed_dim, latent_size, latent_size]
        index_sample = generate(
            gpt_model, c_indices, latent_size ** 2, 
            c_emb_masks,
            cfg_scale=args.cfg_scale,
            temperature=args.temperature, top_k=args.top_k,
            top_p=args.top_p, sample_logits=True, 
            )
        
        samples = vq_model.decode_code(index_sample, qzshape) # output value is between [-1, 1]
        samples = torch.clamp(127.5 * samples + 128.0, 0, 255).permute(0, 2, 3, 1).to("cpu", dtype=torch.uint8).numpy()
        
        # Save samples to disk as individual .png files
        for i, sample in enumerate(samples):
            index = i * dist.get_world_size() + rank + total
            Image.fromarray(sample).save(f"{sample_folder_dir}/images/{index:06d}.png")
        total += global_batch_size

    # Make sure all processes have finished saving their samples before attempting to convert to .npz
    dist.barrier()
    if rank == 0:
        # Save infer result in a jsonl file
        json_items = []
        for idx, prompt in enumerate(prompt_list):
            image_path = os.path.join(sample_folder_dir, "images", f"{idx:06d}.png")
            json_items.append({"text": prompt, "image_path": image_path})
        res_jsonl_path = os.path.join(sample_folder_dir, "result.jsonl")
        print(f"Save jsonl to {res_jsonl_path}...")
        with open(res_jsonl_path, "w") as f:
            for item in json_items:
                f.write(json.dumps(item) + "\n")

        # Save captions to txt
        caption_path = os.path.join(sample_folder_dir, "captions.txt")
        print(f"Save captions to {caption_path}...")
        with open(caption_path, "w") as f:
            for item in prompt_list:
                f.write(f"{item}\n")
        print("Done.")
    
    dist.barrier()
    dist.destroy_process_group()



if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--prompt-csv", type=str, default='./mscoco_text_features')
    parser.add_argument("--t5-feature-max-len", type=int, default=120)
    parser.add_argument("--t5-feature-dim", type=int, default=2048)
    parser.add_argument("--no-left-padding", action='store_true', default=False)
    parser.add_argument("--gpt-ckpt", type=str, default=None)
    parser.add_argument("--gpt-type", type=str, choices=['c2i', 't2i'], default="t2i", help="class->image or text->image")  
    parser.add_argument("--cls-token-num", type=int, default=120, help="max token number of condition input")
    parser.add_argument("--precision", type=str, default='bf16', choices=["none", "fp16", "bf16"]) 
    parser.add_argument("--compile", action='store_true', default=False)
    parser.add_argument("--vq-model", type=str, choices=list(VQ_models.keys()), default="VQ-16")
    parser.add_argument("--vq-ckpt", type=str, default=None, help="ckpt path for vq model")
    parser.add_argument("--codebook-size", type=int, default=16384, help="codebook size for vector quantization")
    parser.add_argument("--codebook-embed-dim", type=int, default=8, help="codebook dimension for vector quantization")
    parser.add_argument("--image-size", type=int, choices=[256, 384, 512], default=256)
    parser.add_argument("--downsample-size", type=int, choices=[8, 16], default=16)
    parser.add_argument("--num-classes", type=int, default=1000)
    parser.add_argument("--cfg-scale", type=float, default=7.5)
    parser.add_argument("--sample-dir", type=str, default="samples_coco", help="samples_coco or samples_parti")
    parser.add_argument("--per-proc-batch-size", type=int, default=32)
    parser.add_argument("--num-fid-samples", type=int, default=30000)
    parser.add_argument("--global-seed", type=int, default=0)
    parser.add_argument("--top-k", type=int, default=1000, help="top-k value to sample with")
    parser.add_argument("--temperature", type=float, default=1.0, help="temperature value to sample with")
    parser.add_argument("--top-p", type=float, default=1.0, help="top-p value to sample with")
    args = parser.parse_args()
    main(args)
