import torch
import torch.nn as nn
from tqdm import tqdm
import yaml
import argparse
from torch_geometric.data import DataLoader
from torch_geometric.nn.pool import global_mean_pool, global_add_pool
from torch.utils.data.distributed import DistributedSampler
from torch.nn.parallel import DistributedDataParallel
import torch.distributed as dist
from torchmetrics.aggregation import CatMetric
from easydict import EasyDict
import wandb
import os
import shutil
import pickle
import sys
sys.path.append('./')

from experiments.fixddp import DistributedSamplerNoDuplicate
from datasets.cmu import CMU
from models.EGTN import EGTN
from diffusion.GeoTDM_base import GeoTDM, ModelMeanType, ModelVarType, LossType
from utils.misc import set_seed, gather_across_gpus

os.environ["WANDB__SERVICE_WAIT"] = "300"
torch.multiprocessing.set_sharing_strategy('file_system')

opt_mode = 'node'

def run(rank, world_size, args):
    # torch.autograd.set_detect_anomaly(True)
    # Load args
    yaml_file = args.train_yaml_file
    with open(yaml_file, 'r') as f:
        params = yaml.safe_load(f)
    config = EasyDict(params)
    
    # Init model and optimizer
    denoise_network = EGTN(**config.denoise_model).to(rank)
    print("init denoise_network")
    diffusion = GeoTDM(denoise_network=denoise_network,
                       model_mean_type=ModelMeanType.EPSILON,
                       model_var_type=ModelVarType.FIXED_LARGE,
                       loss_type=LossType.MSE,
                       device=rank,
                       rescale_timesteps=False,
                       **config.diffusion)
    print("init diffusion")
    
    if world_size > 1:
        denoise_network = DistributedDataParallel(denoise_network, device_ids=[rank] )
        print("init Distributed")
    optimizer = torch.optim.Adam(denoise_network.parameters(), lr=config.train.lr)
    print("init optimizer")

    # Load dataset
    dataset_train = CMU(**config.data.train)
    dataset_val = CMU(**config.data.val)
    
    task_name = config.data.train.act

      # Save args yaml file
    name_path = os.path.join(config.train.output_base_path, task_name)
    output_path = os.path.join(name_path, config.train.exp_name)
    print(output_path)
    if config.train.train_mode == True:
        if rank == 0:
            if not os.path.exists(output_path):
                os.makedirs(output_path, exist_ok=True)

            shutil.copy(yaml_file, output_path)

        set_seed(config.train.seed)
        
        # Load dataset

        if world_size > 1:
            sampler_train = DistributedSampler(dataset_train)
            sampler_val = DistributedSamplerNoDuplicate(dataset_val, shuffle=False, drop_last=False)
        else:
            sampler_train = None
            sampler_val = None

        dataloader_train = DataLoader(dataset_train, batch_size=config.train.batch_size // world_size,
                                    shuffle=(sampler_train is None), sampler=sampler_train, pin_memory=True)
        dataloader_val = DataLoader(dataset_val, batch_size=config.train.eval_batch_size // world_size, shuffle=False,
                                    sampler=sampler_val)

        


        if rank == 0:
            # Wandb config
            if config.wandb.no_wandb:
                mode = 'disabled'
            else:
                mode = 'online'
            if config.train.train_mode == True:
                kwargs_train = {'entity': config.wandb.wandb_usr, 'name': task_name + "_" + config.train.exp_name, 'project': config.wandb.project,
                        'config': params, 'settings': wandb.Settings(_disable_stats=True), 'mode': mode}
            train_log = wandb.init(**kwargs_train)
            train_log.save('*.txt')

        # Start training
        num_epochs = config.train.num_epochs
        tot_step = 0

        best_val_final_error = 1e10
        reduce_placeholder = CatMetric()

        if rank == 0:
            progress_bar = tqdm(total=num_epochs)

        calc_nll = False

        for epoch in range(1, num_epochs + 1):
            if rank == 0:
                # print(f'Start epoch {epoch}')
                progress_bar.set_description(f"Epoch {epoch}")
            denoise_network.train()
            sampler_train.set_epoch(epoch)

            # Training

            train_loss_epoch, counter = torch.zeros(1).to(rank), torch.zeros(1).to(rank)

            for step, data in enumerate(dataloader_train):
                tot_step += 1

                data = data.to(rank)
                model_kwargs = {'h': data.h,
                                'edge_index': data.edge_index,
                                'edge_attr': data.edge_attr,
                                'batch': data.batch}
                x_start = data.x
                # Construct cond mask
                cond_mask = torch.zeros(1, 1, x_start.size(-1)).to(x_start)
                for interval in config.train.cond_mask:
                    cond_mask[..., interval[0]: interval[1]] = 1
                model_kwargs['cond_mask'] = cond_mask
                model_kwargs['x_given'] = x_start
                x_start_ = x_start[..., ~cond_mask.view(-1).bool()]
                training_losses = diffusion.training_losses(x_start=x_start_, t=None, model_kwargs=model_kwargs)
                loss = training_losses['loss']  # [BN]
                if opt_mode == 'graph':  # graph-wise loss
                    loss = global_mean_pool(loss, data.batch)  # [B]

                step_loss_synced = gather_across_gpus(loss, reduce_placeholder).mean().item()
                if rank == 0 and tot_step % config.train.log_every_step == 0:
                    train_log.log({"Step train loss": step_loss_synced}, commit=True, step=tot_step)
                    logs = {"loss": step_loss_synced, "step": tot_step}
                    progress_bar.set_postfix(**logs)

                train_loss_epoch = train_loss_epoch + loss.sum()
                counter = counter + loss.size(0)

                loss = loss.mean()
                loss.backward()
                nn.utils.clip_grad_norm_(denoise_network.parameters(), 1.0)
                optimizer.step()
                optimizer.zero_grad()

            train_loss_epoch = gather_across_gpus(train_loss_epoch, reduce_placeholder).sum().item()
            counter = gather_across_gpus(counter, reduce_placeholder).sum().item()

            if rank == 0:
                train_log.log({"Epoch train loss": train_loss_epoch / counter}, commit=True)

            # Eval on validation set
            if epoch % config.train.eval_every_epoch == 0 and epoch != 0:
                # Start testing
                denoise_network.eval()

                if calc_nll:
                    test_nll_epoch_all, test_mse_epoch_all = [], []  # [B_tot]
                Error_K_all = []  # [B_tot, K, T]
                system_id_all = []  # the index in the test dataset
                reduce_placeholder = CatMetric()

                for step, data in tqdm(enumerate(dataloader_val), disable=rank != 0):
                    data = data.to(rank)
                    model_kwargs = {'h': data.h,
                                    'edge_index': data.edge_index,
                                    'edge_attr': data.edge_attr,
                                    'batch': data.batch}

                    x_start = data.x

                    # Create temporal inpainting mask, 1 to keep the entries unchanged, 0 to modify it by diffusion
                    cond_mask = torch.zeros(1, 1, x_start.size(-1)).to(x_start)
                    for interval in config.train.cond_mask:
                        cond_mask[..., interval[0]: interval[1]] = 1
                    model_kwargs['cond_mask'] = cond_mask
                    model_kwargs['x_given'] = x_start

                    x_start_ = x_start[..., ~cond_mask.view(-1).bool()]
                    if calc_nll:
                        val_results = diffusion.calc_bpd_loop(x_start=x_start_, model_kwargs=model_kwargs, progress=False)
                        total_bpd = val_results['total_bpd']  # [BN]
                        mse = val_results['mse'].mean(dim=1)  # [BN, T] -> [BN]

                        total_bpd = global_add_pool(total_bpd, data.batch)  # [B]
                        mse = global_mean_pool(mse, data.batch)  # [B]
                        test_nll_epoch_all.append(total_bpd)
                        test_mse_epoch_all.append(mse)

                    x_target = x_start[..., ~cond_mask.view(-1).bool()]
                    shape_to_pred = x_target.shape  # [BN, 3, T_p]

                    Error_K = []

                    # Compute traj distance
                    for k in range(config.train.K):
                        x_out = diffusion.p_sample_loop(shape=shape_to_pred, progress=False,
                                                        model_kwargs=model_kwargs)  # [BN, 3, T_p]
                        # x_out = torch.cat((x_start[..., cond_mask.view(-1).bool()], x_out), dim=-1)
                        distance = (x_out - x_target).square().sum(dim=1).sqrt()  # [BN, T_p]
                        distance = global_mean_pool(distance, data.batch)  # [B, T_p]
                        Error_K.append(distance)

                    # Compute minADE, minFDE
                    Error_K = torch.stack(Error_K, dim=-1)  # [B, T_p, K]
                    system_id_all.append(data.system_id)  # [B]
                    Error_K_all.append(Error_K)

                # Analyze
                Error_K_all = torch.cat(Error_K_all, dim=0)  # [B_tot, T_p, K]
                Error_min_all = Error_K_all.min(dim=2).values  # [B_tot, T_p]
                Error_ave_all = Error_K_all.mean(dim=2)  # [B_tot, T_p]
                if calc_nll:
                    nll_all = torch.cat(test_nll_epoch_all, dim=0)  # [B_tot]
                    eps_mse_all = torch.cat(test_mse_epoch_all, dim=0)  # [B_tot]
                system_id_all = torch.cat(system_id_all, dim=0)  # [B_tot]

                results = {}

                if world_size > 1:
                    if calc_nll:
                        nll_all = gather_across_gpus(nll_all, reduce_placeholder)
                        eps_mse_all = gather_across_gpus(eps_mse_all, reduce_placeholder)
                    system_id_all = gather_across_gpus(system_id_all, reduce_placeholder)

                if calc_nll:
                    results['nll'] = nll_all.mean().item()
                results['system_id_range'] = [system_id_all.min().item(), system_id_all.max().item()]

                eval_index = [2, 4, 8, 10, 14, 25]

                eval_steps = {f'{_*40}ms': _ - 1 for _ in eval_index}

                for key in eval_steps:
                    cur_Error_min = Error_min_all[..., eval_steps[key]]  # [B_tot]

                    # Reduce from all gpus and compute metrics
                    if world_size > 1:
                        reduce_placeholder = reduce_placeholder.to(rank)
                        cur_Error_min = gather_across_gpus(cur_Error_min, reduce_placeholder)  # [B_tot * num_gpus]

                    results[f'min_{config.train.K}_' + key] = cur_Error_min.mean().item() / config.data.test.scale

                    cur_Error_ave = Error_ave_all[..., eval_steps[key]]  # [B_tot]

                    # Reduce from all gpus and compute metrics
                    if world_size > 1:
                        cur_Error_ave = gather_across_gpus(cur_Error_ave, reduce_placeholder)  # [B_tot * num_gpus]

                    results[f'ave_{config.train.K}_' + key] = cur_Error_ave.mean().item() / config.data.test.scale

                if rank == 0:
                    name = f'ave_{config.train.K}_' + '80ms'
                    train_log.log({name: results[name]}, commit=False)
                    name = f'min_{config.train.K}_' + '80ms'
                    train_log.log({name: results[name]}, commit=False)
                    name = f'ave_{config.train.K}_' + '1000ms'
                    train_log.log({name: results[name]}, commit=False)
                    name = f'min_{config.train.K}_' + '1000ms'
                    train_log.log({name: results[name]}, commit=True)

                if rank == 0:
                    print(results)
                    print(f'counter: {system_id_all.size()}')

                    better = False
                    name = f'ave_{config.train.K}_' + '1000ms'
                    final_error = results[name]

                    if final_error < best_val_final_error:
                        best_val_final_error = final_error
                        better = True
                    if better:
                        torch.save(denoise_network.state_dict(),
                                os.path.join(output_path, f'ckpt_best.pt'))

            # Save model
            if rank == 0 and config.train.save_model:
                if epoch % config.train.save_every_epoch == 0:
                    torch.save(denoise_network.state_dict(),
                            os.path.join(output_path, f'ckpt_{epoch}.pt'))
                torch.save(denoise_network.state_dict(),
                        os.path.join(output_path, f'ckpt_last.pt'))

            if world_size > 1:
                dist.barrier()
            if rank == 0:
                progress_bar.update(1)
                
        if rank == 0:
            progress_bar.close()
            train_log.finish()

    # Final testing
    if config.test.final_test:
        if rank == 0:
            print('Start final testing')
        dataset_test = CMU(**config.data.test)
        # Load checkpoint
        test_output_path= os.path.join(output_path, config.test.exp_name)
        
        if not os.path.exists(test_output_path):
            os.makedirs(test_output_path, exist_ok=True)
        if rank == 0:
            shutil.copy(yaml_file, test_output_path)
            
        if world_size > 1:
            sampler = DistributedSamplerNoDuplicate(dataset_test, shuffle=False, drop_last=False)
        else:
            sampler = None
        test_dataloader = DataLoader(dataset_test, batch_size=config.train.eval_batch_size // world_size, shuffle=False,
                                    sampler=sampler)
        
        test_model_ckpt_path = os.path.join(output_path, f'ckpt_{config.train.final_test_ckpt}.pt')
        state_dict = torch.load(test_model_ckpt_path)
        
        denoise_network.load_state_dict(state_dict)
        if rank == 0:
            print(f'Model loaded from {test_model_ckpt_path}')
        denoise_network.eval()

        
        if rank == 0:
            # Wandb test config
            if config.wandb.no_wandb:
                mode = 'disabled'
            else:
                mode = 'online'
            
            kwargs_test = {'entity': config.wandb.wandb_usr, 'name': task_name + "_" + config.test.exp_name, 'project': config.wandb.project,
                    'config': params, 'settings': wandb.Settings(_disable_stats=True), 'mode': mode}
            test_log = wandb.init(**kwargs_test,allow_val_change=True)
            test_log.save('*.txt')
        calc_nll = False
        if calc_nll:
            test_nll_epoch_all, test_mse_epoch_all = [], []  # [B_tot]
        Error_K_all = []  # [B_tot, K, T]
        system_id_all = []  # the index in the test dataset
        reduce_placeholder = CatMetric()

        for step, data in tqdm(enumerate(test_dataloader), disable=rank != 0, total=len(test_dataloader)):
            data = data.to(rank)
            model_kwargs = {'h': data.h,
                            'edge_index': data.edge_index,
                            'edge_attr': data.edge_attr,
                            'batch': data.batch}

            x_start = data.x

            # Create temporal inpainting mask, 1 to keep the entries unchanged, 0 to modify it by diffusion
            cond_mask = torch.zeros(1, 1, x_start.size(-1)).to(x_start)
            for interval in config.train.cond_mask:
                cond_mask[..., interval[0]: interval[1]] = 1
            model_kwargs['cond_mask'] = cond_mask
            model_kwargs['x_given'] = x_start

            x_start_ = x_start[..., ~cond_mask.view(-1).bool()]
            if calc_nll:
                val_results = diffusion.calc_bpd_loop(x_start=x_start_, model_kwargs=model_kwargs, progress=False)
                total_bpd = val_results['total_bpd']  # [BN]
                mse = val_results['mse'].mean(dim=1)  # [BN, T] -> [BN]

                total_bpd = global_add_pool(total_bpd, data.batch)  # [B]
                mse = global_mean_pool(mse, data.batch)  # [B]
                test_nll_epoch_all.append(total_bpd)
                test_mse_epoch_all.append(mse)

            x_target = x_start[..., ~cond_mask.view(-1).bool()]
            shape_to_pred = x_target.shape  # [BN, 3, T_p]

            Error_K = []

            # Compute traj distance
            for k in range(config.train.K):
                x_out = diffusion.p_sample_loop(shape=shape_to_pred, progress=False,
                                                model_kwargs=model_kwargs)  # [BN, 3, T_p]
                # x_out = torch.cat((x_start[..., cond_mask.view(-1).bool()], x_out), dim=-1)
                distance = (x_out - x_target).square().sum(dim=1).sqrt()  # [BN, T_p]
                distance = global_mean_pool(distance, data.batch)  # [B, T_p]
                Error_K.append(distance)

            # Compute minADE, minFDE
            Error_K = torch.stack(Error_K, dim=-1)  # [B, T_p, K]
            system_id_all.append(data.system_id)  # [B]
            Error_K_all.append(Error_K)

        # Analyze
        Error_K_all = torch.cat(Error_K_all, dim=0)  # [B_tot, T_p, K]
        Error_min_all = Error_K_all.min(dim=2).values  # [B_tot, T_p]
        Error_ave_all = Error_K_all.mean(dim=2)  # [B_tot, T_p]
        if calc_nll:
            nll_all = torch.cat(test_nll_epoch_all, dim=0)  # [B_tot]
            eps_mse_all = torch.cat(test_mse_epoch_all, dim=0)  # [B_tot]
        system_id_all = torch.cat(system_id_all, dim=0)  # [B_tot]

        results = {}

        if world_size > 1:
            if calc_nll:
                nll_all = gather_across_gpus(nll_all, reduce_placeholder)
                eps_mse_all = gather_across_gpus(eps_mse_all, reduce_placeholder)
            system_id_all = gather_across_gpus(system_id_all, reduce_placeholder)

        if calc_nll:
            results['nll'] = nll_all.mean().item()
        results['system_id_range'] = [system_id_all.min().item(), system_id_all.max().item()]

        eval_index = [2, 4, 8, 10, 14, 25]

        eval_steps = {f'{_ * 40}ms': _ - 1 for _ in eval_index}

        for key in eval_steps:
            cur_Error_min = Error_min_all[..., eval_steps[key]]  # [B_tot]

            # Reduce from all gpus and compute metrics
            if world_size > 1:
                reduce_placeholder = reduce_placeholder.to(rank)
                cur_Error_min = gather_across_gpus(cur_Error_min, reduce_placeholder)  # [B_tot * num_gpus]

            cur_Error_ave = Error_ave_all[..., eval_steps[key]]  # [B_tot]

            # Reduce from all gpus and compute metrics
            if world_size > 1:
                reduce_placeholder = reduce_placeholder.to(rank)
                cur_Error_ave = gather_across_gpus(cur_Error_ave, reduce_placeholder)  # [B_tot * num_gpus]

            act_index_map = dataset_test.act_index_map
            case_index = system_id_all
            sort_idx = torch.argsort(case_index)
            cur_Error_ave = cur_Error_ave[sort_idx]  # change into the correct sorting
            cur_Error_min = cur_Error_min[sort_idx]
            assert (case_index[sort_idx] - torch.arange(len(dataset_test)).to(case_index)).square().sum().item() == 0
            cur_min_, cur_ave_ = [], []
            for scenario in act_index_map:
                cur_index = torch.from_numpy(act_index_map[scenario]).long().to(system_id_all.device)
                cur_ave = cur_Error_ave[cur_index].mean().item() / config.data.test.scale
                cur_min = cur_Error_min[cur_index].mean().item() / config.data.test.scale
                results[f'{scenario}_min_{config.train.K}_' + key] = cur_min
                results[f'{scenario}_ave_{config.train.K}_' + key] = cur_ave
                cur_min_.append(cur_min)
                cur_ave_.append(cur_ave)
            results[f'AVE_min_{config.train.K}_' + key] = sum(cur_min_) / len(cur_min_)
            results[f'AVE_ave_{config.train.K}_' + key] = sum(cur_ave_) / len(cur_ave_)

        if rank == 0:

            print(results)

            name = f'AVE_ave_{config.train.K}_' + '80ms'
            test_log.log({'Test_' + name: results[name]}, commit=False)
            name = f'AVE_min_{config.train.K}_' + '80ms'
            test_log.log({'Test_' + name: results[name]}, commit=False)
            name = f'AVE_ave_{config.train.K}_' + '1000ms'
            test_log.log({'Test_' + name: results[name]}, commit=False)
            name = f'AVE_min_{config.train.K}_' + '1000ms'
            test_log.log({'Test_' + name: results[name]}, commit=True)

            # Save
            save_path = os.path.join(output_path, 'results.pkl')
            save_results = results
            with open(save_path, 'wb') as f:
                pickle.dump(save_results, f)
            print(f'Results saved to {save_path}')


    if world_size > 1:
        dist.barrier()
        dist.destroy_process_group()



def main():

    parser = argparse.ArgumentParser(description='GeoTDM')
    parser.add_argument('--train_yaml_file', type=str, help='path of the train yaml file',
                        default='configs/cmu_train.yaml')
    parser.add_argument('--local-rank', dest='local_rank', type=int, default=0)

    args = parser.parse_args()
    print(args)

    world_size = torch.cuda.device_count()
    print('Let\'s use', world_size, 'GPUs!')

    if world_size > 1:
        dist.init_process_group('nccl', rank=args.local_rank, world_size=world_size)
    run(args.local_rank, world_size, args)


if __name__ == '__main__':
    
    main()

