""" Author: Yifan Lu <yifan_lu@sjtu.edu.cn>

HEAL: An Extensible Framework for Open Heterogeneous Collaborative Perception 
"""

import torch
import torch.nn as nn
import numpy as np
from icecream import ic
from collections import OrderedDict, Counter
from opencood.models.sub_modules.base_bev_backbone_resnet import ResNetBEVBackbone 
from opencood.models.sub_modules.comm import Comm
from opencood.models.sub_modules.feature_alignnet import AlignNet
from opencood.models.sub_modules.downsample_conv import DownsampleConv
from opencood.models.sub_modules.naive_compress import NaiveCompressor
from opencood.models.fuse_modules.pyramid_fuse import PyramidFusion
from opencood.utils.transformation_utils import normalize_pairwise_tfm
from opencood.utils.model_utils import check_trainable_module, fix_bn, unfix_bn
import importlib
import torchvision

class HeterPyramidLSDCollab(nn.Module):
    def __init__(self, args):
        super(HeterPyramidLSDCollab, self).__init__()
        self.args = args
        modality_name_list = list(args.keys())
        modality_name_list = [x for x in modality_name_list if x.startswith("m") and x[1:].isdigit()] 
        self.modality_name_list = modality_name_list

        self.cav_range = args['lidar_range']
        self.sensor_type_dict = OrderedDict()

        self.cam_crop_info = {} 

        # setup each modality model
        for modality_name in self.modality_name_list:
            model_setting = args[modality_name]
            sensor_name = model_setting['sensor_type']
            self.sensor_type_dict[modality_name] = sensor_name

            # import model
            encoder_filename = "opencood.models.heter_encoders"
            encoder_lib = importlib.import_module(encoder_filename)
            encoder_class = None
            target_model_name = model_setting['core_method'].replace('_', '')

            for name, cls in encoder_lib.__dict__.items():
                if name.lower() == target_model_name.lower():
                    encoder_class = cls

            """
            Encoder building
            """
            setattr(self, f"encoder_{modality_name}", encoder_class(model_setting['encoder_args']))
            if model_setting['encoder_args'].get("depth_supervision", False):
                setattr(self, f"depth_supervision_{modality_name}", True)
            else:
                setattr(self, f"depth_supervision_{modality_name}", False)

            """
            Backbone building 
            """
            setattr(self, f"backbone_{modality_name}", ResNetBEVBackbone(model_setting['backbone_args']))

            """
            Aligner building
            """
            setattr(self, f"aligner_{modality_name}", AlignNet(model_setting['aligner_args']))
            if sensor_name == "camera":
                camera_mask_args = model_setting['camera_mask_args']
                setattr(self, f"crop_ratio_W_{modality_name}", (self.cav_range[3]) / (camera_mask_args['grid_conf']['xbound'][1]))
                setattr(self, f"crop_ratio_H_{modality_name}", (self.cav_range[4]) / (camera_mask_args['grid_conf']['ybound'][1]))
                setattr(self, f"xdist_{modality_name}", (camera_mask_args['grid_conf']['xbound'][1] - camera_mask_args['grid_conf']['xbound'][0]))
                setattr(self, f"ydist_{modality_name}", (camera_mask_args['grid_conf']['ybound'][1] - camera_mask_args['grid_conf']['ybound'][0]))
                self.cam_crop_info[modality_name] = {
                    f"crop_ratio_W_{modality_name}": eval(f"self.crop_ratio_W_{modality_name}"),
                    f"crop_ratio_H_{modality_name}": eval(f"self.crop_ratio_H_{modality_name}"),
                }

            """
            Communication module from codebook
            """
            model_setting['comm_args'].update({'local_range': self.cav_range})
            setattr(self, f"comm_{modality_name}", Comm(model_setting['comm_args']))
            
            
            """
            Fusion, by default multiscale fusion: 
            Note the input of PyramidFusion has downsampled 2x. (SECOND required)
            """
            setattr(self, f"pyramid_backbone_{modality_name}", PyramidFusion(model_setting['fusion_backbone']))


            """
            Shrink header
            """
            setattr(self, f"shrink_flag_{modality_name}", False)
            if 'shrink_header' in model_setting:
                setattr(self, f"shrink_flag_{modality_name}", True)
                setattr(self, f"shrink_conv_{modality_name}", DownsampleConv(model_setting['shrink_header']))

            """
            Shared Heads
            """
            setattr(self, f"cls_head_{modality_name}", nn.Conv2d(model_setting['in_head'], model_setting['anchor_number'],
                                    kernel_size=1))
            setattr(self, f"reg_head_{modality_name}", nn.Conv2d(model_setting['in_head'], 7 * model_setting['anchor_number'],
                                    kernel_size=1))            
            setattr(self, f"dir_head_{modality_name}", nn.Conv2d(model_setting['in_head'], \
                            model_setting['dir_args']['num_bins'] * model_setting['anchor_number'],
                                    kernel_size=1))
        
        # compressor will be only trainable
        self.compress = False
        # if 'compressor' in args:
        #     self.compress = True
        #     self.compressor = NaiveCompressor(args['compressor']['input_dim'],
        #                                       args['compressor']['compress_ratio'])

  
        """For feature transformation"""
        self.H = (self.cav_range[4] - self.cav_range[1])
        self.W = (self.cav_range[3] - self.cav_range[0])
        self.fake_voxel_size = 1
        for modality_name in self.modality_name_list:
            self.model_train_init(modality_name)
        # check again which module is not fixed.
        check_trainable_module(self)

    def model_train_init(self, modality_name):
        # for module in self.adjust_modules:
        # for name, p in self.named_parameters():
        #     if (not ('comm' in name)) and (not ('head' in name)):
        #         p.requires_grad = False
                
        # eval(f"self.pyramid_backbone_{modality_name}").train()
        # if  eval(f"self.shrink_flag_{modality_name}"):
        #     eval(f"self.shrink_conv_{modality_name}").train()
        # eval(f"self.cls_head_{modality_name}").train()
        # eval(f"self.reg_head_{modality_name}").train()
        # eval(f"self.dir_head_{modality_name}").train()
        
        
        for name, p in self.named_parameters():
            # 只训练通信模块和特征选择模块的参数
            if (not ('comm' in name)) and (not ('focuser' in name)):
                p.requires_grad = False
                
        eval(f"self.comm_{modality_name}").train()
    
    
    def assemble_heter_features(self, agent_modality_list, modality_feature_dict):
        
        counting_dict = {modality_name:0 for modality_name in self.modality_name_list}
        heter_feature_2d_list = []
        for modality_name in agent_modality_list:
            feat_idx = counting_dict[modality_name]
            heter_feature_2d_list.append(modality_feature_dict[modality_name][feat_idx])
            counting_dict[modality_name] += 1

        heter_feature_2d = torch.stack(heter_feature_2d_list)
        
        return heter_feature_2d
            
    def forward(self, data_dict):
        output_dict = {'pyramid': 'collab'}
        agent_modality_list = data_dict['agent_modality_list'] 
        affine_matrix = normalize_pairwise_tfm(data_dict['pairwise_t_matrix'], self.H, self.W, self.fake_voxel_size)
        record_len = data_dict['record_len'] # 场景内智能车数量
        # print(record_len)
        # print(agent_modality_list)
        modality_count_dict = Counter(agent_modality_list)
        modality_feature_dict = {}

        # 按modality分类推理
        for modality_name in self.modality_name_list:
            if modality_name not in modality_count_dict:
                continue
            feature = eval(f"self.encoder_{modality_name}")(data_dict, modality_name)
            feature = eval(f"self.backbone_{modality_name}")({"spatial_features": feature})['spatial_features_2d']
            feature = eval(f"self.aligner_{modality_name}")(feature)
            modality_feature_dict[modality_name] = feature

        """
        Crop/Padd camera feature map.
        """
        for modality_name in self.modality_name_list:
            if modality_name in modality_count_dict:
                if self.sensor_type_dict[modality_name] == "camera":
                    # should be padding. Instead of masking
                    feature = modality_feature_dict[modality_name]
                    _, _, H, W = feature.shape
                    target_H = int(H*eval(f"self.crop_ratio_H_{modality_name}"))
                    target_W = int(W*eval(f"self.crop_ratio_W_{modality_name}"))

                    # 对齐特征尺寸, 使特征表示的范围相同
                    crop_func = torchvision.transforms.CenterCrop((target_H, target_W))
                    modality_feature_dict[modality_name] = crop_func(feature)
                    if eval(f"self.depth_supervision_{modality_name}"):
                        output_dict.update({
                            f"depth_items_{modality_name}": eval(f"self.encoder_{modality_name}").depth_items
                        })

        """
        Assemble heter features
        """
        # if self.training:
        heter_feature_2d = self.assemble_heter_features(agent_modality_list, modality_feature_dict)
        output_dict.update({'fc_before_send':heter_feature_2d})
        
        
        """
        Communicate by codebook
        """
        
        # 使用不同的智能体对应的发送器发送特征
        modality_weights_dict = {}
        modality_fc_before_rbdc_dict = {}
        for modality_name in self.modality_name_list:
            if modality_name in modality_count_dict:    
                # Add sematic decompsition process here.
                weights, fc_before_rbdc = \
                    eval(f"self.comm_{modality_name}.sender")(modality_feature_dict[modality_name])
                modality_weights_dict[modality_name] = weights
                modality_fc_before_rbdc_dict[modality_name] = fc_before_rbdc
        
        
        # 信息重组为batch
        recevied_weights = self.assemble_heter_features(agent_modality_list, modality_weights_dict)
        fc_before_rbdc = self.assemble_heter_features(agent_modality_list, modality_fc_before_rbdc_dict)
        
        # 只在ego(0号智能体)执行接收和检测
        modality_name = agent_modality_list[0]
        heter_feature_2d, fc_after_rbdc = eval(f"self.comm_{modality_name}.receiver")\
                                            (recevied_weights, record_len, affine_matrix)
        
        
        """
        Save codebook, 
             feature after/before send by decomposed codebooks
             feature after/before reconstruct by decomposed codebooks \
             for Loss computation
        """
        output_dict.update({
                            # 'sg_map': sg_map,
                            'fc_after_send': heter_feature_2d,
                            'fc_before_rbdc': fc_before_rbdc,
                            'fc_after_rbdc': fc_after_rbdc,
                            'codebook': eval(f"self.comm_{modality_name}.codebook")
                            })   
        # if self.compress:
        #     heter_feature_2d = self.compressor(heter_feature_2d)

        # heter_feature_2d is downsampled 2x
        # add croping information to collaboration module
        
        # 推理状态下使用完整的ego信息
        # if not self.training:
            # heter_feature_2d[0] = ego_feature_2d
            
        # 推理时, 已经receiver处将neb特征映射到了ego视角下
        # warp = True
        # if not self.training: 
            # warp = False
        fused_feature, occ_outputs = eval(f"self.pyramid_backbone_{modality_name}.forward_collab")(
                                                heter_feature_2d,
                                                record_len, 
                                                affine_matrix, 
                                                agent_modality_list, 
                                                self.cam_crop_info,
                                                # warp = warp
                                            )

        if eval(f"self.shrink_flag_{modality_name}"):
            fused_feature = eval(f"self.shrink_conv_{modality_name}")(fused_feature)

        cls_preds = eval(f"self.cls_head_{modality_name}")(fused_feature)
        reg_preds = eval(f"self.reg_head_{modality_name}")(fused_feature)
        dir_preds = eval(f"self.dir_head_{modality_name}")(fused_feature)

        output_dict.update({'cls_preds': cls_preds,
                            'reg_preds': reg_preds,
                            'dir_preds': dir_preds})
        
        output_dict.update({'occ_single_list': 
                            occ_outputs})

        return output_dict
