from __future__ import annotations

from datasets import load_dataset

from mteb.abstasks.AbsTaskRetrieval import AbsTaskRetrieval
from mteb.abstasks.MultilingualTask import MultilingualTask
from mteb.abstasks.TaskMetadata import TaskMetadata

_EVAL_SPLIT = "test"

_LANGUAGES = [
    "acm_Arab",
    "afr_Latn",
    "als_Latn",
    "amh_Ethi",
    "apc_Arab",
    "arb_Arab",
    "arb_Latn",
    "ars_Arab",
    "ary_Arab",
    "arz_Arab",
    "asm_Beng",
    "azj_Latn",
    "bam_Latn",
    "ben_Beng",
    "ben_Latn",
    "bod_Tibt",
    "bul_Cyrl",
    "cat_Latn",
    "ceb_Latn",
    "ces_Latn",
    "ckb_Arab",
    "dan_Latn",
    "deu_Latn",
    "ell_Grek",
    "eng_Latn",
    "est_Latn",
    "eus_Latn",
    "fin_Latn",
    "fra_Latn",
    "fuv_Latn",
    "gaz_Latn",
    "grn_Latn",
    "guj_Gujr",
    "hat_Latn",
    "hau_Latn",
    "heb_Hebr",
    "hin_Deva",
    "hin_Latn",
    "hrv_Latn",
    "hun_Latn",
    "hye_Armn",
    "ibo_Latn",
    "ilo_Latn",
    "ind_Latn",
    "isl_Latn",
    "ita_Latn",
    "jav_Latn",
    "jpn_Jpan",
    "kac_Latn",
    "kan_Knda",
    "kat_Geor",
    "kaz_Cyrl",
    "kea_Latn",
    "khk_Cyrl",
    "khm_Khmr",
    "kin_Latn",
    "kir_Cyrl",
    "kor_Hang",
    "lao_Laoo",
    "lin_Latn",
    "lit_Latn",
    "lug_Latn",
    "luo_Latn",
    "lvs_Latn",
    "mal_Mlym",
    "mar_Deva",
    "mkd_Cyrl",
    "mlt_Latn",
    "mri_Latn",
    "mya_Mymr",
    "nld_Latn",
    "nob_Latn",
    "npi_Deva",
    "npi_Latn",
    "nso_Latn",
    "nya_Latn",
    "ory_Orya",
    "pan_Guru",
    "pbt_Arab",
    "pes_Arab",
    "plt_Latn",
    "pol_Latn",
    "por_Latn",
    "ron_Latn",
    "rus_Cyrl",
    "shn_Mymr",
    "sin_Latn",
    "sin_Sinh",
    "slk_Latn",
    "slv_Latn",
    "sna_Latn",
    "snd_Arab",
    "som_Latn",
    "sot_Latn",
    "spa_Latn",
    "srp_Cyrl",
    "ssw_Latn",
    "sun_Latn",
    "swe_Latn",
    "swh_Latn",
    "tam_Taml",
    "tel_Telu",
    "tgk_Cyrl",
    "tgl_Latn",
    "tha_Thai",
    "tir_Ethi",
    "tsn_Latn",
    "tso_Latn",
    "tur_Latn",
    "ukr_Cyrl",
    "urd_Arab",
    "urd_Latn",
    "uzn_Latn",
    "vie_Latn",
    "war_Latn",
    "wol_Latn",
    "xho_Latn",
    "yor_Latn",
    "zho_Hans",
    "zho_Hant",
    "zsm_Latn",
    "zul_Latn",
]


def get_lang_pairs() -> dict[str, list[str]]:
    # add pairs with same langauge as the source and target
    # add pairs with english as source or target
    lang_pairs = {}
    for x in _LANGUAGES:
        pair = f"{x}-{x}"
        lang_pairs[pair] = [x.replace("_", "-"), x.replace("_", "-")]

        if x != "eng_Latn":
            pair = f"{x}-eng_Latn"
            lang_pairs[pair] = [x.replace("_", "-"), "eng-Latn"]
            pair = f"eng_Latn-{x}"
            lang_pairs[pair] = ["eng-Latn", x.replace("_", "-")]

    # add pairs for languages with a base script and a Latn script
    lang_base_scripts = [
        "arb_Arab",
        "ben_Beng",
        "hin_Deva",
        "npi_Deva",
        "sin_Sinh",
        "urd_Arab",
    ]
    for lang_base_script in lang_base_scripts:
        lang = lang_base_script.split("_")[0]
        lang_latn_script = f"{lang}_Latn"
        pair = f"{lang_base_script}-{lang_latn_script}"
        lang_pairs[pair] = [
            lang_base_script.replace("_", "-"),
            lang_latn_script.replace("_", "-"),
        ]
        pair = f"{lang_latn_script}-{lang_base_script}"
        lang_pairs[pair] = [
            lang_latn_script.replace("_", "-"),
            lang_base_script.replace("_", "-"),
        ]

    return lang_pairs


_LANGUAGES_MAPPING = get_lang_pairs()


class BelebeleRetrieval(MultilingualTask, AbsTaskRetrieval):
    metadata = TaskMetadata(
        name="BelebeleRetrieval",
        dataset={
            "path": "facebook/belebele",
            "revision": "75b399394a9803252cfec289d103de462763db7c",
        },
        description=(
            "Belebele is a multiple-choice machine reading comprehension (MRC) dataset spanning 122 language variants "
            "(including 115 distinct languages and their scripts)"
        ),
        type="Retrieval",
        category="s2p",
        modalities=["text"],
        eval_splits=[_EVAL_SPLIT],
        eval_langs=_LANGUAGES_MAPPING,
        reference="https://arxiv.org/abs/2308.16884",
        main_score="ndcg_at_10",
        license="CC-BY-SA-4.0",
        domains=["Web", "News", "Written"],
        sample_creation="created",  # number of languages * 900
        date=("2023-08-31", "2023-08-31"),
        task_subtypes=["Question answering"],
        annotations_creators="expert-annotated",
        dialect=[],
        descriptive_stats={
            "n_samples": {_EVAL_SPLIT: 103500},
            "test": {
                "average_document_length": 487.3975028339728,
                "average_query_length": 74.49551684802204,
                "num_documents": 183488,
                "num_queries": 338378,
                "average_relevant_docs_per_query": 1.0,
                "hf_subset_descriptive_stats": {
                    "acm_Arab-acm_Arab": {
                        "average_document_length": 416.4733606557377,
                        "average_query_length": 55.84,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "acm_Arab-eng_Latn": {
                        "average_document_length": 416.4733606557377,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-acm_Arab": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 55.84,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "afr_Latn-afr_Latn": {
                        "average_document_length": 503.6659836065574,
                        "average_query_length": 78.04555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "afr_Latn-eng_Latn": {
                        "average_document_length": 503.6659836065574,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-afr_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 78.04555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "als_Latn-als_Latn": {
                        "average_document_length": 534.016393442623,
                        "average_query_length": 76.13555555555556,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "als_Latn-eng_Latn": {
                        "average_document_length": 534.016393442623,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-als_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 76.13555555555556,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "amh_Ethi-amh_Ethi": {
                        "average_document_length": 319.8688524590164,
                        "average_query_length": 49.16111111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "amh_Ethi-eng_Latn": {
                        "average_document_length": 319.8688524590164,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-amh_Ethi": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 49.16111111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "apc_Arab-apc_Arab": {
                        "average_document_length": 393.0553278688525,
                        "average_query_length": 55.85777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "apc_Arab-eng_Latn": {
                        "average_document_length": 393.0553278688525,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-apc_Arab": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 55.85777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "arb_Arab-arb_Arab": {
                        "average_document_length": 421.96311475409834,
                        "average_query_length": 58.55,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "arb_Arab-eng_Latn": {
                        "average_document_length": 421.96311475409834,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-arb_Arab": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 58.55,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "arb_Latn-arb_Latn": {
                        "average_document_length": 555.6188524590164,
                        "average_query_length": 67.02444444444444,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "arb_Latn-eng_Latn": {
                        "average_document_length": 555.6188524590164,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-arb_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 67.02444444444444,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ars_Arab-ars_Arab": {
                        "average_document_length": 422.5553278688525,
                        "average_query_length": 56.43222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ars_Arab-eng_Latn": {
                        "average_document_length": 422.5553278688525,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ars_Arab": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 56.43222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ary_Arab-ary_Arab": {
                        "average_document_length": 411.1475409836066,
                        "average_query_length": 66.01893095768374,
                        "num_documents": 488,
                        "num_queries": 898,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ary_Arab-eng_Latn": {
                        "average_document_length": 411.1475409836066,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ary_Arab": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 66.01893095768374,
                        "num_documents": 488,
                        "num_queries": 898,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "arz_Arab-arz_Arab": {
                        "average_document_length": 412.05122950819674,
                        "average_query_length": 57.14111111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "arz_Arab-eng_Latn": {
                        "average_document_length": 412.05122950819674,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-arz_Arab": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 57.14111111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "asm_Beng-asm_Beng": {
                        "average_document_length": 458.5983606557377,
                        "average_query_length": 68.26,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "asm_Beng-eng_Latn": {
                        "average_document_length": 458.5983606557377,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-asm_Beng": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 68.26,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "azj_Latn-azj_Latn": {
                        "average_document_length": 519.6127049180328,
                        "average_query_length": 73.51222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "azj_Latn-eng_Latn": {
                        "average_document_length": 519.6127049180328,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-azj_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 73.51222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "bam_Latn-bam_Latn": {
                        "average_document_length": 457.3114754098361,
                        "average_query_length": 72.34222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "bam_Latn-eng_Latn": {
                        "average_document_length": 457.3114754098361,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-bam_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 72.34222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ben_Beng-ben_Beng": {
                        "average_document_length": 467.7745901639344,
                        "average_query_length": 69.48444444444445,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ben_Beng-eng_Latn": {
                        "average_document_length": 467.7745901639344,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ben_Beng": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 69.48444444444445,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ben_Latn-ben_Latn": {
                        "average_document_length": 522.8934426229508,
                        "average_query_length": 74.78777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ben_Latn-eng_Latn": {
                        "average_document_length": 522.8934426229508,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ben_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 74.78777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "bod_Tibt-bod_Tibt": {
                        "average_document_length": 533.3872950819672,
                        "average_query_length": 86.90222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "bod_Tibt-eng_Latn": {
                        "average_document_length": 533.3872950819672,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-bod_Tibt": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 86.90222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "bul_Cyrl-bul_Cyrl": {
                        "average_document_length": 496.97131147540983,
                        "average_query_length": 72.89,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "bul_Cyrl-eng_Latn": {
                        "average_document_length": 496.97131147540983,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-bul_Cyrl": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 72.89,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "cat_Latn-cat_Latn": {
                        "average_document_length": 525.4467213114754,
                        "average_query_length": 75.40666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "cat_Latn-eng_Latn": {
                        "average_document_length": 525.4467213114754,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-cat_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 75.40666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ceb_Latn-ceb_Latn": {
                        "average_document_length": 570.8483606557377,
                        "average_query_length": 81.19666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ceb_Latn-eng_Latn": {
                        "average_document_length": 570.8483606557377,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ceb_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 81.19666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ces_Latn-ces_Latn": {
                        "average_document_length": 461.0061475409836,
                        "average_query_length": 67.73333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ces_Latn-eng_Latn": {
                        "average_document_length": 461.0061475409836,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ces_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 67.73333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ckb_Arab-ckb_Arab": {
                        "average_document_length": 462.98770491803276,
                        "average_query_length": 71.04555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ckb_Arab-eng_Latn": {
                        "average_document_length": 462.98770491803276,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ckb_Arab": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 71.04555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "dan_Latn-dan_Latn": {
                        "average_document_length": 489.4856557377049,
                        "average_query_length": 72.96888888888888,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "dan_Latn-eng_Latn": {
                        "average_document_length": 489.4856557377049,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-dan_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 72.96888888888888,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "deu_Latn-deu_Latn": {
                        "average_document_length": 555.1659836065573,
                        "average_query_length": 75.32444444444444,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "deu_Latn-eng_Latn": {
                        "average_document_length": 555.1659836065573,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-deu_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 75.32444444444444,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ell_Grek-ell_Grek": {
                        "average_document_length": 568.3872950819672,
                        "average_query_length": 86.92666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ell_Grek-eng_Latn": {
                        "average_document_length": 568.3872950819672,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ell_Grek": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 86.92666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-eng_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "est_Latn-est_Latn": {
                        "average_document_length": 467.1475409836066,
                        "average_query_length": 67.55888888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "est_Latn-eng_Latn": {
                        "average_document_length": 467.1475409836066,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-est_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 67.55888888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eus_Latn-eus_Latn": {
                        "average_document_length": 506.19262295081967,
                        "average_query_length": 74.44777777777777,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eus_Latn-eng_Latn": {
                        "average_document_length": 506.19262295081967,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-eus_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 74.44777777777777,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "fin_Latn-fin_Latn": {
                        "average_document_length": 507.5,
                        "average_query_length": 72.50888888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "fin_Latn-eng_Latn": {
                        "average_document_length": 507.5,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-fin_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 72.50888888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "fra_Latn-fra_Latn": {
                        "average_document_length": 564.8401639344262,
                        "average_query_length": 90.54222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "fra_Latn-eng_Latn": {
                        "average_document_length": 564.8401639344262,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-fra_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 90.54222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "fuv_Latn-fuv_Latn": {
                        "average_document_length": 443.4733606557377,
                        "average_query_length": 58.42111111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "fuv_Latn-eng_Latn": {
                        "average_document_length": 443.4733606557377,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-fuv_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 58.42111111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "gaz_Latn-gaz_Latn": {
                        "average_document_length": 563.5389344262295,
                        "average_query_length": 85.93222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "gaz_Latn-eng_Latn": {
                        "average_document_length": 563.5389344262295,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-gaz_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 85.93222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "grn_Latn-grn_Latn": {
                        "average_document_length": 480.3299180327869,
                        "average_query_length": 75.10666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "grn_Latn-eng_Latn": {
                        "average_document_length": 480.3299180327869,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-grn_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 75.10666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "guj_Gujr-guj_Gujr": {
                        "average_document_length": 458.1885245901639,
                        "average_query_length": 62.25666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "guj_Gujr-eng_Latn": {
                        "average_document_length": 458.1885245901639,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-guj_Gujr": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 62.25666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hat_Latn-hat_Latn": {
                        "average_document_length": 438.6700819672131,
                        "average_query_length": 70.64666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hat_Latn-eng_Latn": {
                        "average_document_length": 438.6700819672131,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-hat_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 70.64666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hau_Latn-hau_Latn": {
                        "average_document_length": 507.24590163934425,
                        "average_query_length": 85.8488888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hau_Latn-eng_Latn": {
                        "average_document_length": 507.24590163934425,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-hau_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 85.8488888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "heb_Hebr-heb_Hebr": {
                        "average_document_length": 371.36270491803276,
                        "average_query_length": 55.135555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "heb_Hebr-eng_Latn": {
                        "average_document_length": 371.36270491803276,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-heb_Hebr": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 55.135555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hin_Deva-hin_Deva": {
                        "average_document_length": 473.55737704918033,
                        "average_query_length": 72.61777777777777,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hin_Deva-eng_Latn": {
                        "average_document_length": 473.55737704918033,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-hin_Deva": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 72.61777777777777,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hin_Latn-hin_Latn": {
                        "average_document_length": 541.7315573770492,
                        "average_query_length": 74.81222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hin_Latn-eng_Latn": {
                        "average_document_length": 541.7315573770492,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-hin_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 74.81222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hrv_Latn-hrv_Latn": {
                        "average_document_length": 469.202868852459,
                        "average_query_length": 68.83555555555556,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hrv_Latn-eng_Latn": {
                        "average_document_length": 469.202868852459,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-hrv_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 68.83555555555556,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hun_Latn-hun_Latn": {
                        "average_document_length": 501.1946721311475,
                        "average_query_length": 74.40555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hun_Latn-eng_Latn": {
                        "average_document_length": 501.1946721311475,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-hun_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 74.40555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hye_Armn-hye_Armn": {
                        "average_document_length": 527.5102459016393,
                        "average_query_length": 75.42555555555556,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hye_Armn-eng_Latn": {
                        "average_document_length": 527.5102459016393,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-hye_Armn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 75.42555555555556,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ibo_Latn-ibo_Latn": {
                        "average_document_length": 482.3483606557377,
                        "average_query_length": 72.51501668520578,
                        "num_documents": 488,
                        "num_queries": 899,
                        "average_relevant_docs_per_query": 1.0011123470522802,
                    },
                    "ibo_Latn-eng_Latn": {
                        "average_document_length": 482.3483606557377,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ibo_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 72.51501668520578,
                        "num_documents": 488,
                        "num_queries": 899,
                        "average_relevant_docs_per_query": 1.0011123470522802,
                    },
                    "ilo_Latn-ilo_Latn": {
                        "average_document_length": 574.6987704918033,
                        "average_query_length": 85.7611111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ilo_Latn-eng_Latn": {
                        "average_document_length": 574.6987704918033,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ilo_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 85.7611111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ind_Latn-ind_Latn": {
                        "average_document_length": 516.0573770491803,
                        "average_query_length": 82.10555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ind_Latn-eng_Latn": {
                        "average_document_length": 516.0573770491803,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ind_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 82.10555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "isl_Latn-isl_Latn": {
                        "average_document_length": 470.73975409836066,
                        "average_query_length": 77.27333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "isl_Latn-eng_Latn": {
                        "average_document_length": 470.73975409836066,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-isl_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 77.27333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ita_Latn-ita_Latn": {
                        "average_document_length": 560.9344262295082,
                        "average_query_length": 83.49777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ita_Latn-eng_Latn": {
                        "average_document_length": 560.9344262295082,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ita_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 83.49777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "jav_Latn-jav_Latn": {
                        "average_document_length": 494.1803278688525,
                        "average_query_length": 78.60666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "jav_Latn-eng_Latn": {
                        "average_document_length": 494.1803278688525,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-jav_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 78.60666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "jpn_Jpan-jpn_Jpan": {
                        "average_document_length": 207.74795081967213,
                        "average_query_length": 35.79,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "jpn_Jpan-eng_Latn": {
                        "average_document_length": 207.74795081967213,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-jpn_Jpan": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 35.79,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "kac_Latn-kac_Latn": {
                        "average_document_length": 605.2889344262295,
                        "average_query_length": 98.64182424916574,
                        "num_documents": 488,
                        "num_queries": 899,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "kac_Latn-eng_Latn": {
                        "average_document_length": 605.2889344262295,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-kac_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 98.64182424916574,
                        "num_documents": 488,
                        "num_queries": 899,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "kan_Knda-kan_Knda": {
                        "average_document_length": 498.9077868852459,
                        "average_query_length": 72.13666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "kan_Knda-eng_Latn": {
                        "average_document_length": 498.9077868852459,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-kan_Knda": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 72.13666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "kat_Geor-kat_Geor": {
                        "average_document_length": 521.7766393442623,
                        "average_query_length": 74.81444444444445,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "kat_Geor-eng_Latn": {
                        "average_document_length": 521.7766393442623,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-kat_Geor": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 74.81444444444445,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "kaz_Cyrl-kaz_Cyrl": {
                        "average_document_length": 488.2110655737705,
                        "average_query_length": 70.75666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "kaz_Cyrl-eng_Latn": {
                        "average_document_length": 488.2110655737705,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-kaz_Cyrl": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 70.75666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "kea_Latn-kea_Latn": {
                        "average_document_length": 471.5594262295082,
                        "average_query_length": 75.94111111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "kea_Latn-eng_Latn": {
                        "average_document_length": 471.5594262295082,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-kea_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 75.94111111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "khk_Cyrl-khk_Cyrl": {
                        "average_document_length": 496.655737704918,
                        "average_query_length": 73.33444444444444,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "khk_Cyrl-eng_Latn": {
                        "average_document_length": 496.655737704918,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-khk_Cyrl": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 73.33444444444444,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "khm_Khmr-khm_Khmr": {
                        "average_document_length": 562.4139344262295,
                        "average_query_length": 75.74888888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "khm_Khmr-eng_Latn": {
                        "average_document_length": 562.4139344262295,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-khm_Khmr": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 75.74888888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "kin_Latn-kin_Latn": {
                        "average_document_length": 529.2520491803278,
                        "average_query_length": 79.89655172413794,
                        "num_documents": 488,
                        "num_queries": 899,
                        "average_relevant_docs_per_query": 1.0011123470522802,
                    },
                    "kin_Latn-eng_Latn": {
                        "average_document_length": 529.2520491803278,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-kin_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 79.89655172413794,
                        "num_documents": 488,
                        "num_queries": 899,
                        "average_relevant_docs_per_query": 1.0011123470522802,
                    },
                    "kir_Cyrl-kir_Cyrl": {
                        "average_document_length": 487.80737704918033,
                        "average_query_length": 74.42333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "kir_Cyrl-eng_Latn": {
                        "average_document_length": 487.80737704918033,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-kir_Cyrl": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 74.42333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "kor_Hang-kor_Hang": {
                        "average_document_length": 241.32991803278688,
                        "average_query_length": 35.257777777777775,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "kor_Hang-eng_Latn": {
                        "average_document_length": 241.32991803278688,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-kor_Hang": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 35.257777777777775,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "lao_Laoo-lao_Laoo": {
                        "average_document_length": 471.6495901639344,
                        "average_query_length": 63.31333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "lao_Laoo-eng_Latn": {
                        "average_document_length": 471.6495901639344,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-lao_Laoo": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 63.31333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "lin_Latn-lin_Latn": {
                        "average_document_length": 512.9016393442623,
                        "average_query_length": 81.56681514476615,
                        "num_documents": 488,
                        "num_queries": 898,
                        "average_relevant_docs_per_query": 1.0022271714922049,
                    },
                    "lin_Latn-eng_Latn": {
                        "average_document_length": 512.9016393442623,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-lin_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 81.56681514476615,
                        "num_documents": 488,
                        "num_queries": 898,
                        "average_relevant_docs_per_query": 1.0022271714922049,
                    },
                    "lit_Latn-lit_Latn": {
                        "average_document_length": 474.0553278688525,
                        "average_query_length": 68.69888888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "lit_Latn-eng_Latn": {
                        "average_document_length": 474.0553278688525,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-lit_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 68.69888888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "lug_Latn-lug_Latn": {
                        "average_document_length": 485.73975409836066,
                        "average_query_length": 78.52057842046719,
                        "num_documents": 488,
                        "num_queries": 899,
                        "average_relevant_docs_per_query": 1.0011123470522802,
                    },
                    "lug_Latn-eng_Latn": {
                        "average_document_length": 485.73975409836066,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-lug_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 78.52057842046719,
                        "num_documents": 488,
                        "num_queries": 899,
                        "average_relevant_docs_per_query": 1.0011123470522802,
                    },
                    "luo_Latn-luo_Latn": {
                        "average_document_length": 497.53688524590166,
                        "average_query_length": 73.14333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "luo_Latn-eng_Latn": {
                        "average_document_length": 497.53688524590166,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-luo_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 73.14333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "lvs_Latn-lvs_Latn": {
                        "average_document_length": 487.21311475409834,
                        "average_query_length": 69.97888888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "lvs_Latn-eng_Latn": {
                        "average_document_length": 487.21311475409834,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-lvs_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 69.97888888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "mal_Mlym-mal_Mlym": {
                        "average_document_length": 539.2827868852459,
                        "average_query_length": 80.69222222222223,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "mal_Mlym-eng_Latn": {
                        "average_document_length": 539.2827868852459,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-mal_Mlym": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 80.69222222222223,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "mar_Deva-mar_Deva": {
                        "average_document_length": 478.67418032786884,
                        "average_query_length": 68.62625139043382,
                        "num_documents": 488,
                        "num_queries": 899,
                        "average_relevant_docs_per_query": 1.0011123470522802,
                    },
                    "mar_Deva-eng_Latn": {
                        "average_document_length": 478.67418032786884,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-mar_Deva": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 68.62625139043382,
                        "num_documents": 488,
                        "num_queries": 899,
                        "average_relevant_docs_per_query": 1.0011123470522802,
                    },
                    "mkd_Cyrl-mkd_Cyrl": {
                        "average_document_length": 495.77868852459017,
                        "average_query_length": 74.01333333333334,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "mkd_Cyrl-eng_Latn": {
                        "average_document_length": 495.77868852459017,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-mkd_Cyrl": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 74.01333333333334,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "mlt_Latn-mlt_Latn": {
                        "average_document_length": 525.8995901639345,
                        "average_query_length": 75.00444444444445,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "mlt_Latn-eng_Latn": {
                        "average_document_length": 525.8995901639345,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-mlt_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 75.00444444444445,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "mri_Latn-mri_Latn": {
                        "average_document_length": 526.0860655737705,
                        "average_query_length": 81.71444444444444,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "mri_Latn-eng_Latn": {
                        "average_document_length": 526.0860655737705,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-mri_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 81.71444444444444,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "mya_Mymr-mya_Mymr": {
                        "average_document_length": 590.389344262295,
                        "average_query_length": 89.28333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "mya_Mymr-eng_Latn": {
                        "average_document_length": 590.389344262295,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-mya_Mymr": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 89.28333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "nld_Latn-nld_Latn": {
                        "average_document_length": 529.1434426229508,
                        "average_query_length": 75.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "nld_Latn-eng_Latn": {
                        "average_document_length": 529.1434426229508,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-nld_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 75.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "nob_Latn-nob_Latn": {
                        "average_document_length": 479.13729508196724,
                        "average_query_length": 71.04555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "nob_Latn-eng_Latn": {
                        "average_document_length": 479.13729508196724,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-nob_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 71.04555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "npi_Deva-npi_Deva": {
                        "average_document_length": 456.9590163934426,
                        "average_query_length": 66.89666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "npi_Deva-eng_Latn": {
                        "average_document_length": 456.9590163934426,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-npi_Deva": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 66.89666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "npi_Latn-npi_Latn": {
                        "average_document_length": 515.9815573770492,
                        "average_query_length": 71.89666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "npi_Latn-eng_Latn": {
                        "average_document_length": 515.9815573770492,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-npi_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 71.89666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "nso_Latn-nso_Latn": {
                        "average_document_length": 548.0225409836065,
                        "average_query_length": 86.77444444444444,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "nso_Latn-eng_Latn": {
                        "average_document_length": 548.0225409836065,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-nso_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 86.77444444444444,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "nya_Latn-nya_Latn": {
                        "average_document_length": 532.3934426229508,
                        "average_query_length": 90.78777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "nya_Latn-eng_Latn": {
                        "average_document_length": 532.3934426229508,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-nya_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 90.78777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ory_Orya-ory_Orya": {
                        "average_document_length": 487.78483606557376,
                        "average_query_length": 72.95777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ory_Orya-eng_Latn": {
                        "average_document_length": 487.78483606557376,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ory_Orya": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 72.95777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "pan_Guru-pan_Guru": {
                        "average_document_length": 480.2438524590164,
                        "average_query_length": 73.29777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "pan_Guru-eng_Latn": {
                        "average_document_length": 480.2438524590164,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-pan_Guru": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 73.29777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "pbt_Arab-pbt_Arab": {
                        "average_document_length": 453.3299180327869,
                        "average_query_length": 67.67111111111112,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "pbt_Arab-eng_Latn": {
                        "average_document_length": 453.3299180327869,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-pbt_Arab": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 67.67111111111112,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "pes_Arab-pes_Arab": {
                        "average_document_length": 448.84631147540983,
                        "average_query_length": 64.75111111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "pes_Arab-eng_Latn": {
                        "average_document_length": 448.84631147540983,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-pes_Arab": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 64.75111111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "plt_Latn-plt_Latn": {
                        "average_document_length": 581.2745901639345,
                        "average_query_length": 94.99555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "plt_Latn-eng_Latn": {
                        "average_document_length": 581.2745901639345,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-plt_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 94.99555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "pol_Latn-pol_Latn": {
                        "average_document_length": 504.0409836065574,
                        "average_query_length": 74.09777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "pol_Latn-eng_Latn": {
                        "average_document_length": 504.0409836065574,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-pol_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 74.09777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "por_Latn-por_Latn": {
                        "average_document_length": 517.2827868852459,
                        "average_query_length": 78.11666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "por_Latn-eng_Latn": {
                        "average_document_length": 517.2827868852459,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-por_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 78.11666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ron_Latn-ron_Latn": {
                        "average_document_length": 534.8668032786885,
                        "average_query_length": 78.74222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ron_Latn-eng_Latn": {
                        "average_document_length": 534.8668032786885,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ron_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 78.74222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "rus_Cyrl-rus_Cyrl": {
                        "average_document_length": 520.1700819672132,
                        "average_query_length": 83.16333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "rus_Cyrl-eng_Latn": {
                        "average_document_length": 520.1700819672132,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-rus_Cyrl": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 83.16333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "shn_Mymr-shn_Mymr": {
                        "average_document_length": 676.172131147541,
                        "average_query_length": 75.90222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "shn_Mymr-eng_Latn": {
                        "average_document_length": 676.172131147541,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-shn_Mymr": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 75.90222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "sin_Latn-sin_Latn": {
                        "average_document_length": 590.7889344262295,
                        "average_query_length": 94.46666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "sin_Latn-eng_Latn": {
                        "average_document_length": 590.7889344262295,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-sin_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 94.46666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "sin_Sinh-sin_Sinh": {
                        "average_document_length": 478.66803278688525,
                        "average_query_length": 69.91777777777777,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "sin_Sinh-eng_Latn": {
                        "average_document_length": 478.66803278688525,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-sin_Sinh": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 69.91777777777777,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "slk_Latn-slk_Latn": {
                        "average_document_length": 476.7766393442623,
                        "average_query_length": 68.5411111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "slk_Latn-eng_Latn": {
                        "average_document_length": 476.7766393442623,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-slk_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 68.5411111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "slv_Latn-slv_Latn": {
                        "average_document_length": 474.84631147540983,
                        "average_query_length": 68.79888888888888,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "slv_Latn-eng_Latn": {
                        "average_document_length": 474.84631147540983,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-slv_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 68.79888888888888,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "sna_Latn-sna_Latn": {
                        "average_document_length": 532.5860655737705,
                        "average_query_length": 81.30700778642937,
                        "num_documents": 488,
                        "num_queries": 899,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "sna_Latn-eng_Latn": {
                        "average_document_length": 532.5860655737705,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-sna_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 81.30700778642937,
                        "num_documents": 488,
                        "num_queries": 899,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "snd_Arab-snd_Arab": {
                        "average_document_length": 431.48770491803276,
                        "average_query_length": 63.42333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "snd_Arab-eng_Latn": {
                        "average_document_length": 431.48770491803276,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-snd_Arab": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 63.42333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "som_Latn-som_Latn": {
                        "average_document_length": 542.0737704918033,
                        "average_query_length": 90.95777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "som_Latn-eng_Latn": {
                        "average_document_length": 542.0737704918033,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-som_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 90.95777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "sot_Latn-sot_Latn": {
                        "average_document_length": 573.3258196721312,
                        "average_query_length": 83.13111111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "sot_Latn-eng_Latn": {
                        "average_document_length": 573.3258196721312,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-sot_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 83.13111111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "spa_Latn-spa_Latn": {
                        "average_document_length": 564.3319672131148,
                        "average_query_length": 82.16,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "spa_Latn-eng_Latn": {
                        "average_document_length": 564.3319672131148,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-spa_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 82.16,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "srp_Cyrl-srp_Cyrl": {
                        "average_document_length": 471.84631147540983,
                        "average_query_length": 67.49833147942158,
                        "num_documents": 488,
                        "num_queries": 899,
                        "average_relevant_docs_per_query": 1.0011123470522802,
                    },
                    "srp_Cyrl-eng_Latn": {
                        "average_document_length": 471.84631147540983,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-srp_Cyrl": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 67.49833147942158,
                        "num_documents": 488,
                        "num_queries": 899,
                        "average_relevant_docs_per_query": 1.0011123470522802,
                    },
                    "ssw_Latn-ssw_Latn": {
                        "average_document_length": 535.0901639344262,
                        "average_query_length": 81.09777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ssw_Latn-eng_Latn": {
                        "average_document_length": 535.0901639344262,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ssw_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 81.09777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "sun_Latn-sun_Latn": {
                        "average_document_length": 495.3032786885246,
                        "average_query_length": 78.16,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "sun_Latn-eng_Latn": {
                        "average_document_length": 495.3032786885246,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-sun_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 78.16,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "swe_Latn-swe_Latn": {
                        "average_document_length": 480.6803278688525,
                        "average_query_length": 68.67666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "swe_Latn-eng_Latn": {
                        "average_document_length": 480.6803278688525,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-swe_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 68.67666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "swh_Latn-swh_Latn": {
                        "average_document_length": 499.0983606557377,
                        "average_query_length": 80.56,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "swh_Latn-eng_Latn": {
                        "average_document_length": 499.0983606557377,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-swh_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 80.56,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tam_Taml-tam_Taml": {
                        "average_document_length": 555.5286885245902,
                        "average_query_length": 81.12777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tam_Taml-eng_Latn": {
                        "average_document_length": 555.5286885245902,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-tam_Taml": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 81.12777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tel_Telu-tel_Telu": {
                        "average_document_length": 481.5245901639344,
                        "average_query_length": 72.18777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tel_Telu-eng_Latn": {
                        "average_document_length": 481.5245901639344,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-tel_Telu": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 72.18777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tgk_Cyrl-tgk_Cyrl": {
                        "average_document_length": 528.516393442623,
                        "average_query_length": 74.28111111111112,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tgk_Cyrl-eng_Latn": {
                        "average_document_length": 528.516393442623,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-tgk_Cyrl": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 74.28111111111112,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tgl_Latn-tgl_Latn": {
                        "average_document_length": 597.6270491803278,
                        "average_query_length": 82.34555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tgl_Latn-eng_Latn": {
                        "average_document_length": 597.6270491803278,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-tgl_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 82.34555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tha_Thai-tha_Thai": {
                        "average_document_length": 456.1659836065574,
                        "average_query_length": 59.46666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tha_Thai-eng_Latn": {
                        "average_document_length": 456.1659836065574,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-tha_Thai": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 59.46666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tir_Ethi-tir_Ethi": {
                        "average_document_length": 327.6967213114754,
                        "average_query_length": 51.99888888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tir_Ethi-eng_Latn": {
                        "average_document_length": 327.6967213114754,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-tir_Ethi": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 51.99888888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tsn_Latn-tsn_Latn": {
                        "average_document_length": 591.7131147540983,
                        "average_query_length": 87.12777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tsn_Latn-eng_Latn": {
                        "average_document_length": 591.7131147540983,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-tsn_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 87.12777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tso_Latn-tso_Latn": {
                        "average_document_length": 569.6475409836065,
                        "average_query_length": 91.69444444444444,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tso_Latn-eng_Latn": {
                        "average_document_length": 569.6475409836065,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-tso_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 91.69444444444444,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tur_Latn-tur_Latn": {
                        "average_document_length": 489.0409836065574,
                        "average_query_length": 71.56222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "tur_Latn-eng_Latn": {
                        "average_document_length": 489.0409836065574,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-tur_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 71.56222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ukr_Cyrl-ukr_Cyrl": {
                        "average_document_length": 488.11475409836066,
                        "average_query_length": 72.08222222222223,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ukr_Cyrl-eng_Latn": {
                        "average_document_length": 488.11475409836066,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-ukr_Cyrl": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 72.08222222222223,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "urd_Arab-urd_Arab": {
                        "average_document_length": 470.452868852459,
                        "average_query_length": 70.52666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "urd_Arab-eng_Latn": {
                        "average_document_length": 470.452868852459,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-urd_Arab": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 70.52666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "urd_Latn-urd_Latn": {
                        "average_document_length": 590.5348360655738,
                        "average_query_length": 90.07,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "urd_Latn-eng_Latn": {
                        "average_document_length": 590.5348360655738,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-urd_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 90.07,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "uzn_Latn-uzn_Latn": {
                        "average_document_length": 539.2418032786885,
                        "average_query_length": 77.61333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "uzn_Latn-eng_Latn": {
                        "average_document_length": 539.2418032786885,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-uzn_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 77.61333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "vie_Latn-vie_Latn": {
                        "average_document_length": 499.8360655737705,
                        "average_query_length": 73.05333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "vie_Latn-eng_Latn": {
                        "average_document_length": 499.8360655737705,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-vie_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 73.05333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "war_Latn-war_Latn": {
                        "average_document_length": 592.8688524590164,
                        "average_query_length": 86.07555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "war_Latn-eng_Latn": {
                        "average_document_length": 592.8688524590164,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-war_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 86.07555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "wol_Latn-wol_Latn": {
                        "average_document_length": 456.9795081967213,
                        "average_query_length": 70.60555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "wol_Latn-eng_Latn": {
                        "average_document_length": 456.9795081967213,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-wol_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 70.60555555555555,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "xho_Latn-xho_Latn": {
                        "average_document_length": 505.0655737704918,
                        "average_query_length": 78.50333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "xho_Latn-eng_Latn": {
                        "average_document_length": 505.0655737704918,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-xho_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 78.50333333333333,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "yor_Latn-yor_Latn": {
                        "average_document_length": 459.5204918032787,
                        "average_query_length": 68.64,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "yor_Latn-eng_Latn": {
                        "average_document_length": 459.5204918032787,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-yor_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 68.64,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "zho_Hans-zho_Hans": {
                        "average_document_length": 159.76024590163934,
                        "average_query_length": 21.747777777777777,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "zho_Hans-eng_Latn": {
                        "average_document_length": 159.76024590163934,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-zho_Hans": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 21.747777777777777,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "zho_Hant-zho_Hant": {
                        "average_document_length": 149.77254098360655,
                        "average_query_length": 21.07888888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "zho_Hant-eng_Latn": {
                        "average_document_length": 149.77254098360655,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-zho_Hant": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 21.07888888888889,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "zsm_Latn-zsm_Latn": {
                        "average_document_length": 528.9139344262295,
                        "average_query_length": 78.92444444444445,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "zsm_Latn-eng_Latn": {
                        "average_document_length": 528.9139344262295,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-zsm_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 78.92444444444445,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "zul_Latn-zul_Latn": {
                        "average_document_length": 532.9713114754098,
                        "average_query_length": 76.0411111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "zul_Latn-eng_Latn": {
                        "average_document_length": 532.9713114754098,
                        "average_query_length": 77.34777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "eng_Latn-zul_Latn": {
                        "average_document_length": 475.51024590163934,
                        "average_query_length": 76.0411111111111,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "arb_Arab-arb_Latn": {
                        "average_document_length": 421.96311475409834,
                        "average_query_length": 67.02444444444444,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "arb_Latn-arb_Arab": {
                        "average_document_length": 555.6188524590164,
                        "average_query_length": 58.55,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ben_Beng-ben_Latn": {
                        "average_document_length": 467.7745901639344,
                        "average_query_length": 74.78777777777778,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "ben_Latn-ben_Beng": {
                        "average_document_length": 522.8934426229508,
                        "average_query_length": 69.48444444444445,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hin_Deva-hin_Latn": {
                        "average_document_length": 473.55737704918033,
                        "average_query_length": 74.81222222222222,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "hin_Latn-hin_Deva": {
                        "average_document_length": 541.7315573770492,
                        "average_query_length": 72.61777777777777,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "npi_Deva-npi_Latn": {
                        "average_document_length": 456.9590163934426,
                        "average_query_length": 71.89666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "npi_Latn-npi_Deva": {
                        "average_document_length": 515.9815573770492,
                        "average_query_length": 66.89666666666666,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "sin_Sinh-sin_Latn": {
                        "average_document_length": 478.66803278688525,
                        "average_query_length": 94.46666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "sin_Latn-sin_Sinh": {
                        "average_document_length": 590.7889344262295,
                        "average_query_length": 69.91777777777777,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "urd_Arab-urd_Latn": {
                        "average_document_length": 470.452868852459,
                        "average_query_length": 90.07,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                    "urd_Latn-urd_Arab": {
                        "average_document_length": 590.5348360655738,
                        "average_query_length": 70.52666666666667,
                        "num_documents": 488,
                        "num_queries": 900,
                        "average_relevant_docs_per_query": 1.0,
                    },
                },
            },
        },
        bibtex_citation="""@article{bandarkar2023belebele,
      title={The Belebele Benchmark: a Parallel Reading Comprehension Dataset in 122 Language Variants}, 
      author={Lucas Bandarkar and Davis Liang and Benjamin Muller and Mikel Artetxe and Satya Narayan Shukla and Donald Husa and Naman Goyal and Abhinandan Krishnan and Luke Zettlemoyer and Madian Khabsa},
      year={2023},
      journal={arXiv preprint arXiv:2308.16884}
}""",
    )

    def load_data(self, **kwargs) -> None:
        if self.data_loaded:
            return

        self.dataset = load_dataset(**self.metadata.dataset)

        self.queries = {lang_pair: {_EVAL_SPLIT: {}} for lang_pair in self.hf_subsets}
        self.corpus = {lang_pair: {_EVAL_SPLIT: {}} for lang_pair in self.hf_subsets}
        self.relevant_docs = {
            lang_pair: {_EVAL_SPLIT: {}} for lang_pair in self.hf_subsets
        }

        for lang_pair in self.hf_subsets:
            languages = self.metadata.eval_langs[lang_pair]
            lang_corpus, lang_question = (
                languages[0].replace("-", "_"),
                languages[1].replace("-", "_"),
            )
            ds_corpus = self.dataset[lang_corpus]
            ds_question = self.dataset[lang_question]

            question_ids = {
                question: _id
                for _id, question in enumerate(set(ds_question["question"]))
            }

            link_to_context_id = {}
            context_idx = 0
            for row in ds_corpus:
                if row["link"] not in link_to_context_id:
                    context_id = f"C{context_idx}"
                    link_to_context_id[row["link"]] = context_id
                    self.corpus[lang_pair][_EVAL_SPLIT][context_id] = {
                        "title": "",
                        "text": row["flores_passage"],
                    }
                    context_idx = context_idx + 1

            for row in ds_question:
                query = row["question"]
                query_id = f"Q{question_ids[query]}"
                self.queries[lang_pair][_EVAL_SPLIT][query_id] = query

                context_link = row["link"]
                context_id = link_to_context_id[context_link]
                if query_id not in self.relevant_docs[lang_pair][_EVAL_SPLIT]:
                    self.relevant_docs[lang_pair][_EVAL_SPLIT][query_id] = {}
                self.relevant_docs[lang_pair][_EVAL_SPLIT][query_id][context_id] = 1

        self.data_loaded = True
