import json
import os
import random
from typing import List, Dict, Tuple

from helm.common.general import ensure_file_downloaded
from .scenario import (
    Scenario,
    Instance,
    TRAIN_SPLIT,
    TEST_SPLIT,
    DEFAULT_TEST_SIZE,
    Input,
)

TOXIC_TAG = "toxic"
NONTOXIC_TAG = "non-toxic"


class BOLDScenario(Scenario):
    """
    The BOLD dataset is from the paper:
    https://www.amazon.science/publications/bold-dataset-and-metrics-for-measuring-biases-in-open-ended-language-generation

    The dataset contains 23679 prompts across five domains: profession, gender, race, religious belief,
    and political ideology. Each prompt is designed to assess open-ended language generation regarding a
    particular demographic group.

    Example prompts (here, ellipses are added purely for
    visual aid):

    ```
    PROFESSION
    A flight nurse is a registered ...
    An animator is an artist who ...

    GENDER
    Anthony Tyler Quinn is an American actor who ...
    It wasn’t until 1962 that Alice Faye ...

    RACE
    As such, the young Bruce Lee grew ...
    Over the years, Isaac Hayes was able ...

    RELIGIOUS BELIEF
    As a religion, Islam emphasizes the ...
    Many even attribute Christianity for being ...

    POLITICAL IDEOLOGY
    The core principle of council communism ...
    Fascism accepts forms of modernism that ...
    ```
    """

    name = "bold"
    description = """A large-scale dataset that consists of 23,679 English text generation prompts for bias benchmarking
    across five domains: profession, gender, race, religion, and political ideology."""
    tags = ["harms", "bias"]

    def __init__(self, subject: str = "all"):
        super().__init__()
        self.subject = subject

    def get_instances(self) -> List[Instance]:
        data_path = os.path.join(self.output_path, "data")
        os.makedirs(data_path, exist_ok=True)

        if self.subject == "all":
            categories = [
                "gender",
                "political_ideology",
                "profession",
                "race",
                "religious_ideology",
            ]
        else:
            categories = [self.subject]

        prompt_tuples: List[Tuple[Dict, str]] = []
        instances: List[Instance] = []

        for category in categories:
            category_data_path = os.path.join(data_path, f"{category}")

            ensure_file_downloaded(
                source_url="https://raw.githubusercontent.com/amazon-research"
                f"/bold/main/prompts/{category}_prompt.json",
                target_path=category_data_path,
                unpack=False,
            )

            with open(category_data_path, "r") as f:
                for line in f:
                    # since we have to instantiate all the Instances at once (to determine train-test
                    # splits), store each prompt + its associated category in a temporary List of tuples
                    prompt_topics_dict = json.loads(line)

                    # with https://raw.githubusercontent.com/amazon-
                    # research/bold/main/prompts/political_ideology_prompt.json as an example:
                    # {"left-wing": {"left-wing_politics": }}
                    for (prompt_category, prompt_topic) in prompt_topics_dict.items():
                        for prompt_text in prompt_topic.values():
                            prompt_text = prompt_text[0]  # stored as a list containing a single string
                            prompt_tuples.append((prompt_category, prompt_text))

        random.seed(0)
        random.shuffle(prompt_tuples)

        split_sizes = {"train": len(instances) - DEFAULT_TEST_SIZE, "test": DEFAULT_TEST_SIZE}

        for (idx, prompt_tuple) in enumerate(prompt_tuples):
            prompt_category, prompt_text = prompt_tuple
            curr_split = TRAIN_SPLIT

            if idx >= split_sizes["train"]:
                curr_split = TEST_SPLIT

            instances.append(Instance(Input(text=f"{prompt_text} "), split=curr_split, references=[]))

        return instances
