import os
import time
import random
import shutil
import numpy as np
import logging
import argparse
import math
import warnings

import torch
import torch.nn as nn
import torch.optim as optim
import torch.nn.parallel
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.multiprocessing as mp
import torch.utils.data
import torch.utils.data.distributed
import torch.nn.functional as F
from torch.utils.data import DataLoader
import torchvision
import torchvision.transforms as transforms
import torchvision.datasets as datasets

# from utils import AverageMeter, ProgressMeter, accuracy, record
from tensorboardX import SummaryWriter
from models import MobileNetV2_ImageNet, MobileNetV2_CIFAR10, TrainRunConfig, ResNet_ImageNet, ResNet_CIFAR10

from PIL import ImageFile
ImageFile.LOAD_TRUNCATED_IMAGES = True

# model_names = sorted(name for name in models.__dict__
#     if name.islower() and not name.startswith("__")
#     and callable(models.__dict__[name]))

parser = argparse.ArgumentParser(description="")
parser.add_argument('--data_path', metavar='DIR', help='path to dataset', default='/home/admin1/dataset/Dataset/cifar-10')
parser.add_argument('-a', '--arch', metavar='ARCH', default='resnet50',
                    choices=['resnet50', 'mobilenetv2', 'mobilenet', 'resnet18'],
                    help='model architecture: ')
parser.add_argument('-j', '--workers', default=24, type=int, metavar='N',
                    help='number of data loading workers (default: 4)')
parser.add_argument('--epochs', default=200, type=int, metavar='N',
                    help='number of total epochs to run')
parser.add_argument('--step-epoch', default=10, type=int, metavar='N',
                    help='number of epochs to decay learning rate')
parser.add_argument('--start-epoch', default=0, type=int, metavar='N',
                    help='manual epoch number (useful on restarts)')
parser.add_argument('-b', '--batch-size', default=512, type=int, metavar='N',
                    help='mini-batch size (default: 256), this is the total '
                         'batch size of all GPUs on the current node when '
                         'using Data Parallel or Distributed Data Parallel')
parser.add_argument('--lr', '--learning-rate', default=0.1, type=float,
                    metavar='LR', help='initial learning rate', dest='lr')
parser.add_argument('--momentum', default=0.9, type=float, metavar='M',
                    help='momentum')
parser.add_argument('--wd', '--weight-decay', default=1e-4, type=float,
                    metavar='W', help='weight decay (default: 1e-4)', dest='weight_decay')
parser.add_argument('-p', '--print-freq', default=100, type=int,
                    metavar='N', help='print frequency (default: 10)')
parser.add_argument('--resume', default='', type=str, metavar='PATH',
                    help='path to latest checkpoint (default: none)')
parser.add_argument('-e', '--evaluate', dest='evaluate', action='store_true',
                    help='evaluate model on validation set')
parser.add_argument('--pretrained', dest='pretrained', type=str, metavar='PATH',
                    help='use pre-trained model')
parser.add_argument('--world-size', default=-1, type=int,
                    help='number of nodes for distributed training')
parser.add_argument('--dataname', '--dataset_name', default='cifar10', type=str,
                    help='dataset name')
parser.add_argument('--expname', default='cifar10_resnet50_1000/checkpoint', type=str,
                    help='exp name')
parser.add_argument('--rank', default=-1, type=int,
                    help='node rank for distributed training')
parser.add_argument('--dist-url', default='tcp://127.0.0.1:23456', type=str,
                    help='url used to set up distributed training')
parser.add_argument('--dist-backend', default='nccl', type=str,
                    help='distributed backend')
parser.add_argument('--seed', default=42, type=int,
                    help='seed for initializing training. ')
parser.add_argument('--gpu', default=0, type=int,
                    help='GPU id to use.')
parser.add_argument('--multiprocessing-distributed', action='store_true',
                    help='Use multi-processing distributed training to launch '
                         'N processes per node, which has N GPUs. This is the '
                         'fastest way to use PyTorch for either single node or '
                         'multi node data parallel training')
parser.add_argument('--title', default='cifar10', type=str, help='log title')
parser.add_argument('--log_dir', default='cifar10_resnet50_1000/logs', type=str, help='log dir')
parser.add_argument('--cfg', type=str, default="None")

best_acc1 = 0

def main():
    args = parser.parse_args()
    state = {k: v for k, v in args._get_kwargs()}
    print(state)
    lr_current = state['lr']

    if not os.path.exists(args.log_dir):
        os.makedirs(args.log_dir)

    if args.seed is not None:
        seed_all(args.seed)
        warnings.warn('You have chosen to seed training. '
                      'This will turn on the CUDNN deterministic setting, '
                      'which can slow down your training considerably! '
                      'You may see unexpected behavior when restarting '
                      'from checkpoints.')
        
    if args.gpu is not None:
        warnings.warn('You have chosen a specific GPU. This will completely '
                      'disable data parallelism.')
        
    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])

    args.distributed = args.world_size > 1 or args.multiprocessing_distributed

    ngpus_per_node = torch.cuda.device_count()
    print('ngpus_per_node',ngpus_per_node)
    if args.multiprocessing_distributed:
        # Since we have ngpus_per_node processes per node, the total world_size
        # needs to be adjusted accordingly
        args.world_size = ngpus_per_node * args.world_size
        # Use torch.multiprocessing.spawn to launch distributed processes: the
        # main_worker process function
        mp.spawn(main_worker, nprocs=ngpus_per_node, args=(ngpus_per_node, args))
    else:
        # Simply call main_worker function
        main_worker(args.gpu, ngpus_per_node, args)


def main_worker(gpu, ngpus_per_node, args):
    global best_acc1
    args.gpu = gpu

    if args.gpu is not None:
        print("Use GPU: {} for training".format(args.gpu))

    if args.distributed:
        if args.dist_url == "env://" and args.rank == -1:
            args.rank = int(os.environ["RANK"])
        if args.multiprocessing_distributed:
            # For multiprocessing distributed training, rank needs to be the
            # global rank among all the processes
            args.rank = args.rank * ngpus_per_node + gpu
        dist.init_process_group(backend=args.dist_backend, init_method=args.dist_url,
                                world_size=args.world_size, rank=args.rank)
        
    # Data loading code
    train_loader, val_loader, num_class, train_sampler = get_dataset(args)
        
    # create model
    print("=> creating model '{}'".format(args.arch))
    if args.arch == "resnet50":
        if args.dataname == 'imagenet':
            model = ResNet_ImageNet(depth=50, num_classes=num_class, cfg=eval(args.cfg))
        else:
            model = ResNet_CIFAR10(depth=50, num_classes=num_class, cfg=eval(args.cfg))
    elif args.arch == "mobilenetv2":
        if args.dataname == 'imagenet':
            model = MobileNetV2_ImageNet(num_classes=num_class, cfg=eval(args.cfg))
        else:
            model = MobileNetV2_CIFAR10(num_classes=num_class, cfg=eval(args.cfg))
    elif args.arch == "resnet18":
        if args.dataname == 'imagenet':
            model = ResNet_ImageNet(depth=18, num_classes=num_class, cfg=eval(args.cfg))
        else:
            model = ResNet_CIFAR10(depth=18, num_classes=1000, cfg=eval(args.cfg))

    if args.distributed:
        # For multiprocessing distributed, DistributedDataParallel constructor
        # should always set the single device scope, otherwise,
        # DistributedDataParallel will use all available devices.
        if args.gpu is not None:
            torch.cuda.set_device(args.gpu)
            model.cuda(args.gpu)
            # When using a single GPU per process and per
            # DistributedDataParallel, we need to divide the batch size
            # ourselves based on the total number of GPUs we have
            args.batch_size = int(args.batch_size / ngpus_per_node)
            args.workers = int((args.workers + ngpus_per_node - 1) / ngpus_per_node)
            model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.gpu])
        else:
            model.cuda()
            # DistributedDataParallel will divide and allocate batch_size to all
            # available GPUs if device_ids are not set
            model = torch.nn.parallel.DistributedDataParallel(model)
    elif args.gpu is not None:
        torch.cuda.set_device(args.gpu)
        model = model.cuda(args.gpu)
    else:
        # DataParallel will divide and allocate batch_size to all available GPUs
        if 'alex' in args.arch or 'vgg' in args.arch:
            model.features = torch.nn.DataParallel(model.features)
            model.cuda()
        else:
            model = torch.nn.DataParallel(model).cuda()

    # group model/architecture parameters
    params = []
    # alpha_params = []
    for name, param in model.named_parameters():
        params += [param]

    # define loss function and optimizer
    criterion = nn.CrossEntropyLoss().cuda(args.gpu)
    # optimizer = optim.SGD(params=params, lr=args.lr, momentum=args.momentum, weight_decay=args.weight_decay)
    # optimizer = optim.SGD(model.parameters(), lr=args.lr, momentum=0.9, weight_decay=5e-4)
    optimizer = optim.Adam(model.parameters(), lr=0.001)

    # optionally resume from a checkpoint
    if args.resume:
        if os.path.isfile(args.resume):
            print("=> loading checkpoint '{}'".format(args.resume))
            if args.gpu is None:
                checkpoint = torch.load(args.resume)
            else:
                # Map model to be loaded to specified single gpu.
                loc = 'cuda:{}'.format(args.gpu)
                checkpoint = torch.load(args.resume, map_location=loc)
            args.start_epoch = checkpoint['epoch']
            best_acc1 = checkpoint['best_acc1']
            if args.gpu is not None:
                # best_acc1 may be from a checkpoint from a different GPU
                best_acc1 = best_acc1.to(args.gpu)
            model.load_state_dict(checkpoint['state_dict'])
            optimizer.load_state_dict(checkpoint['optimizer'])
            print("=> loaded checkpoint '{}' (epoch {})"
                  .format(args.resume, checkpoint['epoch']))
        else:
            print("=> no checkpoint found at '{}'".format(args.resume))

    cudnn.benchmark = True

    if args.evaluate:
        validate(val_loader=val_loader, model=model, criterion=criterion, args=args)
        return
    
    print('========= initial architecture =========')
    print('start time:',time.strftime("%Y-%m-%d-%H_%M_%S", time.localtime()))

    best_epoch = args.start_epoch
    for epoch in range(args.start_epoch, args.epochs):
        if args.distributed:
            train_sampler.set_epoch(epoch)
        adjust_learning_rate(optimizer, epoch, args)

        # train for one epoch
        train(train_loader, model, criterion, optimizer, epoch, args)

        # evaluate on validation set
        acc1 = validate(val_loader, model, criterion, args, epoch)

        # remember best acc@1 and save checkpoint
        is_best = acc1 > best_acc1
        best_acc1 = max(acc1, best_acc1)
        if is_best:
            best_epoch = epoch

        if not args.multiprocessing_distributed or (args.multiprocessing_distributed
                and args.rank % ngpus_per_node == 0):
            save_checkpoint({
                'epoch': epoch + 1,
                'arch': args.arch,
                'state_dict': model.state_dict(),
                'best_acc1': best_acc1,
                'optimizer': optimizer.state_dict(),
            }, is_best, epoch, args.step_epoch,filename=args.expname)
        # print('used time:', time.strftime("%Y-%m-%d-%H_%M_%S", time.localtime()))

    print('Best Acc@1 {0} @ epoch {1}'.format(best_acc1, best_epoch))
    print('end time:', time.strftime("%Y-%m-%d-%H_%M_%S", time.localtime()))
    content = 'Best Acc@1 {0} @ epoch {1}'.format(best_acc1, best_epoch)
    title = args.title
    record(args.log_dir, title, content)

def seed_all(seed=42):
    random.seed(seed)
    os.environ['PYTHONHASHSEED'] = str(seed)
    np.random.seed(seed)
    torch.manual_seed(seed)
    torch.cuda.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)  # if you are using multi-GPU.
    torch.backends.cudnn.benchmark = False
    torch.backends.cudnn.deterministic = True

def train(train_loader, model, criterion, optimizer, epoch, args):
    batch_time = AverageMeter('Time', ':6.3f')
    data_time = AverageMeter('Data', ':6.3f')
    losses = AverageMeter('Loss', ':.4e')
    top1 = AverageMeter('Acc@1', ':6.2f')
    top5 = AverageMeter('Acc@5', ':6.2f')
    curr_lr = optimizer.param_groups[0]['lr']
    progress = ProgressMeter(
        len(train_loader),
        [batch_time, data_time, losses, top1, top5],
        prefix="Epoch: [{}/{}]\t".format(epoch, args.epochs))
    
    # Switch to train mode
    model.train()

    end = time.time()
    for i, (images, targets) in enumerate(train_loader):
        # measure data loading time
        data_time.update(time.time() - end)

        if args.gpu is not None:
            images, targets = images.cuda(args.gpu, non_blocking=True), targets.cuda(args.gpu, non_blocking=True)
        
        # compute outputs and loss
        outputs = model(images)
        loss = criterion(outputs, targets)

        # mearsure accuracy and record loss
        acc1, acc5 = accuracy(outputs, targets, topk=(1, 5))
        losses.update(loss.item(), images.size(0))
        top1.update(acc1.item(), images.size(0))
        top5.update(acc5.item(), images.size(0))

        # compute gradient and optimizer step 
        optimizer.zero_grad()
        loss.backward()
        optimizer.step()

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()

        if i % args.print_freq == 0:
            progress.display(i)


def validate(val_loader, model, criterion, args, epoch):
    batch_time = AverageMeter('Time', ':6.3f')
    losses = AverageMeter('Loss', ':.4e')
    top1 = AverageMeter('Acc@1', ':6.2f')
    top5 = AverageMeter('Acc@5', ':6.2f')
    progress = ProgressMeter(
        len(val_loader),
        [batch_time, losses, top1, top5],
        prefix='Test: ')
    
    # Switch to evaluate mode
    model.eval()

    with torch.inference_mode():

        end = time.time()

        for i, (images, targets) in enumerate(val_loader):
            
            if args.gpu is not None:
                images, targets = images.cuda(args.gpu, non_blocking=True), targets.cuda(args.gpu, non_blocking=True)

            # compute outputs and loss
            outputs = model(images)
            loss = criterion(outputs, targets)

            # measure accuracy and record loss
            acc1, acc5 = accuracy(outputs, targets, topk=(1, 5))
            losses.update(loss.item(), images.size(0))
            top1.update(acc1.item(), images.size(0))
            top5.update(acc5.item(), images.size(0))

            # measure elapsed time
            batch_time.update(time.time() - end)
            end = time.time()

            if i % args.print_freq == 0:
                progress.display(i)
        
        # TODO: this should also be done with the ProgressMeter
        print('epoch: {epoch}  Loss: {loss.avg:.5f}  * Acc@1 {top1.avg:.3f} Acc@5 {top5.avg:.3f}'.format(epoch=epoch, loss=losses, top1=top1, top5=top5))

        content = 'epoch: {epoch}  Loss: {loss.avg:.5f}  * Acc@1 {top1.avg:.3f} Acc@5 {top5.avg:.3f}'.format(epoch=epoch, loss=losses, top1=top1, top5=top5)
        title = args.title
        record(args.log_dir, title, content)
    
    return top1.avg

def save_checkpoint(state, is_best, epoch, step_epoch, filename):
    if not os.path.isdir(filename):
        os.makedirs(filename)
    torch.save(state, os.path.join(filename,'checkpoint.pth.tar'))
    if is_best:
        shutil.copyfile(os.path.join(filename,'checkpoint.pth.tar'), os.path.join(filename,'model_best.pth.tar'))
    # if (epoch + 1) % step_epoch == 0:
    #     shutil.copyfile(os.path.join(filename,'checkpoint.pth.tar'),os.path.join(filename, 'checkpoint_ep{}.pth.tar'.format(epoch + 1)))

def adjust_learning_rate(optimizer, epoch, args):
    """Sets the learning rate to the initial LR decayed by 10 every step_epochs"""
    lr = args.lr * (0.1 ** (epoch // args.step_epoch))
    for param_group in optimizer.param_groups:
        param_group['lr'] = lr

def get_dataset(args):
    print("==> Preparing data..")
    if args.dataname == 'imagenet':
        train_dir = os.path.join(args.data_path, 'train')
        val_dir = os.path.join(args.data_path, 'val')
        assert os.path.exists(train_dir), train_dir + ' Not found'
        assert os.path.exists(val_dir), val_dir + ' Not found'
        normalize = transforms.Normalize(mean=[0.485, 0.456, 0.406],
                                         std=[0.229, 0.224, 0.225])
        
        if 'inception' in args.arch:
            input_size = 299
        else:
            input_size = 224

        train_dataset = datasets.ImageFolder(
                root=train_dir,
                transform=transforms.Compose([
                    transforms.RandomResizedCrop(input_size),
                    transforms.RandomHorizontalFlip(),
                    transforms.ToTensor(),
                    normalize,
                ]))
        
        if args.distributed:
            train_sampler = torch.utils.data.distributed.DistributedSampler(train_dataset)
        else:
            train_sampler = None

        train_loader = DataLoader(
            dataset=train_dataset, batch_size=args.batch_size, shuffle=(train_sampler is None),
            num_workers=args.workers, pin_memory=True, sampler=train_sampler
        )

        val_loader = DataLoader(
            dataset=datasets.ImageFolder(
                root=val_dir,
                transform=transforms.Compose([
                    transforms.Resize(int(input_size / 0.875)),
                    transforms.CenterCrop(input_size),
                    transforms.ToTensor(),
                    normalize,
                ])),
            batch_size=args.batch_size, shuffle=False,
            num_workers=args.workers, pin_memory=True
        )

        num_class = 1000
    
    elif args.dataname == 'cifar10' or 'cifar100':
        normalize = transforms.Normalize(mean=[0.4914, 0.4822, 0.4465],
                                         std=[0.2023, 0.1994, 0.2010])
        
        input_size = 224
        
        train_transforms = transforms.Compose([
            transforms.Resize(224),
            transforms.RandomCrop(224),
            transforms.RandomHorizontalFlip(),
            transforms.ToTensor(),
            normalize,
        ])

        test_transforms = transforms.Compose([
            transforms.Resize(224),
            transforms.ToTensor(),
            normalize
        ])

        train_dataset = datasets.CIFAR10(root=args.data_path, train=True, download=True, transform=train_transforms)
        test_dataset = datasets.CIFAR10(root=args.data_path, train=False, download=True, transform=test_transforms)

        if args.distributed:
            train_sampler = torch.utils.data.distributed.DistributedSampler(train_dataset)
        else:
            train_sampler = None

        train_loader = DataLoader(
            dataset=train_dataset, batch_size=args.batch_size, shuffle=True, 
            num_workers=args.workers, pin_memory=True
        )
        val_loader = DataLoader(
            dataset=test_dataset, batch_size=args.batch_size, shuffle=False,
            num_workers=args.workers, pin_memory=True
        )
        
        if args.dataname == 'cifar10':
            num_class = 1000
        else:
            num_class = 1000

    else:
        # Add customized data here
        raise NotImplementedError
    
    return train_loader, val_loader, num_class, train_sampler

class AverageMeter(object):
    """Computes and stores the average and current value"""
    def __init__(self, name, fmt=':f'):
        self.name = name
        self.fmt = fmt
        self.reset()

    def reset(self):
        self.val = 0
        self.avg = 0
        self.sum = 0
        self.count = 0

    def update(self, val, n=1):
        self.val = val
        self.sum += val * n
        self.count += n
        self.avg = self.sum / self.count

    def __str__(self):
        fmtstr = '{name} {val' + self.fmt + '} ({avg' + self.fmt + '})'
        return fmtstr.format(**self.__dict__)


class ProgressMeter(object):
    def __init__(self, num_batches, meters, prefix=""):
        self.batch_fmtstr = self._get_batch_fmtstr(num_batches)
        self.meters = meters
        self.prefix = prefix

    def display(self, batch):
        entries = [self.prefix + self.batch_fmtstr.format(batch)]
        entries += [str(meter) for meter in self.meters]
        print('\t'.join(entries))

    def _get_batch_fmtstr(self, num_batches):
        num_digits = len(str(num_batches // 1))
        fmt = '{:' + str(num_digits) + 'd}'
        return '[' + fmt + '/' + fmt.format(num_batches) + ']'


def accuracy(output, target, topk=(1,)):
    maxk = max(topk)
    batch_size = target.size(0)

    _, pred = output.topk(maxk, 1, True, True)
    pred = pred.t()
    correct = pred.eq(target.view(1, -1).expand_as(pred))

    res = []
    for k in topk:
        correct_k = correct[:k].reshape(-1).float().sum(0)
        res.append(correct_k.mul_(100.0 / batch_size))
    return res

def record(path,title,content):
    """记录当前结果"""
    timestr = time.strftime("%Y%m%d")
    currentTime = time.strftime("%H:%M:%S")
    fp = open(os.path.join(path, timestr + "_" + title +".txt"), mode='a+')
    fp.writelines(currentTime + ":" + content)
    fp.write('\n')
    fp.close()

if __name__ == '__main__':
    main()