import React, {useCallback, useState} from 'react';
import Button from 'antd/es/button';
import Card from 'antd/es/card';
import Badge from 'antd/es/badge';
import Col from 'antd/es/col';
import Drawer from 'antd/es/drawer';
import Form from 'antd/es/form';
import Layout from 'antd/es/layout';
import Row from 'antd/es/row';
import Select from 'antd/es/select';
import Slider from 'antd/es/slider';
import Steps from 'antd/es/steps';
import Tag from 'antd/es/tag';
import Radio from 'antd/es/radio';
import { LoadingOutlined } from '@ant-design/icons';
import Settings, {selectedDatasetAtom, selectedSampleAtom, inputOutputSelectedAtom} from "./components/settings";
import {atom, useAtom} from "jotai";
import DecisionTreeCard, {selectedTreePruningAtom} from "./decisionTreeCard";
import GraphCard, { preprocessSample } from "./components/graphCard";

import {getDatasets, getSample, getTrees} from "./db";
import LoadingCard from "./components/loadingCard";


const {Step} = Steps;
const {Content} = Layout;
const {Option} = Select;

const colors = ['yellow', 'blue', 'red', 'green', 'orange', 'purple', 'lime', 'cyan', '#f33eda', '#C8F3A9']

export const selectedLayerAtom = atom(0)
export const decisionPathAtom = atom([])

function App() {

    const [selectedLayer, setSelectedLayer] = useAtom(selectedLayerAtom)
    const [, setSelectedDecisionPath] = useAtom(decisionPathAtom)
    const [currentSample, setSelectedSample] = useAtom(selectedSampleAtom)
    const [selectedDataset, setSelectedDataset] = useAtom(selectedDatasetAtom);
    const [inputOutputSelected, setInputOutputSelected] = useAtom(inputOutputSelectedAtom);

    const [selectedTreePruning, setSelectedTreePruning] = useAtom(selectedTreePruningAtom)
    const [dbDatasets, setDbDatasets] = useState(undefined);
    const [treeData, setTreeData] = useState(undefined);
    const [currentTree, setCurrentTree] = useState(undefined);
    const [graphData, setGraphData] = useState(undefined);
    const [currentGraph, setCurrentGraph] = useState(undefined);
    const [sampleIsLoading, setSampleIsLoading] = useState(true);
    const [treeIsLoading, setTreeIsLoading] = useState(true);

    const onInputOutputSelect = (e) => {
        setInputOutputSelected(e.target.value)
    };

    const onHooverNode = useCallback((hooveredNode) => {
        if (hooveredNode !== -1) {
            const path = currentGraph.nodes[hooveredNode]["decision_paths"][selectedLayer]
            setSelectedDecisionPath(path)
        } else {
            setSelectedDecisionPath([])
        }
    }, [currentGraph, selectedLayer, setSelectedDecisionPath]);


    const [settingsVisible, setSettingsVisible] = useState(false);
    const showDrawer = () => {
        setSettingsVisible(true);
    };
    const onClose = () => {
        setSettingsVisible(false);
    };

    React.useEffect(() => {
        getDatasets().then(data => {
            setSelectedSample(0);
            setSelectedDataset(0);
            setDbDatasets(data);
            setSelectedLayer(data[0]["layers"].length - 1);
        });
    }, [setDbDatasets, setSelectedDataset, setSelectedLayer, setSelectedSample]);

    React.useEffect(() => {
        if (dbDatasets) {
            setSelectedLayer(dbDatasets[selectedDataset]["layers"].length - 1);
            setTreeIsLoading(true);
            getTrees(dbDatasets[selectedDataset].dataset).then(data => {
                setTreeData(data[0]);
                setTreeIsLoading(false);
            });
            setSampleIsLoading(true);
            getSample(dbDatasets[selectedDataset].dataset, 0).then(data => {
                setGraphData(data[0]);
                setSampleIsLoading(false);
            });
        }
    }, [dbDatasets, selectedDataset, setSampleIsLoading, setSelectedLayer, setTreeIsLoading]);

    React.useEffect(() => {
        if (dbDatasets && !sampleIsLoading) {
            setSampleIsLoading(true);
            getSample(dbDatasets[selectedDataset].dataset, currentSample).then(data => {
                setGraphData(data[0]);
                setSampleIsLoading(false);
            });
        }
    }, [currentSample, dbDatasets, selectedDataset, setSampleIsLoading]);

    React.useEffect(() => {
        if (treeData) {
            setCurrentTree(treeData.data[selectedTreePruning].trees[selectedLayer]);
        }
        if (graphData) {
            const sampleGraph = {...graphData.data[selectedTreePruning], edges: graphData.edges};
            preprocessSample(sampleGraph, selectedLayer);
            setCurrentGraph(sampleGraph);
        }
    }, [treeData, selectedTreePruning, selectedLayer, graphData]);

    const somethingIsLoading = treeIsLoading || sampleIsLoading;
    return (
        <div className="app" style={{height: "100%"}}>
            <Layout style={{height: "100%", background: "var(--background)"}}>
                <div style={{position: 'absolute', zIndex: 20, padding: "16px 0 0 24px"}}>
                    {dbDatasets && dbDatasets[selectedDataset].legend.map((legend, i) => {
                        return (<div>
                            <Badge color={colors[legend.index]} text={`${legend.index}: ${legend.label}`} key={i}/>
                        </div>)
                    })}
                </div>
                <Content style={{backgroundColor: "var(--background)", height: "100%"}}>
                    <div style={{display: "flex", flexDirection: 'column', padding: 16, height: "100%"}}>
                        <Row gutter={16} style={{height: "100%"}}>
                            <Col span={12}>
                                <Card style={{width: "100%", height: "100%", padding: 0}}
                                      bodyStyle={{height: "100%", padding: 0}}>
                                    <LoadingCard loading={sampleIsLoading}/>
                                    {graphData && (
                                        <GraphCard selectedLayer={selectedLayer} onHooverNode={onHooverNode}
                                                   graphData={graphData}
                                                   selectedTreePruning={selectedTreePruning}/>
                                    )}
                                </Card>
                            </Col>
                            <Col span={12}>
                                <Card style={{width: "100%", height: "100%", padding: 0}}
                                      bodyStyle={{height: "100%", padding: 0}}>
                                    <LoadingCard loading={treeIsLoading}/>
                                    {currentTree && (
                                        <DecisionTreeCard selectedLayer={selectedLayer}
                                                          treeData={currentTree}/>
                                    )}
                                </Card>
                            </Col>
                        </Row>
                        <Steps size="small" current={selectedLayer} onChange={(value) => setSelectedLayer(value)}
                               style={{
                                   background: "var(--background)",
                                   color: "var(--background)",
                                   width: '100%',
                                   margin: "16px 0"
                               }}>
                            {dbDatasets ? (dbDatasets[selectedDataset]["layers"].map((layer, i) => {
                                return (<Step title={layer} key={i} disabled={somethingIsLoading}/>)
                            })) : (
                                <>
                                    <Step title="Loading" status="process" icon={<LoadingOutlined/>} disabled/>
                                    <Step title="Loading" status="process" icon={<LoadingOutlined/>} disabled/>
                                </>
                            )}
                        </Steps>
                        <div style={{
                            display: "flex",
                            flexDirection: 'row',
                            justifyContent: 'space-between',
                            marginBottom: 8
                        }}>
                            <Form
                                name="sample-form"
                                layout="inline"
                            >
                                <Form.Item label="Show Node State">
                                    <Radio.Group value={inputOutputSelected} buttonStyle="solid" onChange={onInputOutputSelect} disabled={somethingIsLoading}>
                                        <Radio.Button value="input">Input</Radio.Button>
                                        <Radio.Button value="output">Output</Radio.Button>
                                    </Radio.Group>
                                </Form.Item>
                            </Form>
                            {/*<Button type="primary" onClick={showDrawer} style={{width: 100, marginRight: 16}} disabled={somethingIsLoading}>*/}
                            {/*    Settings*/}
                            {/*</Button>*/}
                            {treeData && dbDatasets[selectedDataset].steps && (
                                <Form
                                    name="sample-form"
                                    layout="inline"
                                >
                                    <Form.Item label="Prune Trees">
                                        <Slider value={selectedTreePruning} step={1} min={0}
                                                max={9} style={{width: 100}}
                                                onChange={(value) => setSelectedTreePruning(value)}
                                                tipFormatter={(value) => `${value * 10} %`}/>
                                    </Form.Item>
                                    <Form.Item label="Acc. Train" style={{marginRight: 0}}>
                                        <Tag>{treeData.data[selectedTreePruning]["train_score"].toFixed(2)}</Tag>
                                    </Form.Item>
                                    <Form.Item label="Test" style={{marginRight: 0}}>
                                        <Tag>{treeData.data[selectedTreePruning]["test_score"].toFixed(2)}</Tag>
                                    </Form.Item>
                                </Form>
                            )}
                            <Form
                                name="sample-form"
                                layout="inline"
                            >
                                <Form.Item label="Dataset">
                                    <Select placeholder="Select a Dataset" value={selectedDataset} style={{width: 120}}
                                            onChange={(value => {
                                                setSampleIsLoading(true);
                                                setSelectedTreePruning(0);
                                                setSelectedSample(0);
                                                setSelectedDataset(value);
                                            })} loading={treeIsLoading}>
                                        {dbDatasets && dbDatasets.map((sample, i) => {
                                            return (<Option value={i} key={i}>{sample.title}</Option>)
                                        })}
                                    </Select>
                                </Form.Item>
                                <Form.Item label="Sample" style={{marginRight: 0}}>
                                    <Select placeholder="Select a Sample" value={currentSample} style={{width: 300}}
                                            onChange={(value => setSelectedSample(value))} loading={sampleIsLoading}>
                                        {dbDatasets && dbDatasets[selectedDataset].samples.map((sample, i) => {
                                            return (<Option value={i} key={i}>{sample}</Option>)
                                        })}
                                    </Select>
                                </Form.Item>
                            </Form>
                        </div>
                        <Drawer title="Settings" placement="left" onClose={onClose} visible={settingsVisible}
                                width={360}>
                            <Settings/>
                        </Drawer>
                    </div>
                </Content>
            </Layout>
        </div>
    );
}

export default App;
