# Copyright 2023 PKU-Alignment Team. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================

from __future__ import annotations

from datasets import load_dataset
from safe_rlhf.datasets.base import RawDataset, RawSample
from typing import ClassVar

import pandas as pd

__all__ = ['TLDRComparisonTrainDataset', 'TLDRComparisonTestDataset', 'TLDRAxisTrainDataset', 'TLDRAxisTestDataset']




class TLDRComparisonDataset(RawDataset):
    SPLIT: ClassVar[str]
    PATH: ClassVar[str]

    def __init__(self, path: str | None = None) -> None:
        self.data= pd.read_parquet(self.PATH)
        self.data = self.data[self.data['info'].apply(lambda x: x['post'] is not None)]
        # self.data = self.data[self.data.apply(self.filter_rows, axis=1)]

    def __getitem__(self, index: int) -> RawSample:
        data = self.data.iloc[index]
        pre_prompt = 'Please provide a concise summary of the following content, capturing the main ideas and key points. Limit your summary to no more than three sentences. Content: '

        return RawSample(
                input=pre_prompt+data['info']['post'],
                answer=data['summaries'][0]['text'],
                other_answer=data['summaries'][1]['text'],
                better=int(data['choice']) == 0,
        )
    def filter_rows(self, row):
        return row['summaries'][0]['text'] != row['summaries'][1]['text']



    def __len__(self) -> int:
        return len(self.data)



class TLDRComparisonTrainDataset(TLDRComparisonDataset):
    NAME: str = 'TLDRComparison/train'
    ALIASES: tuple[str, ...] = ('tldr-comparison/train',)

    PATH: str = './datasets/tldr_rlhf/tldr/comparisons/train/0000.parquet'
    SPLIT: str = 'train'


class TLDRComparisonTestDataset(TLDRComparisonDataset):
    NAME: str = 'TLDRComparison/test'
    ALIASES: tuple[str, ...] = ('tldr-comparison/test',)
    PATH: str = './datasets/tldr_rlhf/tldr/comparisons/validation/0000.parquet'
    SPLIT: str = 'test'




class TLDRAxisDataset(RawDataset):
    SPLIT: ClassVar[str]
    PATH: ClassVar[str]

    def __init__(self, path: str | None = None) -> None:
         self.data= pd.read_parquet(self.PATH)
         self.data = self.data[self.data['info'].apply(lambda x: x['post'] is not None)]

    def __getitem__(self, index: int) -> RawSample:
        data = self.data.iloc[index]
        pre_prompt = 'Please provide a concise summary of the following content, capturing the main ideas and key points. Limit your summary to no more than three sentences. Content: '
        if isinstance(data['info']['post'], tuple):
            input_text = data['info']['post'][0]
        else:
            input_text = data['info']['post']
        return RawSample(
                input=pre_prompt+input_text,
                answer=data['summary']['text'],
        )

    def __len__(self) -> int:
        return len(self.data)




class TLDRAxisTrainDataset(TLDRAxisDataset):
    NAME: str = 'TLDRAxis/train'
    ALIASES: tuple[str, ...] = ('tldr-axis/train',)
    PATH: str = './datasets/tldr_rlhf/tldr/axis/validation/0000.parquet'
    SPLIT: str = 'train'

class TLDRAxisTestDataset(TLDRAxisDataset):
    NAME: str = 'TLDRAxis/test'
    ALIASES: tuple[str, ...] = ('tldr-axis/test',)
    PATH: str = './datasets/tldr_rlhf/tldr/axis/test/0000.parquet'
    SPLIT: str = 'test'

